package remoteviz.loadbalancing.loadbalancedrenderingservice;

import java.io.Console;
import java.util.Random;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.nodes.SoCone;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer;
import com.openinventor.remoteviz.rendering.RenderArea;
import com.openinventor.remoteviz.rendering.Service;
import com.openinventor.remoteviz.rendering.ServiceListener;
import com.openinventor.remoteviz.rendering.ServiceSettings;

import remoteviz.loadbalancing.loadbalancedrenderingservice.LoadBalancerClient;
import remoteviz.loadbalancing.loadbalancedrenderingservice.Instance;

public class LoadBalancedService extends ServiceListener
{

  private static boolean running = true;
  private static final Thread mainThread = Thread.currentThread();
  private SbColor color;

  /**
   * @param args
   * @throws InterruptedException
   */
  public static void main(String[] args)
  {
    // Default value for service IP address
    String serviceIPaddress = "127.0.0.1";
    // Default value for service port
    short servicePort = 8080;

    // Use passed values for the service IP address and port
    if ( args.length == 2 )
    {
      serviceIPaddress = args[0];
      servicePort = Short.parseShort(args[1]);
    }

    // Instantiate a service settings class
    ServiceSettings settings = new ServiceSettings();
    // set the IP address used by the service
    settings.setIP(serviceIPaddress);
    // set the port used by the service
    settings.setPort(servicePort);

    // Add a HelloConeService class as listener
    Service.instance().addListener(new LoadBalancedService(settings));

    // Create instance object
    Instance instance = new Instance();
    instance.setIp(serviceIPaddress);
    instance.setPort(servicePort);

    // To use an Open Inventor extension with RemoteViz, uncomment the following
    // line and set the extension name.
    // settings.setUsedExtensions(ServiceSettings.Extensions.VOLUMEVIZLDM);

    // Open the service by using the settings
    if ( Service.instance().open(settings) )
    {
      Runtime.getRuntime().addShutdownHook(new Thread()
      {
        @Override
        public void run()
        {
          try
          {
            running = false;
            mainThread.join();
          }
          catch (InterruptedException e)
          {}
        }
      });

      System.out.println("The HelloCone Rendering Service is running. Press Ctrl + C to stop.");
      try
      {
        // Register the instance to the load balancer.
        if (!LoadBalancerClient.registerInstance(instance))
        {
          // An error occured with the load balancer API.
          running = false;
        }

        // Main loop
        while ( running )
        {
          Service.instance().dispatch();
          Thread.sleep(1);
        }
      }
      catch (InterruptedException ex)
      {}
      finally
      {
        // Unregister the instance to the load balancer.
        LoadBalancerClient.unregisterInstance(instance);
        // Close the service
        Service.instance().close();
      }
    }
  }

  public LoadBalancedService(ServiceSettings settings)
  {
    String key = String.format("%s:%d", settings.getIP(), settings.getPort());
    Random random = new Random(key.hashCode());
    float r = random.nextFloat();
    float g = random.nextFloat();
    float b = random.nextFloat();
    this.color = new SbColor(r, g, b);
  }

  @Override
  public void onInstantiatedRenderArea(RenderArea renderArea)
  {
    LoadBalancedRenderAreaListener renderAreaListener = new LoadBalancedRenderAreaListener();
    renderArea.addListener(renderAreaListener);
    renderArea.getTouchManager().addDefaultRecognizers();

    SceneExaminer sceneExaminer = new SceneExaminer();
    sceneExaminer.addChild(new SoGradientBackground());
    SoMaterial material = new SoMaterial();
    material.diffuseColor.setValue(this.color);
    sceneExaminer.addChild(material);
    sceneExaminer.addChild(new SoCone());

    renderArea.getSceneManager().setSceneGraph(sceneExaminer);
    sceneExaminer.viewAll(renderArea.getSceneManager().getViewportRegion());
  }

  @Override
  public void onDisposedRenderArea(String renderAreaId)
  {
    LoadBalancerClient.deleteRenderArea(renderAreaId);
  }
}
