"use strict";

/**
 * "better-sqlite3" is used instead of "sqlite3" because it does 
 * not requires asynchronous functions.
 */
const sqlite3 = require('better-sqlite3');

/**
 * This class is used to describe a RemoteViz Instance.
 */
class Instance {
  /**
   * Create a new instance object.
   * @param  {} ip The instance IP address, e.g. "127.0.0.1".
   * @param  {} port The instance port, e.g. 8080.
   * @param  {} id The instance id (optionnal).
   */
  constructor(ip, port, id) {
    this._ip = ip;
    this._port = port;
    this._id = id;
  }
  get ip() {
    return this._ip;
  }
  get port() {
    return this._port;
  }
  get id() {
    return this._id;
  }
  toJSON() {
    return {
      ip: this._ip,
      port: this._port,
      id: this._id
    }
  }
}

/**
 * This class is used to describe a RemoteViz RenderArea.
 */
class RenderArea {
  /**
   * Create a new RenderArea object.
   * @param  {} name The name of the RenderArea, e.g. "TheCone".
   */
  constructor(name) {
    this._name = name;
  }
  get name() {
    return this._name;
  }
  set name(name) {
    this._name = name;
  }
  toJSON() {
    return {
      name: this._name
    }
  }
}

/**
 * The class is used to manage all the database accesses.
 */
class Database {
  /**
   * Create a new Database object.
   * @param  {} db_path The path to the sqlite database file, can be ":memory:" for an in memory database.
   * @param  {} log Activate logging.
   */
  constructor(db_path, log = false) {
    var options = {};
    if (log)
      options = { verbose: console.log };
    // Create SQLite database
    this.db = new sqlite3(db_path, options);
    // Create tables to store the instances and their renderareas.
    this.db.exec(`
      CREATE TABLE IF NOT EXISTS Instance
      (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        ip VARCHAR(255) NOT NULL,
        port INTEGER NOT NULL,
        UNIQUE (ip, port)
      );
      CREATE TABLE IF NOT EXISTS RenderArea
      (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name VARCHAR(255) NOT NULL UNIQUE
      );
      CREATE TABLE IF NOT EXISTS InstanceRenderareas
      (
        InstanceRefId INTEGER NOT NULL REFERENCES Instance(id) ON DELETE CASCADE,
        RenderAreaRefId INTEGER NOT NULL REFERENCES RenderArea(id) ON DELETE CASCADE,
        UNIQUE (InstanceRefId, RenderAreaRefId)
      );
    `);
  }
  /**
   * Close the SQLite database properly.
   */
  close() {
    this.db.close();
  }
  /**
   * Create a new entry in the database for the specified ip/port pair.
   * @param  {} ip The new instance IP address, e.g. "127.0.0.1".
   * @param  {} port The new instance port, e.g. 8080.
   * @returns {Instance} An Instance object or null.
   */
  registerInstance(ip, port) {
    // Create instance
    this.db.prepare(`INSERT INTO Instance (ip,port) VALUES (?, ?)`).run(ip, port);

    return this.getInstanceByIpAndPort(ip, port);
  }
  /**
   * Delete the specified instance from the database.
   * @param  {Instance} instance An Instance object.
   */
  deleteInstance(instance) {
    if (!(instance instanceof Instance)) {
      throw new Error('Please provide an Instance object.');
    }
    this.db.prepare(`DELETE FROM Instance WHERE id = ?`).run(instance.id);
  }
  /**
   * Returns the instance linked to the specified RenderArea.
   * @param  {RenderArea} renderArea A RenderArea object.
   * @returns {Instance} An Instance object or null.
   */
  getInstance(renderArea) {
    if (!(renderArea instanceof RenderArea)) {
      throw new Error('Please provide a RenderArea object.');
    }
    // Get Instances linked to a RenderArea
    const stmt = this.db.prepare(`
      SELECT * FROM Instance AS i
      INNER JOIN InstanceRenderareas AS itora ON itora.InstanceRefId = i.id
      INNER JOIN RenderArea AS ra ON ra.id = itora.RenderAreaRefId
      WHERE ra.name = ?;
    `);
    const instance = stmt.get(renderArea.name);
    if (instance)
      return new Instance(instance.ip, instance.port, instance.id);
    return null;
  }
  /**
  * Returns the instance from its id.
  * @param  {id} instance id.
  * @returns {Instance} An Instance object or null.
  */
  getInstanceById(id) {
    // Get Instances linked to a RenderArea
    const stmt = this.db.prepare(`
      SELECT * FROM Instance
      WHERE id = ?;
    `);
    const instance = stmt.get(id);
    if (instance)
      return new Instance(instance.ip, instance.port, instance.id);
    return null;
  }
  /**
  * Returns the instance from its ip address and port number.
  * @param  {ip} instance ip address.
  * @param  {port} instance port number.
  * @returns {Instance} An Instance object or null.
  */
  getInstanceByIpAndPort(ip, port) {
    // Get Instances linked to a RenderArea
    const stmt = this.db.prepare(`
      SELECT * FROM Instance
      WHERE ip = ? AND port = ?;
   `);
    const instance = stmt.get(ip, port);
    if (instance)
      return new Instance(instance.ip, instance.port, instance.id);
    return null;
  }
  /**
   * Returns all the current registered instances.
   * @returns {Instance[]} An array of Instances.
   */
  getInstances() {
    var instances = [];
    const stmt = this.db.prepare('SELECT * FROM Instance');
    for (const i of stmt.iterate()) {
      instances.push(new Instance(i.ip, i.port, i.id));
    }
    return instances;
  }
  /**
   * Returns the number of RenderAreas linked to an Instance.
   * @param  {Instance} instance An Instance object.
   * @returns {number} The number of RenderAreas linked to "instance".
   */
  getNumRenderAreas(instance) {
    if (!(instance instanceof Instance)) {
      throw new Error('Please provide an Instance object.');
    }
    // Count the number of Render Areas linked to this Instance
    const stmt = this.db.prepare(`
      SELECT COUNT(RenderAreaRefId)
      FROM InstanceRenderareas
      WHERE InstanceRefId = ?;
    `);
    return stmt.get(instance.id)['COUNT(RenderAreaRefId)'];
  }
  /**
   * Create a new entry in the database for the specified renderArea 
   * if not already here, and link it to the specified instance.
   * @param  {RenderArea} renderArea A RenderArea object.
   * @param  {Instance} instance An Instance object.
   */
  createRenderArea(renderArea, instance) {
    if (!(instance instanceof Instance)) {
      throw new Error('Please provide an Instance object.');
    }
    if (!(renderArea instanceof RenderArea)) {
      throw new Error('Please provide a RenderArea object.');
    }
    // Create a new entry in the database for the renderArea, ignore if already here.
    this.db.prepare(`INSERT OR IGNORE INTO RenderArea (name) VALUES (?)`).run(renderArea.name);
    // Get the id of the previously added renderArea.
    const renderAreaObj = this.db.prepare(`SELECT id FROM RenderArea WHERE name = ?`).get(renderArea.name);
    // Create a link between the renderArea and the instance.
    this.db.prepare(`INSERT INTO InstanceRenderareas (InstanceRefId,RenderAreaRefId) VALUES (?,?)`).run(instance.id, renderAreaObj.id);
  }
  /**
   * Remove the specified renderArea from the database.
   * @param  {RenderArea} renderArea A RenderArea object.
   */
  deleteRenderArea(renderArea) {
    if (!(renderArea instanceof RenderArea)) {
      throw new Error('Please provide a RenderArea object.');
    }
    this.db.prepare(`DELETE FROM RenderArea WHERE name = ?`).run(renderArea.name);
  }
}

/**
 * Exports all classes previously created.
 */
module.exports = {
  Instance,
  RenderArea,
  Database
};
