package remoteviz.monitoring.monitoringcsv;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;

import com.openinventor.remoteviz.rendering.Client;
import com.openinventor.remoteviz.rendering.Connection;
import com.openinventor.remoteviz.rendering.MetricsListener;
import com.openinventor.remoteviz.rendering.RenderArea;

public class MonitoringMetrics extends MetricsListener
{
  private StringJoiner csvJoiner;
  private List<String> csvHeaders;
  private final String csvSeparator = ";";
  private final String timestampCsvField = "timestamp";
  private final String clientIdCsvField = "client id";
  private final String connectionIdCsvField = "connection id";
  private final String renderAreaIdCsvField = "renderarea id";
  private final String networkLatencyCsvField = "network latency";
  private final String decodingTimeCsvField = "decoding time";
  private final String renderingTimeCsvField = "rendering time";
  private final String encodingTimeCsvField = "encoding time";
  private final String numClientsCsvField = "number clients";
  private final String numConnectionsCsvField = "number connections";
  private final String numRenderAreasCsvField = "number render areas";

  public MonitoringMetrics()
  {
    csvJoiner = new StringJoiner("\n");
    csvHeaders = Arrays.asList(
      timestampCsvField,
      clientIdCsvField,
      connectionIdCsvField,
      renderAreaIdCsvField,
      networkLatencyCsvField,
      decodingTimeCsvField,
      renderingTimeCsvField,
      encodingTimeCsvField,
      numClientsCsvField,
      numConnectionsCsvField,
      numRenderAreasCsvField
    );
    csvJoiner.add(String.join(csvSeparator, csvHeaders));
  }

  private void writeInColumn(Map<Integer, String> values)
  {
    String[] line = new String[csvHeaders.size()];
    Arrays.fill(line, "");
    line[0] = String.valueOf(System.currentTimeMillis());

    for (Map.Entry<Integer, String> value : values.entrySet())
    {
      line[value.getKey()] = value.getValue();
    }
    csvJoiner.add(String.join(csvSeparator, line));
  }

  public void SaveMetrics()
  {
    try
    {
      PrintWriter pw = new PrintWriter(new File("monitoring.csv"));
      pw.write(csvJoiner.toString());
      pw.close();
    }
    catch (FileNotFoundException e)
    {
      System.out.println("Can't create monitoring.csv : "+e);
    }
  }

  @Override
  public void onMeasuredNetworkLatency(int time, Client client)
  {
    Map<Integer, String> values = new HashMap<>();
    values.put(csvHeaders.indexOf(clientIdCsvField), client.getId());
    values.put(csvHeaders.indexOf(networkLatencyCsvField), String.valueOf(time));
    writeInColumn(values);
  }
  
  @Override
  public void onMeasuredDecodingTime(int time, Connection connection)
  {
    Map<Integer, String> values = new HashMap<>();
    values.put(csvHeaders.indexOf(connectionIdCsvField), connection.getId());
    values.put(csvHeaders.indexOf(decodingTimeCsvField), String.valueOf(time));
    writeInColumn(values);
  }
  
  @Override
  public void onMeasuredRenderingTime(int time, RenderArea renderArea)
  {
    Map<Integer, String> values = new HashMap<>();
    values.put(csvHeaders.indexOf(renderAreaIdCsvField), renderArea.getId());
    values.put(csvHeaders.indexOf(renderingTimeCsvField), String.valueOf(time));
    writeInColumn(values);
  }
  
  @Override
  public void onMeasuredEncodingTime(int time, Connection connection)
  {
    Map<Integer, String> values = new HashMap<>();
    values.put(csvHeaders.indexOf(connectionIdCsvField), connection.getId());
    values.put(csvHeaders.indexOf(encodingTimeCsvField), String.valueOf(time));
    writeInColumn(values);
  }
  
  @Override
  public void onChangedNumClients(int number)
  {
    Map<Integer, String> values = new HashMap<>();
    values.put(csvHeaders.indexOf(numClientsCsvField), String.valueOf(number));
    writeInColumn(values);
  }
  
  @Override
  public void onChangedNumConnections(int number)
  {
    Map<Integer, String> values = new HashMap<>();
    values.put(csvHeaders.indexOf(numConnectionsCsvField), String.valueOf(number));
    writeInColumn(values);
  }
  
  @Override
  public void onChangedNumRenderAreas(int number)
  {
    Map<Integer, String> values = new HashMap<>();
    values.put(csvHeaders.indexOf(numRenderAreasCsvField), String.valueOf(number));
    writeInColumn(values);
  }
  
}
