package volumeviz.advanced.volRend;

import java.io.File;

import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbVec3i32;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoOffscreenRenderArea;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.SoSceneManager.AntialiasingModes;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.actions.SoWriteAction;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.SoEventCallback;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoPickStyle;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor.CameraMode;
import com.openinventor.ldm.SoLDMGlobalResourceParameters;
import com.openinventor.ldm.nodes.SoLDMResourceParameters;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRendering;

import util.editors.MaterialEditor;

public class VolRend
{
  public static final float PIO2 = 1.570796326794896619231f;
  public static final int MIN_COLORMAP = 0;
  public static final int MAX_COLORMAP = 255;
  private static final int CMAP_SHIFT = 1;
  private static final String DEFAULT_COLORMAP_FILENAME = "BlueWhiteRed.txt";
  public static final String DEFAULT_ALPHA = "0.3f";

  String[] m_fileNames = { "3DHEAD.ldm", "virus.am", "brill.am", "CThead8.am", "CThead16.am", "agipAVS.fld",
      "SYN_64.VOL", "3DHEAD.VOL", "ENGINE.VOL", "LOBSTER.VOL", "COLT.VOL", };
  static String m_filePrefix;
  private SbVec3i32 m_volDataDim;
  boolean m_showFrameRate;
  String m_userColorMapName;
  static int m_minColorMap, m_maxColorMap, m_minOpaqueMap, m_maxOpaqueMap;
  static int m_minSliceColorMap, m_maxSliceColorMap, m_minSliceOpaqueMap, m_maxSliceOpaqueMap;
  static boolean m_invertTransparency;
  static boolean m_invertSliceTransparency;
  boolean m_newFile = true;

  VolRendProfView m_profileView;
  IViewerExaminer m_viewer;
  SoSeparator m_root;
  SoVolumeData m_volData;
  SoTransferFunction m_sliceTransferFunction;

  // Slice group
  SliceGroup m_sliceGroup;

  SoMaterial m_material;
  MaterialEditor m_mtlEditor;
  VolAxesSwitch m_volAxesSwitch;
  VolBBoxSwitch m_volBBoxSwitch;
  SoEventCallback m_mouseMoveEvent;
  SoEventCallback m_mouseKeyEvent;
  SoTransferFunction m_transferFunction;

  // VolRender switch
  VolRenderSwitch m_volRenderSwitch;

  // Info
  InfoSep m_infoSep;

  // Background
  SoGradientBackground m_background;

  SoSwitch m_textSwitch;

  public VolRend()
  {
    m_filePrefix = SoPreferences.getValue("OIVJHOME") + "/data/volumeviz/";
    m_showFrameRate = false;
    m_profileView = null;

    m_minColorMap = MIN_COLORMAP;
    m_maxColorMap = MAX_COLORMAP;
    m_minOpaqueMap = MIN_COLORMAP;
    m_maxOpaqueMap = MAX_COLORMAP;
    m_minSliceColorMap = MIN_COLORMAP;
    m_maxSliceColorMap = MAX_COLORMAP;
    m_minSliceOpaqueMap = MIN_COLORMAP;
    m_maxSliceOpaqueMap = MAX_COLORMAP;

    m_invertTransparency = false;
    m_invertSliceTransparency = false;
  }

  public void stop()
  {
    m_mtlEditor.dispose();
    m_volRenderSwitch.m_headlightEd.dispose();
    if ( m_profileView != null )
      m_profileView.hideProfile();
  }

  public SoSeparator makeSceneGraph()
  {
    setGlobalOptions();

    // Volume data
    m_volData = new SoVolumeData();
    m_volData.setName("volumeData");
    m_volData.fileName.setValue(m_filePrefix + m_fileNames[0]);
    m_volData.usePalettedTexture.setValue(true);

    setVolRenderingParam();

    m_minOpaqueMap = m_minColorMap = MIN_COLORMAP + CMAP_SHIFT;
    m_maxOpaqueMap = m_maxColorMap = MAX_COLORMAP;
    // Don't apply CMAP_SHIFT if user specified min value
    String min_colorMap_prop = System.getProperty("VOLREND_minColorMap");
    if ( min_colorMap_prop != null && Integer.valueOf(min_colorMap_prop).intValue() != 0 )
    {
      String min_value = System.getProperty("VOLREND_minColorMap", String.valueOf(MIN_COLORMAP));
      m_minColorMap = Integer.valueOf(min_value).intValue();
      String max_value = System.getProperty("VOLREND_maxColorMap", String.valueOf(MAX_COLORMAP));
      m_maxColorMap = Integer.valueOf(max_value).intValue();
      min_value = System.getProperty("VOLREND_minOpaqueMap", min_value);
      m_minOpaqueMap = Integer.valueOf(min_value).intValue();
      max_value = System.getProperty("VOLREND_minColorMap", max_value);
      m_maxOpaqueMap = Integer.valueOf(max_value).intValue();
    }

    // Slice transfer function
    m_sliceTransferFunction = new SoTransferFunction();
    String sliceColorMap_prop = System.getProperty("VOLREND_sliceColorMap",
        String.valueOf(SoTransferFunction.PredefColorMaps.TEMPERATURE.getValue()));
    int sliceColorMap = Integer.valueOf(sliceColorMap_prop).intValue();

    if ( sliceColorMap == 0 )
    {
      String filename = System.getProperty("VOLREND_sliceColorMapFile");
      if ( filename == null || filename.length() == 0 )
        filename = m_filePrefix + DEFAULT_COLORMAP_FILENAME;
      UserColorMap ucm = new UserColorMap(filename);
      if ( ucm.m_values != null )
      {
        m_sliceTransferFunction.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.NONE);
        m_sliceTransferFunction.colorMapType.setValue(SoTransferFunction.ColorMapTypes.RGBA);
        m_sliceTransferFunction.colorMap.setValues(0, ucm.m_values);
        m_userColorMapName = filename;
      }
    }
    else
      m_sliceTransferFunction.predefColorMap.setValue(sliceColorMap);
    m_sliceTransferFunction.minValue.setValue(m_minColorMap);
    m_sliceTransferFunction.maxValue.setValue(m_maxColorMap);
    updateOpaqueRegion(m_sliceTransferFunction, m_minOpaqueMap, m_maxOpaqueMap, m_invertTransparency);

    // Ortho slices
    m_sliceGroup = new SliceGroup(m_volData);

    // Lighting / Material
    // The ambientColor only seems to have effect when lighting is
    // enabled, but we need a high value to get "reasonable" appearance.
    String alpha_prop = System.getProperty("VOLREND_globalAlpha", DEFAULT_ALPHA);
    float alpha = Float.parseFloat(alpha_prop);
    if ( alpha < 0 || alpha > 1 )
      alpha = Float.parseFloat(DEFAULT_ALPHA);
    m_material = new SoMaterial();
    m_material.ambientColor.setValue(.7f, .7f, .7f);
    m_material.diffuseColor.setValue(1, 1, 1);
    m_material.transparency.setValue(1 - alpha);
    m_material.specularColor.setValue(1, 1, 1);
    m_material.shininess.setValue(0.5f);
    m_mtlEditor = new MaterialEditor();
    m_mtlEditor.setTitle("Material of Volume Rendering");
    m_mtlEditor.attach(m_material);
    // When the material editor comes up it does not display the
    // actual values in this material. Probably a bug, but this
    // should force it to be correct.
    m_mtlEditor.setMaterial(m_material);

    // Volume render
    m_volRenderSwitch = new VolRenderSwitch(m_volData);
    m_sliceGroup.insertChild(m_volRenderSwitch.m_volROISwitch, 0);

    // Axes, Bounding box
    m_volBBoxSwitch = new VolBBoxSwitch(m_volData);
    m_volAxesSwitch = new VolAxesSwitch(m_volData);
    m_volBBoxSwitch.setName("BBoxSwitch"); // debug
    m_volAxesSwitch.setName("AxesSwitch"); // debug

    // Update BBox/Axes (doesn't do anything if not enabled)
    m_volAxesSwitch.update();
    m_volBBoxSwitch.update();

    SoPickStyle pUnpickable = new SoPickStyle();
    pUnpickable.style.setValue(SoPickStyle.Styles.UNPICKABLE);

    m_mouseMoveEvent = new SoEventCallback();
    // Add an event callback to catch mouse key event.
    m_mouseKeyEvent = new SoEventCallback();
    // Also use this node to see key press events for frame rate
    // (we need the top level window handle to display frame rate)
    m_mouseKeyEvent.addEventCallback(SoKeyboardEvent.class, new KeyEventCB());

    // Transfer function for volume
    m_transferFunction = new SoTransferFunction();
    String prop =
        System.getProperty("VOLREND_colorMap", String.valueOf(SoTransferFunction.PredefColorMaps.SEISMIC.getValue()));
    int colorMapIndex = Integer.valueOf(prop).intValue();

    // Check for user defined colormap
    if ( colorMapIndex == 0 )
    {
      String filename = System.getProperty("VOLREND_colorMapFile");
      if ( filename == null || filename.length() == 0 )
        filename = m_filePrefix + DEFAULT_COLORMAP_FILENAME;
      UserColorMap ucm = new UserColorMap(filename);
      if ( ucm.m_values != null )
      {
        m_transferFunction.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.NONE);
        m_transferFunction.colorMapType.setValue(SoTransferFunction.ColorMapTypes.RGBA);
        m_transferFunction.colorMap.setValues(0, ucm.m_values);
        m_userColorMapName = filename;
      }
      else
        colorMapIndex = SoTransferFunction.PredefColorMaps.SEISMIC.getValue();
    }
    // Use predef colormap if no user colormap
    if ( colorMapIndex > 0 )
      m_transferFunction.predefColorMap.setValue(colorMapIndex);
    m_transferFunction.minValue.setValue(m_minColorMap);
    m_transferFunction.maxValue.setValue(m_maxColorMap);

    updateOpaqueRegion(m_transferFunction, m_minOpaqueMap, m_maxOpaqueMap, m_invertTransparency);
    m_transferFunction.setName("TransferFunction");

    // background
    m_background = new SoGradientBackground();

    m_root = new SoSeparator();
    {
      m_root.addChild(m_background);
      m_root.addChild(m_volData);

      m_root.addChild(m_sliceTransferFunction);
      m_root.addChild(m_sliceGroup);
      m_root.addChild(m_material);

      // Moved the bbox and axes up under m_root, instead of VolRendGrp,
      // so they can still be visible even when the volume is turned off.
      // Note: BBox and axes must not be pickable or they will interfere
      // with picking the slices and volume
      m_root.addChild(pUnpickable);
      m_root.addChild(m_volAxesSwitch);
      m_root.addChild(m_volBBoxSwitch);
      m_root.addChild(new SoPickStyle());

      // This part is used for the picking in VolumeRendering.
      // Add an event callback to catch mouse moves.
      m_root.addChild(m_mouseMoveEvent);
      m_root.addChild(m_mouseKeyEvent);
      m_root.addChild(m_transferFunction);
      m_root.addChild(m_volRenderSwitch);
    }

    // Info
    m_infoSep = new InfoSep(m_volData, m_transferFunction, m_sliceTransferFunction);

    SoSeparator main_root = new SoSeparator();
    {
      main_root.addChild(m_root);
      main_root.addChild(m_infoSep);
    }

    m_sliceGroup.addOrthoDraggerSwitch(m_root);

    // Get the voxel dimensions of the volume
    m_volDataDim = m_volData.data.getSize();

    return main_root;
  }

  public void setUpViewer(IViewerExaminer viewer, SoSeparator root)
  {
    m_viewer = viewer;

    // Use perspective camera by default
    // But note orthographic camera works better with VolumePro
    boolean use_persp_cam = true;
    String prop = System.getProperty("VOLREND_cameraType");
    if ( prop != null && (prop.startsWith("0") || prop.startsWith("o") || prop.startsWith("O")) )
      use_persp_cam = false;

    if ( use_persp_cam )
      m_viewer.getRenderArea().setCameraType(CameraMode.PERSPECTIVE);
    else
      m_viewer.getRenderArea().setCameraType(CameraMode.ORTHOGRAPHIC);

    m_viewer.getRenderArea().setTransparencyType(SoGLRenderAction.TransparencyTypes.NO_SORT);
    m_viewer.setSceneGraph(root);
    m_viewer.viewAll();

    m_viewer.getRenderArea().saveCamera();
  }

  private void setGlobalOptions()
  {

    // Enable writing alternateRep geometry for VolViz slices
    SoVolumeRendering.setWriteAlternateRep(true);

    System.setProperty("SO_DRAGGER_DIR", ".");

    String value;
    value = System.getProperty("VOLREND_viewCulling", "true");
    SoLDMGlobalResourceParameters.setViewCulling(Boolean.valueOf(value).booleanValue());

    value = System.getProperty("VOLREND_viewPointRefine", "true");
    SoLDMGlobalResourceParameters.setViewpointRefinement(Boolean.valueOf(value).booleanValue());

    value = System.getProperty("VOLREND_sliceEqualRes", "false");
    SoLDMGlobalResourceParameters.setSliceEqualResolution(Boolean.valueOf(value).booleanValue());

    value = System.getProperty("VOLREND_tileOutline", "false");
    SoLDMGlobalResourceParameters.setVisualFeedbackParam(
        SoLDMGlobalResourceParameters.VisualFeedbackParams.DRAW_TILE_OUTLINE, Boolean.valueOf(value).booleanValue());

    value = System.getProperty("VOLREND_dataOutline", "false");
    SoLDMGlobalResourceParameters.setVisualFeedbackParam(
        SoLDMGlobalResourceParameters.VisualFeedbackParams.DRAW_TOPOLOGY, Boolean.valueOf(value).booleanValue());

    // For VolumePro, disable embedded geometry support
    // Allows better default performance for demos. :-)
    System.setProperty("IVVP_EMBEDDED_GEOM", "false");
  }

  // Set volume rendering parameters
  private void setVolRenderingParam()
  {
    String prop = System.getProperty("VOLREND_mainMemSize", "-1"); // 1 - 512
    int value = Integer.valueOf(prop).intValue();
    if ( value >= 0 )
      m_volData.ldmResourceParameters.getValue().maxMainMemory.setValue(value);

    prop = System.getProperty("VOLREND_texMemSize", "-1"); // 1 - 128
    value = Integer.valueOf(prop).intValue();
    if ( value >= 0 )
      m_volData.ldmResourceParameters.getValue().maxTexMemory.setValue(value);

    prop = System.getProperty("VOLREND_texLoadRate", "-1"); // 0 - 64
    value = Integer.valueOf(prop).intValue();
    if ( value >= 0 )
      m_volData.ldmResourceParameters.getValue().tex3LoadRate.setValue(value);

    prop = System.getProperty("VOLREND_sliceTexMem", "-1"); // 1 - 1024
    value = Integer.valueOf(prop).intValue();
    if ( value >= 0 )
      m_volData.ldmResourceParameters.getValue().max2DTexMemory.setValue(value);

    prop = System.getProperty("VOLREND_sliceTexLoadRate", "-1"); // 1- 256
    value = Integer.valueOf(prop).intValue();
    if ( value >= 0 )
      m_volData.ldmResourceParameters.getValue().tex2LoadRate.setValue(value);

    // Load tiles policy
    prop = System.getProperty("VOLREND_moveLowRes", "false");
    if ( !Boolean.valueOf(prop) )
      m_volData.ldmResourceParameters.getValue().loadPolicy.setValue(SoLDMResourceParameters.LoadPolicies.ALWAYS);
  }

  ///////////////////////////////////////////////////////////////////////
  //
  // Make the color map completely transparent *except*
  // in the specified range (where alpha is left alone)
  public void updateOpaqueRegion(SoTransferFunction tf, int min, int max, boolean invertFlag)
  {

    float[] actualColorMapF = tf.actualColorMap.getValues(0);
    int length = actualColorMapF.length / 4;
    if ( !invertFlag )// remap inside min map
    {
      // If min is zero, no change.
      // If min is less than zero, also no change (bogus case).
      for ( int i = 0; i < min; i++ )
        actualColorMapF[4 * i + 3] = 0.f;
      // If max = length-1 (last index in color map), no change.
      // If max > length-1, also no change (bogus case).
      for ( int k = max + 1; k < length; k++ )
        actualColorMapF[4 * k + 3] = 0.f;
    }
    else
    {
      for ( int i = min; i <= max; i++ )
      {
        actualColorMapF[4 * i + 3] = 0.f;
      }
    }
    tf.actualColorMap.setValues(0, actualColorMapF);
  }

  public void doDataChange(File f)
  {
    m_newFile = true;
    if ( m_textSwitch != null )
      m_textSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);

    m_volData.fileName.setValue(f.getAbsolutePath());
    setVolRenderingParam();

    // Update scale factor
    float[] vol_size_values = m_volData.extent.getValue().getSize().getValue();
    float avg_size = (vol_size_values[0] + vol_size_values[1] + vol_size_values[2]) / 3;
    float scale_factor = SliceGroup.SCALE_FACTOR_INIT * avg_size;
    m_sliceGroup.m_scaleFactor = scale_factor;
    m_sliceGroup.m_draggerObliSlice.scaleFactor.setValue(scale_factor, scale_factor, scale_factor);

    // Update color map
    m_minOpaqueMap = m_minColorMap = MIN_COLORMAP + CMAP_SHIFT;
    m_maxOpaqueMap = m_maxColorMap = MAX_COLORMAP;
    m_transferFunction.minValue.setValue(m_minColorMap);
    m_transferFunction.maxValue.setValue(m_maxColorMap);

    // Get new volume dimensions
    m_volDataDim = m_volData.data.getSize();

    // Reset subvolume and ROI boxes
    SbVec3i32 volMax = new SbVec3i32(m_volDataDim.getX() - 1, m_volDataDim.getY() - 1, m_volDataDim.getZ() - 1);
    // SbVec3i32 center = new SbVec3i32(50, 50, 50);//volMax / 2;
    // SbVec3i32 subBoxMin = center.minus(volMax.times(1/2));
    // SbVec3i32 subBoxMax = center.plus(volMax.times(1/2));
    m_volRenderSwitch.m_volROI.subVolume.setValue(new SbVec3i32(0, 0, 0), volMax);// (subBoxMin,
                                                                                  // subBoxMax);
    m_volRenderSwitch.m_volROI.box.setValue(new SbVec3i32(0, 0, 0), volMax);
    m_volRenderSwitch.m_boxInitSize.setBounds(new SbVec3i32(0, 0, 0), volMax);// (subBoxMin,
                                                                              // subBoxMax);
    m_volRenderSwitch.m_subVolInitSize.setBounds(new SbVec3i32(0, 0, 0), volMax);
    m_volRenderSwitch.m_ROIManip.subVolume.setValue(new SbVec3i32(0, 0, 0), volMax);// (subBoxMin,
                                                                                    // subBoxMax);
    m_volRenderSwitch.m_ROIManip.box.setValue(new SbVec3i32(0, 0, 0), volMax);
    m_volRenderSwitch.m_roiManipBoxBBoxSwitch.updateInitTransform();
    m_volRenderSwitch.m_roiManipSubVolBBoxSwitch.updateInitTransform();

    m_volRenderSwitch.m_volRend.numSlices.setValue(m_volDataDim.getX() - 1);
    // Re-create bbox, axes, legend, etc.
    m_volBBoxSwitch.update();
    m_volAxesSwitch.update();
    m_infoSep.createHistoLegend(0);

    // Update the volume info annotation text
    m_infoSep.updateText();
  }

  public void doCmapChange(int whichColormap, SoTransferFunction transferFunc)
  {
    switch ( whichColormap )
    {
    case 0 :
      transferFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.SEISMIC);
      break;
    case 1 :
      transferFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.STANDARD);
      break;
    case 2 :
      transferFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.GLOW);
      break;
    case 3 :
      transferFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.BLUE_RED);
      break;
    case 4 :
      transferFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.PHYSICS);
      break;
    case 5 :
      transferFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.TEMPERATURE);
      break;
    case 6 :
      transferFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.GREY);
      break;
    case 7 :
      transferFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.BLUE_WHITE_RED);
      break;
    case 8 :
      // Load user defined colormap
      if ( m_userColorMapName == "" )
        m_userColorMapName = m_filePrefix + DEFAULT_COLORMAP_FILENAME;

      UserColorMap ucm = new UserColorMap(m_userColorMapName);

      if ( ucm.m_numEntries == 0 )
        return;

      // Load values into transfer function node
      // Note: setNum() is required in case the new color map has
      // fewer values than the current one (we compute the length
      // of the colormap based on the number of values in this field).
      transferFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.NONE);
      switch ( ucm.m_numComps )
      {
      case 1 : // Alpha
        transferFunc.colorMapType.setValue(SoTransferFunction.ColorMapTypes.ALPHA);
        transferFunc.colorMap.setNum(ucm.m_numEntries);
        transferFunc.colorMap.setValues(0, ucm.m_values);
        break;
      case 2 : // Luminance_alpha
        transferFunc.colorMapType.setValue(SoTransferFunction.ColorMapTypes.LUM_ALPHA);
        transferFunc.colorMap.setNum(ucm.m_numEntries * 2);
        transferFunc.colorMap.setValues(0, ucm.m_values);
        break;
      case 4 : // RGBA
        transferFunc.colorMapType.setValue(SoTransferFunction.ColorMapTypes.RGBA);
        transferFunc.colorMap.setNum(ucm.m_numEntries * 4);
        transferFunc.colorMap.setValues(0, ucm.m_values);
        break;
      default:
        // Error, unsupported format
        break;
      }
      break;
    default:
      break;
    }

    int minColorMap, maxColorMap, minOpaqueMap, maxOpaqueMap;
    boolean invertFlag;
    if ( transferFunc == m_transferFunction )
    {
      minColorMap = m_minColorMap;
      maxColorMap = m_maxColorMap;
      minOpaqueMap = m_minOpaqueMap;
      maxOpaqueMap = m_maxOpaqueMap;
      invertFlag = m_invertTransparency;
    }
    else
    {
      minColorMap = m_minSliceColorMap;
      maxColorMap = m_maxSliceColorMap;
      minOpaqueMap = m_minSliceOpaqueMap;
      maxOpaqueMap = m_maxSliceOpaqueMap;
      invertFlag = m_invertSliceTransparency;
    }
    transferFunc.minValue.setValue(minColorMap);
    transferFunc.maxValue.setValue(maxColorMap);
    updateOpaqueRegion(transferFunc, minOpaqueMap, maxOpaqueMap, invertFlag);
  }

  class KeyEventCB extends SoEventCallbackCB
  {
    int change = 0;

    @Override
    public void invoke(SoEventCallback node)
    {
      SoEvent pEvent = node.getEvent();

      // Key 'R' requests a redraw
      if ( SoKeyboardEvent.isKeyPressEvent(pEvent, SoKeyboardEvent.Keys.R) )
      {
        m_viewer.getRenderArea().scheduleRedraw();
      }
      else if ( SoKeyboardEvent.isKeyReleaseEvent(pEvent, SoKeyboardEvent.Keys.A) )
      { // Antialiasing (for production screen shots)
        float antialiasingQuality = m_viewer.getRenderArea().getAntialiasingQuality();
        if ( antialiasingQuality == 0 )
        {
          if ( pEvent.wasShiftDown() )
            antialiasingQuality = 1.0f;
          else
            antialiasingQuality = 0.5f;
        }
        else
          antialiasingQuality = 0.f; // disable
        m_viewer.getRenderArea().setAntialiasingMode(AntialiasingModes.AUTO);
        m_viewer.getRenderArea().setAntialiasingQuality(antialiasingQuality);
      }
      else if ( SoKeyboardEvent.isKeyReleaseEvent(pEvent, SoKeyboardEvent.Keys.B) )
      {
        int wDivide = 1;
        int hDivide = 1;
        int zDivide = 5;
        int wValue = m_volDataDim.getX() / wDivide;
        int hValue = m_volDataDim.getY() / hDivide;
        int dValue = m_volDataDim.getZ() / zDivide;
        switch ( change )
        {
        case 0 :
          m_volRenderSwitch.m_ROIManip.subVolume.setValue(0, 0, 0, wValue, hValue, dValue);
          break;
        case 1 :
          m_volRenderSwitch.m_ROIManip.subVolume.setValue(0, 0, 2 * dValue, wValue, hValue, 3 * dValue);
          break;
        case 2 :
          m_volRenderSwitch.m_ROIManip.subVolume.setValue(0, 0, 4 * dValue, wValue, hValue, 5 * dValue);
          break;
        }
        change++;
        if ( change == 3 )
          change = 0;
      }
      else if ( SoKeyboardEvent.isKeyReleaseEvent(pEvent, SoKeyboardEvent.Keys.E) )
      { // Embedded geometry toggle for VolumePro 1000
        boolean state = Boolean.valueOf(System.getProperty("IVVP_EMBEDDED_GEOM", "true")).booleanValue();
        if ( state )
          state = false;
        else
          state = true;
        System.setProperty("IVVP_EMBEDDED_GEOM", String.valueOf(state));
        m_viewer.getRenderArea().scheduleRedraw();
      }
      else if ( SoKeyboardEvent.isKeyReleaseEvent(pEvent, SoKeyboardEvent.Keys.P) )
      { // Print
        float antiAliasingQuality = m_viewer.getRenderArea().getAntialiasingQuality();
        SbViewportRegion vpRegion = new SbViewportRegion(m_viewer.getRenderArea().getComponent().getSize());
        SoOffscreenRenderArea offscreen = new SoOffscreenRenderArea();
        offscreen.setSceneGraph(m_viewer.getRenderArea().getSceneInteractor());
        offscreen.setViewportRegion(vpRegion);
        offscreen.setTransparencyType(SoGLRenderAction.TransparencyTypes.NO_SORT);
        offscreen.setAntialiasingQuality(antiAliasingQuality);
        offscreen.setAntialiasingMode(AntialiasingModes.AUTO);
        boolean ok = offscreen.renderToFile("offscreen.jpg");
        if ( !ok )
          System.err.println("OffscreenRender failed");
      }
      else if ( SoKeyboardEvent.isKeyReleaseEvent(pEvent, SoKeyboardEvent.Keys.W) )
      {
        // Write to file with alternateRep enabled.
        // (don't write headlight, viewer will set up its own when file is read)
        SoVolumeRendering.setWriteAlternateRep(true);
        SoWriteAction wa = new SoWriteAction();
        boolean isHL = m_viewer.getRenderArea().getSceneInteractor().isHeadlightEnabled();
        m_viewer.getRenderArea().getSceneInteractor().enableHeadlight(false);
        wa.getOutput().openFile("test.iv");
        wa.apply(m_root);
        wa.getOutput().closeFile();
        m_viewer.getRenderArea().getSceneInteractor().enableHeadlight(isHL);
      }
      else if ( SoKeyboardEvent.isKeyReleaseEvent(pEvent, SoKeyboardEvent.Keys.X) )
      { // X axis view
        if ( pEvent.wasShiftDown() )
          m_viewer.getRenderArea().viewAxis(new SbVec3f(1, 0, 0), new SbVec3f(0, 0, 1));
        else
          m_viewer.getRenderArea().viewAxis(new SbVec3f(-1, 0, 0), new SbVec3f(0, 0, 1));
        m_viewer.viewAll();
      }
      else if ( SoKeyboardEvent.isKeyReleaseEvent(pEvent, SoKeyboardEvent.Keys.Y) )
      { // Y axis view
        if ( pEvent.wasShiftDown() )
          m_viewer.getRenderArea().viewAxis(new SbVec3f(0, -1, 0), new SbVec3f(-1, 0, 0));
        else
          m_viewer.getRenderArea().viewAxis(new SbVec3f(0, -1, 0), new SbVec3f(1, 0, 0));
        m_viewer.viewAll();
      }
      else if ( SoKeyboardEvent.isKeyReleaseEvent(pEvent, SoKeyboardEvent.Keys.Z) )
      { // Z axis view
        if ( pEvent.wasShiftDown() )
          m_viewer.getRenderArea().viewAxis(new SbVec3f(0, 0, 1), new SbVec3f(0, 1, 0));
        else
          m_viewer.getRenderArea().viewAxis(new SbVec3f(0, 0, -1), new SbVec3f(0, 1, 0));
        m_viewer.viewAll();
      }
    }
  }
}
