package volumeviz.sample.backgroundLDMConversion;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.SwingWorker;

import com.openinventor.inventor.SbBox2i32;
import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec3i32;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.devices.SoBufferObject;
import com.openinventor.inventor.devices.SoBufferObject.AccessModes;
import com.openinventor.inventor.devices.SoCpuBufferObject;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.ldm.converters.SoConverter;
import com.openinventor.ldm.converters.SoConverterParameters;
import com.openinventor.ldm.nodes.SoDataSet.DataTypes;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.volumeviz.converters.SoVolumeConverter;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;

import util.Example;
import util.ViewerComponentsFactory;

/**
 * Shows how to launch a custom converter in a background thread.<br>
 * Conversion is launched in a SwingWorker and its progression is visible
 * through a progress bar.
 *
 */
public class Main extends Example
{
  private static final String INPUT_FILENAME = SoPreferences.getValue("OIVJHOME") + "/data/volumeviz/bonsai2.raw";
  private static final String OUTPUT_FILENAME = "./bonsai2.ldm";

  private IViewerExaminer myViewer;
  private GenericReader m_reader;

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain("Background LDM conversion");
  }

  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    m_reader = new GenericReader();

    // data 1
    SoVolumeData pVolData = new SoVolumeData();
    pVolData.setReader(m_reader);

    SoMaterial pMaterial = new SoMaterial();
    pMaterial.transparency.setValue(0.7f);

    // Use a predefined colorMap with the SoTransferFunction
    SoTransferFunction pTransFunc = new SoTransferFunction();
    pTransFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.GLOW);
    pTransFunc.minValue.setValue(40);
    pTransFunc.maxValue.setValue(250);

    // Node in charge of drawing the volume
    SoVolumeRender pVolRender = new SoVolumeRender();
    pVolRender.numSlicesControl.setValue(SoVolumeRender.NumSlicesControls.ALL);

    SbColor bkgColor = new SbColor(new SbColor(.5f, .5f, .5f));
    SoGradientBackground background = new SoGradientBackground();
    background.color0.setValue(bkgColor);
    background.color1.setValue(bkgColor);

    // Assemble the scene graph
    SoSeparator root = new SoSeparator();
    {
      root.addChild(background);
      root.addChild(pVolData);
      root.addChild(pMaterial);
      root.addChild(pTransFunc);
      root.addChild(pVolRender);
    }

    myViewer.setSceneGraph(root);
    myViewer.viewAll();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);

    // Conversion
    JProgressBar progressBar = new JProgressBar();
    JButton convertButton = new JButton("Convert...");
    convertButton.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        ConvertTask task = new ConvertTask();

        task.addPropertyChangeListener(new PropertyChangeListener()
        {
          @Override
          public void propertyChange(PropertyChangeEvent evt)
          {
            if ( "progress".equals(evt.getPropertyName()) )
            {
              progressBar.setValue((Integer) evt.getNewValue());
            }
            else if ( "state".equals(evt.getPropertyName()) )
            {
              if ( SwingWorker.StateValue.STARTED.equals(evt.getNewValue()) )
                convertButton.setEnabled(false);
              else if ( SwingWorker.StateValue.DONE.equals(evt.getNewValue()) )
                convertButton.setEnabled(true);
            }
          }
        });
        task.execute();
      }
    });

    JPanel southPanel = new JPanel(new BorderLayout());
    southPanel.add(convertButton, BorderLayout.WEST);
    southPanel.add(progressBar, BorderLayout.CENTER);
    add(southPanel, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  /**
   * Custom SwingWorker which will launch the conversion.
   *
   */
  class ConvertTask extends SwingWorker<Integer, Integer>
  {
    ConvertTask()
    {}

    @Override
    public Integer doInBackground()
    {
      SoConverterParameters parameters = new SoConverterParameters();
      parameters.setInputFileName(INPUT_FILENAME);
      parameters.setHeaderFileName(OUTPUT_FILENAME);
      parameters.setTileDim(16);
      parameters.setCompressionLevel(9);
      parameters.setCompressionName("gzip");
      parameters.setVerbose(false);

      // Custom converter to update the progress bar
      SoVolumeConverter converter = new SoVolumeConverter()
      {
        @Override
        public boolean progress(int numTilesGenerated, int numTilesToGenerate)
        {
          setProgress(numTilesGenerated * 100 / numTilesToGenerate);
          return true;
        }
      };
      converter.setReader(m_reader);

      try
      {
        // launch conversion
        return converter.convert(parameters);
      }
      catch (Exception e)
      {
        System.out.println(e.getClass().getName() + " : " + e.getMessage());
      }
      return SoConverter.ConverterErrors.CVT_ABORTED.getValue();
    }
  }

  class GenericReader extends com.openinventor.ldm.readers.SoVolumeReader
  {
    private SbBox3f m_size;
    private SbVec3i32 m_dim;
    private int m_type;
    private int m_bytesPerVoxel;
    private int m_headerSize;
    private RandomAccessFile m_dataFile;

    public GenericReader()
    {
      m_size = new SbBox3f(0, 0, 0, 10, 10, 10);
      m_dim = new SbVec3i32(256, 256, 256);
      m_type = SoVolumeData.DataTypes.UNSIGNED_BYTE.getValue();
      m_bytesPerVoxel = 1;
      m_headerSize = 0;
      try
      {
        m_dataFile = new RandomAccessFile(INPUT_FILENAME, "r");
      }
      catch (Exception e)
      {}
    }

    @Override
    public DataInfo getDataChar()
    {
      DataInfo dataInfo = new DataInfo();
      dataInfo.dim = m_dim;
      dataInfo.size = m_size;
      dataInfo.type = DataTypes.valueOf(m_type);
      dataInfo.readError = ReadErrors.RD_NO_ERROR;

      return dataInfo;
    }

    public SbBox3f getVolumeSize()
    {
      return m_size;
    }

    public SbVec3i32 getDataDimension()
    {
      return m_dim;
    }

    public int getDataType()
    {
      return m_type;
    }

    @Override
    public void getSubSlice(SbBox2i32 subSlice, int sliceNumber, SoBufferObject bufferObject)
    {

      long lineSize = m_dim.getX() * m_bytesPerVoxel;
      long sliceSize = m_dim.getY() * lineSize;

      long offset = m_headerSize + sliceNumber * sliceSize + (subSlice.getMin()).getY() * lineSize
          + (long) (subSlice.getMin()).getX() * m_bytesPerVoxel;

      int dataToRead = ((subSlice.getMax()).getX() - (subSlice.getMin()).getX() + 1) * m_bytesPerVoxel;

      SoCpuBufferObject cpuBufObj = new SoCpuBufferObject();
      bufferObject.map(cpuBufObj, AccessModes.SET);

      byte[] data_src = new byte[dataToRead];
      ByteBuffer data_dest = cpuBufObj.map(AccessModes.SET);
      data_dest.rewind();

      try
      {
        for ( int i = (subSlice.getMin()).getY(); i <= (subSlice.getMax()).getY(); i++ )
        {

          m_dataFile.seek(offset);
          m_dataFile.read(data_src);

          data_dest.put(data_src);

          offset += lineSize;
        }
      }
      catch (Exception e)
      {}
      finally
      {
        cpuBufObj.unmap();
        bufferObject.unmap(cpuBufObj);
      }

    }
  }
}
