///////////////////////////////////////////////////////////////////////
//
// This example shows how to color a horizon with VolumeData values
// using a custom shader.
//
///////////////////////////////////////////////////////////////////////

package volumeviz.sample.horizonInVolume;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.io.File;

import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import com.openinventor.inventor.Axis;
import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbDataType;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec2f;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.SoSceneManager.AntialiasingModes;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.draggers.SoDragPointDragger;
import com.openinventor.inventor.draggers.SoJackDragger;
import com.openinventor.inventor.fields.SoMFFloat;
import com.openinventor.inventor.manips.SoJackManip;
import com.openinventor.inventor.nodes.SoBBox;
import com.openinventor.inventor.nodes.SoComplexity;
import com.openinventor.inventor.nodes.SoDrawStyle;
import com.openinventor.inventor.nodes.SoFile;
import com.openinventor.inventor.nodes.SoFragmentShader;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoGroup;
import com.openinventor.inventor.nodes.SoLightModel;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoPhysicalMaterial;
import com.openinventor.inventor.nodes.SoPickStyle;
import com.openinventor.inventor.nodes.SoRotation;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoShaderParameter1i;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.nodes.SoTransform;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;
import com.openinventor.inventor.viewercomponents.nodes.SceneOrbiter;
import com.openinventor.inventor.viewercomponents.nodes.SoViewingCube;
import com.openinventor.ldm.nodes.SoDataSet;
import com.openinventor.ldm.nodes.SoMultiDataSeparator;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.volumeviz.draggers.SoOrthoSliceDragger;
import com.openinventor.volumeviz.nodes.SoHeightFieldGeometry;
import com.openinventor.volumeviz.nodes.SoHeightFieldRender;
import com.openinventor.volumeviz.nodes.SoOrthoSlice;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;
import com.openinventor.volumeviz.nodes.SoVolumeShader;
import com.openinventor.volumeviz.nodes.SoVolumeSkin;

import util.Example;
import util.ViewerComponentsFactory;

/**
 * Horizon colored with VolumeData values
 *
 * Demonstrates how to color a horizon with VolumeData values using a custom
 * shader.
 */
public class Main extends Example
{
  private IRenderAreaInteractive m_viewer;
  SceneGraph m_scene;

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain("Horizon In Volume");
  }

  private class SceneGraph
  {
    private SoNode m_sceneGraph;
    private SoVolumeData m_volumeData;
    private SoHeightFieldGeometry m_heightFieldGeom;
    private SoOrthoSliceDragger m_orthoSliceDragger;
    private SoOrthoSlice m_orthoSlice;
    private SoMaterial m_skinMaterial;
    private SoMaterial m_volumeMaterial;
    private SoSwitch m_horizonSwitch;
    private SoRotation m_horizonAxisRotation;
    private SoTransform m_horizonTransform;
    private SoTransform m_horizonUserTransform;
    private SoBBox m_horizonBBox;
    private SoJackManip m_horizonManip;
    private SoSwitch m_sliceSwitch;
    private SoSwitch m_volumeRenderSwitch;
    private SoSwitch m_volumeSkinSwitch;
    private SoSwitch m_shaderSwitch;

    public SceneGraph(String volumeDataPath, String horizonGeomPath)
    {
      m_sceneGraph = buildSceneGraph();
      setVolumeData(volumeDataPath);
      setHorizonGeometry(horizonGeomPath);
      setSliceToCenter();
    }

    public SoNode getSceneGraph()
    {
      return m_sceneGraph;
    }

    public SoVolumeData getVolumeData()
    {
      return m_volumeData;
    }

    public SoTransform getHorizonUserTransform()
    {
      return m_horizonUserTransform;
    }

    public void setHorizonAxis(Axis axis)
    {
      SoDragPointDragger dragPtDragger =
          (SoDragPointDragger) ((SoJackDragger) m_horizonManip.getDragger()).getPart("translator", false);
      switch ( axis )
      {
      case X :
        m_horizonAxisRotation.rotation.setValue(new SbRotation(new SbVec3f(0.0f, 1.0f, 0.0f), 0.5f * (float) Math.PI));
        dragPtDragger.showDraggerSet(SoDragPointDragger.DraggerSets.X_AXIS);
        break;
      case Y :
        m_horizonAxisRotation.rotation.setValue(new SbRotation(new SbVec3f(-1.0f, 0.0f, 0.0f), 0.5f * (float) Math.PI));
        dragPtDragger.showDraggerSet(SoDragPointDragger.DraggerSets.Y_AXIS);
        break;
      default:
      case Z :
        m_horizonAxisRotation.rotation.setValue(new SbRotation());
        dragPtDragger.showDraggerSet(SoDragPointDragger.DraggerSets.Z_AXIS);
        break;
      }
    }

    public void setSliceToCenter()
    {
      m_orthoSlice.sliceNumber.setValue(m_volumeData.getDimension().getValue()[m_orthoSlice.axis.getValue()] / 2);
    }

    public SoOrthoSlice.AxisType getSliceAxis()
    {
      return m_orthoSlice.axis.getValue(SoOrthoSlice.AxisType.class);
    }

    public void setSliceAxis(SoOrthoSlice.AxisType sliceAxis)
    {
      m_orthoSlice.axis.setValue(sliceAxis);
      setSliceToCenter();
    }

    public SoOrthoSlice.ClippingSides getClippingSide()
    {
      return m_orthoSlice.clippingSide.getValue(SoOrthoSlice.ClippingSides.class);
    }

    public void setClippingSide(SoOrthoSlice.ClippingSides clippingSide)
    {
      m_orthoSlice.clippingSide.setValue(clippingSide);
    }

    public float getVolumeOpacity()
    {
      if ( isVolumeRenderDisplayed() )
        return getOpacity(m_volumeMaterial.transparency, 16.0f);
      else
        return getOpacity(m_skinMaterial.transparency, 1.0f);
    }

    public void setVolumeOpacity(float opacityValue)
    {
      setOpacity(m_skinMaterial.transparency, opacityValue, 1.0f);
      setOpacity(m_volumeMaterial.transparency, opacityValue, 16.0f);
    }

    public void setVolumeData(String dataSetPath)
    {
      m_volumeData.fileName.setValue((dataSetPath == null) ? "" : dataSetPath);
      adjustExtent(m_volumeData);
    }

    public void displayHorizon(boolean value)
    {
      displaySwitch(m_horizonSwitch, value);
    }

    public boolean isHorizonDisplayed()
    {
      return isSwitchDisplayed(m_horizonSwitch);
    }

    public void displaySlice(boolean value)
    {
      displaySwitch(m_sliceSwitch, value);
    }

    public boolean isSliceDisplayed()
    {
      return isSwitchDisplayed(m_sliceSwitch);
    }

    public void displayVolumeRender(boolean value)
    {
      displaySwitch(m_volumeRenderSwitch, value);
    }

    public boolean isVolumeRenderDisplayed()
    {
      return isSwitchDisplayed(m_volumeRenderSwitch);
    }

    public void displayVolumeSkin(boolean value)
    {
      displaySwitch(m_volumeSkinSwitch, value);
    }

    public void setHorizonGeometry(String horizonGeomPath)
    {
      m_heightFieldGeom.fileName.setValue((horizonGeomPath == null) ? "" : horizonGeomPath);
      adjustExtent(m_heightFieldGeom);
    }

    // Normalize biggest dimension to 1 in order to avoid
    // big size differences between Volume Data and Horizon
    public void adjustExtent(SoDataSet dataSet)
    {
      SbBox3f extent = dataSet.extent.getValue();

      boolean isHeightField = dataSet instanceof SoHeightFieldGeometry;
      if ( isHeightField )
      {
        double dataMin = 0.0;
        double dataMax = 1.0;
        if ( !new SbDataType(SbDataType.DataTypes.values()[dataSet.getDataType().getValue()]).isInteger() )
        {
          long minMax[] = dataSet.getMinMax();
          dataMin = minMax[0];
          dataMax = minMax[1];
        }

        SbVec3f newBBoxMin = m_horizonBBox.boundingBox.getValue().getMin();
        SbVec3f newBBoxMax = m_horizonBBox.boundingBox.getValue().getMax();
        newBBoxMin.setZ((float) dataMin);
        newBBoxMax.setZ((float) dataMax);
        m_horizonBBox.boundingBox.setValue(new SbBox3f(newBBoxMin, newBBoxMax));

        SbVec3f newExtentMin = extent.getMin();
        SbVec3f newExtentMax = extent.getMax();
        newExtentMin.setZ((float) dataMin);
        newExtentMax.setZ((float) dataMax);
        extent.setValue(new SbBox3f(newExtentMin, newExtentMax));
      }

      SbVec3f scale = extent.getMax().minus(extent.getMin());
      float maxScale = scale.getValue()[0];
      for ( int i = 1; i < 3; i++ )
      {
        if ( scale.getValue()[i] > maxScale )
          maxScale = scale.getValue()[i];
      }

      float factor = 2.0f / maxScale;
      scale.multiply(factor);
      SbVec3f newMin = scale.times(-0.5f);
      SbVec3f newMax = scale.plus(newMin);
      if ( isHeightField )
      {
        m_horizonTransform.translation
            .setValue(new SbVec3f(0.0f, 0.0f, -0.5f * factor * (extent.getMin().getZ() + extent.getMax().getZ())));
        m_horizonTransform.scaleFactor.setValue(new SbVec3f(1.0f, 1.0f, factor));
      }
      dataSet.extent.setValue(newMin, newMax);
    }

    private void displaySwitch(SoSwitch switchNode, boolean value)
    {
      switchNode.whichChild.setValue(value ? SoSwitch.SO_SWITCH_ALL : SoSwitch.SO_SWITCH_NONE);
    }

    private boolean isSwitchDisplayed(SoSwitch switchNode)
    {
      return (switchNode.whichChild.getValue() == SoSwitch.SO_SWITCH_ALL);
    }

    private float getOpacity(SoMFFloat transparencyField, float power)
    {
      return 1.0f - (float) Math.pow(transparencyField.getValueAt(0), power);
    }

    private void setOpacity(SoMFFloat transparencyField, float opacityValue, float power)
    {
      transparencyField.setValue((float) Math.pow(1.0 - opacityValue, 1.0 / power));
    }

    private SoNode buildSceneGraph()
    {
      SoMultiDataSeparator root = new SoMultiDataSeparator();

      SoGradientBackground gradientBackground = new SoGradientBackground();
      gradientBackground.color0.setValue(new SbColor(1.0f, 1.0f, 1.0f));
      gradientBackground.color1.connectFrom(gradientBackground.color0);
      root.addChild(gradientBackground);

      m_volumeData = new SoVolumeData();
      m_volumeData.dataSetId.setValue(1);

      SoTransferFunction transferFunction = new SoTransferFunction();
      transferFunction.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.STANDARD);
      root.addChild(transferFunction);

      // Horizon
      {
        m_horizonSwitch = new SoSwitch();
        m_horizonSwitch.setName("HEIGHT_FIELD_RENDER");
        m_horizonSwitch.whichChild.setValue(SoSwitch.WhichChild.SO_SWITCH_ALL.getValue());
        root.addChild(m_horizonSwitch);

        SoSeparator horizonSep = new SoSeparator();
        m_horizonSwitch.addChild(horizonSep);

        // Volume Data
        horizonSep.addChild(m_volumeData);

        // Manip
        {
          m_horizonManip = new SoJackManip();

          // Remove scaler and rotator parts
          SoJackDragger dragger = (SoJackDragger) m_horizonManip.getDragger();
          dragger.setPart("scaler", null);
          dragger.setPart("rotator", null);

          m_horizonManip.scaleFactor.setValue(new SbVec3f(0.5f, 0.5f, 0.5f));
          horizonSep.addChild(m_horizonManip);
        }

        m_horizonAxisRotation = new SoRotation();
        horizonSep.addChild(m_horizonAxisRotation);

        m_horizonUserTransform = new SoTransform();
        horizonSep.addChild(m_horizonUserTransform);

        // Horizon Transform
        m_horizonTransform = new SoTransform();
        horizonSep.addChild(m_horizonTransform);

        // Fake BBox to make the manip smaller
        {
          m_horizonBBox = new SoBBox();
          m_horizonBBox.mode.setValue(SoBBox.Modes.USER_DEFINED);
          m_horizonBBox.boundingBox.setValue(new SbBox3f(-0.5f, -0.5f, 0.0f, 0.5f, 0.5f, 1.0f));
          horizonSep.addChild(m_horizonBBox);
        }

        SoLightModel lightModel = new SoLightModel();
        lightModel.model.setValue(SoLightModel.Models.PHYSICALLY_BASED);
        horizonSep.addChild(lightModel);

        SoPhysicalMaterial horizonMaterial = new SoPhysicalMaterial();
        horizonMaterial.baseColor.setValue(1.0f, 1.0f, 1.0f, 1.0f);
        horizonMaterial.specular.setValue(1.0f);
        horizonMaterial.roughness.setValue(0.3f);
        horizonSep.addChild(horizonMaterial);

        m_heightFieldGeom = new SoHeightFieldGeometry();
        m_heightFieldGeom.dataSetId.setValue(2);
        horizonSep.addChild(m_heightFieldGeom);

        // Combine Shader
        {
          m_shaderSwitch = new SoSwitch();
          m_shaderSwitch.setName("COMBINE_SHADER");
          m_shaderSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
          horizonSep.addChild(m_shaderSwitch);

          String pkgName = this.getClass().getPackage().getName();
          pkgName = pkgName.replace('.', File.separatorChar);

          SoFragmentShader combineShader = new SoFragmentShader();
          combineShader.sourceProgram.setValue(SoPreferences.getValue("OIVJHOME") + File.separator + "examples"
              + File.separator + pkgName + File.separator + "combine_frag.glsl");

          SoShaderParameter1i volDataParam = new SoShaderParameter1i();
          volDataParam.name.setValue("volData");
          volDataParam.value.connectFrom(m_volumeData.dataSetId);
          combineShader.parameter.addShaderParameter(volDataParam);

          SoShaderParameter1i hfGeomParam = new SoShaderParameter1i();
          hfGeomParam.name.setValue("hfGeom");
          hfGeomParam.value.connectFrom(m_heightFieldGeom.dataSetId);
          combineShader.parameter.addShaderParameter(hfGeomParam);

          // Setup data combine shader in DATA_COMBINE_FUNCTION slot
          SoVolumeShader volumeShader = new SoVolumeShader();
          volumeShader.shaderObject.set1Value(SoVolumeShader.ShaderPositions.DATA_COMBINE_FUNCTION.getValue(),
              combineShader);
          m_shaderSwitch.addChild(volumeShader);
        }

        // Disable picking for performances
        {
          SoPickStyle pickStyle = new SoPickStyle();
          pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
          horizonSep.addChild(pickStyle);
        }

        // Finally, add the render node
        horizonSep.addChild(new SoHeightFieldRender());
      }

      // Volume Rendering
      {
        SoSeparator volumeRenderingSep = new SoSeparator();
        root.addChild(volumeRenderingSep);

        volumeRenderingSep.addChild(m_volumeData);

        SoVolumeSkin skin = new SoVolumeSkin();

        // Volume Box
        SoSwitch skinBoxSwitch = new SoSwitch();
        skinBoxSwitch.setName("VOLUME_BOX");
        skinBoxSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
        {
          SoSeparator boxSep = new SoSeparator();

          SoPickStyle boxPickStyle = new SoPickStyle();
          boxPickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
          boxSep.addChild(boxPickStyle);

          SoComplexity boxComplexity = new SoComplexity();
          boxComplexity.type.setValue(SoComplexity.Types.BOUNDING_BOX);
          boxSep.addChild(boxComplexity);

          SoLightModel lightModel = new SoLightModel();
          lightModel.model.setValue(SoLightModel.Models.BASE_COLOR);
          boxSep.addChild(lightModel);

          SoDrawStyle drawStyle = new SoDrawStyle();
          drawStyle.style.setValue(SoDrawStyle.Styles.LINES);
          boxSep.addChild(drawStyle);

          SoMaterial boxMaterial = new SoMaterial();
          boxMaterial.diffuseColor.setValue(new SbColor(0.0f, 0.0f, 0.0f));
          boxSep.addChild(boxMaterial);

          boxSep.addChild(skin);

          skinBoxSwitch.addChild(boxSep);
        }
        volumeRenderingSep.addChild(skinBoxSwitch);

        // Slices
        m_sliceSwitch = new SoSwitch();
        m_sliceSwitch.setName("ORTHO_SLICE");
        m_sliceSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
        {
          SoGroup sliceAndDraggerGroup = new SoGroup();

          SoGroup sliceGroup = new SoGroup();
          sliceAndDraggerGroup.addChild(sliceGroup);

          m_orthoSlice = new SoOrthoSlice();
          m_orthoSlice.axis.setValue(SoOrthoSlice.AxisType.Y);
          m_orthoSlice.clipping.setValue(true);
          m_orthoSlice.clippingSide.setValue(SoOrthoSlice.ClippingSides.FRONT);
          sliceGroup.addChild(m_orthoSlice);

          SoPath slicePath = new SoPath(sliceGroup);
          slicePath.regular.append(m_orthoSlice);

          m_orthoSliceDragger = new SoOrthoSliceDragger();
          m_orthoSliceDragger.orthoSlicePath.setValue(slicePath);
          sliceAndDraggerGroup.addChild(m_orthoSliceDragger);

          m_sliceSwitch.addChild(sliceAndDraggerGroup);
        }
        volumeRenderingSep.addChild(m_sliceSwitch);

        // Volume Skin
        m_volumeSkinSwitch = new SoSwitch();
        m_volumeSkinSwitch.setName("VOLUME_SKIN");
        m_volumeSkinSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
        {
          SoSeparator skinSep = new SoSeparator();

          SoPickStyle skinPickStyle = new SoPickStyle();
          skinPickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
          skinSep.addChild(skinPickStyle);

          m_skinMaterial = new SoMaterial();
          m_skinMaterial.transparency.setValue(0.6f);
          skinSep.addChild(m_skinMaterial);

          skin.faceMode.setValue(SoVolumeSkin.FaceModes.FRONT_AND_BACK);
          skinSep.addChild(skin);

          m_volumeSkinSwitch.addChild(skinSep);
        }
        volumeRenderingSep.addChild(m_volumeSkinSwitch);

        // Volume Render
        m_volumeRenderSwitch = new SoSwitch();
        m_volumeRenderSwitch.setName("VOLUME_RENDER");
        m_volumeRenderSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
        {
          SoSeparator volumeRenderSep = new SoSeparator();

          SoPickStyle volumeRenderPickStyle = new SoPickStyle();
          volumeRenderPickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
          volumeRenderSep.addChild(volumeRenderPickStyle);

          m_volumeMaterial = new SoMaterial();
          m_volumeMaterial.transparency.setValue(0.985f);
          volumeRenderSep.addChild(m_volumeMaterial);

          SoVolumeRenderingQuality vrq = new SoVolumeRenderingQuality();
          vrq.deferredLighting.setValue(false);
          volumeRenderSep.addChild(vrq);

          SoVolumeRender volumeRender = new SoVolumeRender();
          volumeRenderSep.addChild(volumeRender);

          m_volumeRenderSwitch.addChild(volumeRenderSep);
        }
        volumeRenderingSep.addChild(m_volumeRenderSwitch);
      }

      return root;
    }
  };

  @Override
  public void start()
  {
    m_viewer = ViewerComponentsFactory.createRenderAreaOrbiter();
    m_viewer.setTransparencyType(SoGLRenderAction.TransparencyTypes.SORTED_PIXEL);
    m_viewer.setAntialiasingMode(AntialiasingModes.SMAA);
    m_viewer.setAntialiasingQuality(1.0f);

    SceneOrbiter sceneOrbiter = (SceneOrbiter) m_viewer.getSceneInteractor();
    sceneOrbiter.getViewingCube().edgeStyle.setValue(SoViewingCube.EdgeStyles.CORNER);

    sceneOrbiter.getViewingCube().size.setValue(new SbVec2f(200.0f, 200.0f));
    sceneOrbiter.getViewingCube().opacityMin.setValue(0.0f);
    sceneOrbiter.getViewingCube().opacityMax.setValue(0.8f);
    SoFile compass = new SoFile();
    compass.name.setValue("$OIVJHOME/data/examples/ViewingCube/Compass/compass-northsouth-y.iv");
    sceneOrbiter.getViewingCube().compass.setValue(compass);
    sceneOrbiter.getViewingCube().facePosX.setValue("$OIVJHOME/data/examples/ViewingCube/Faces/NSEW/xpos.png");
    sceneOrbiter.getViewingCube().faceNegX.setValue("$OIVJHOME/data/examples/ViewingCube/Faces/NSEW/xneg.png");
    sceneOrbiter.getViewingCube().facePosY.setValue("$OIVJHOME/data/examples/ViewingCube/Faces/NSEW/ypos.png");
    sceneOrbiter.getViewingCube().faceNegY.setValue("$OIVJHOME/data/examples/ViewingCube/Faces/NSEW/yneg.png");
    sceneOrbiter.getViewingCube().facePosZ.setValue("$OIVJHOME/data/examples/ViewingCube/Faces/NSEW/zpos.png");
    sceneOrbiter.getViewingCube().faceNegZ.setValue("$OIVJHOME/data/examples/ViewingCube/Faces/NSEW/zneg.png");

    final String defaultVolumeData = "$OIVJHOME/data/volumeviz/colt-float.ldm";
    final String defaultHorizonGeom = "$OIVJHOME/data/volumeviz/horizon.ldm";
    m_scene = new SceneGraph(defaultVolumeData, defaultHorizonGeom);

    // Special parameters for the initial datasets
    m_scene.setHorizonAxis(Axis.Y);
    m_scene.getHorizonUserTransform().translation.setValue(new SbVec3f(0.0f, 0.0f, 0.3f));
    m_scene.getHorizonUserTransform().scaleFactor.setValue(new SbVec3f(1.0f, 1.0f, 0.5f));
    m_scene.setClippingSide(SoOrthoSlice.ClippingSides.BACK);
    m_scene.getVolumeData().extent
        .setValue(new SbBox3f(new SbVec3f(-1.0f, -1.0f, -1.0f), new SbVec3f(1.0f, 1.0f, 1.0f)));

    // Set up viewer:
    m_viewer.setSceneGraph(m_scene.getSceneGraph());

    // Set default camera orientation
    sceneOrbiter.getCamera().orientation.setValue(new SbRotation(new SbVec3f(-1.0f, 0.0f, 0.0f), 0.5f)
        .times(new SbRotation(new SbVec3f(0.0f, 1.0f, 0.0f), 0.4f)));

    m_viewer.viewAll(new SbViewportRegion());

    final Component component = m_viewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(1280, 768));
    setLayout(new BorderLayout());

    JPanel northPanel = new JPanel(new BorderLayout());
    northPanel.add(createUIPanel(), BorderLayout.CENTER);

    add(northPanel, BorderLayout.NORTH);
    add(component, BorderLayout.CENTER);
  }

  @Override
  public void stop()
  {
    m_viewer.dispose();
  }

  // SWING part
  // Add GUI allowing to interact with scene graph.
  private JPanel createUIPanel()
  {
    JPanel uiPanel = new JPanel();
    uiPanel.setLayout(new GridBagLayout());
    GridBagConstraints gridConstraints = new GridBagConstraints();
    gridConstraints.fill = GridBagConstraints.HORIZONTAL;
    gridConstraints.ipadx = 40;
    gridConstraints.gridx = 0;
    gridConstraints.gridy = 0;

    // Check Boxes
    {
      JPanel checkBoxesPanel = new JPanel();
      checkBoxesPanel.setLayout(new BoxLayout(checkBoxesPanel, BoxLayout.X_AXIS));

      JCheckBox horizonCheckBox = new JCheckBox("Horizon");
      horizonCheckBox.setSelected(m_scene.isHorizonDisplayed());
      horizonCheckBox.addChangeListener(new ChangeListener()
      {
        @Override
        public void stateChanged(ChangeEvent arg0)
        {
          m_scene.displayHorizon(horizonCheckBox.isSelected());
        }
      });
      checkBoxesPanel.add(horizonCheckBox);

      JCheckBox sliceCheckBox = new JCheckBox("Ortho Slice");
      sliceCheckBox.setSelected(m_scene.isSliceDisplayed());
      sliceCheckBox.addChangeListener(new ChangeListener()
      {
        @Override
        public void stateChanged(ChangeEvent arg0)
        {
          m_scene.displaySlice(sliceCheckBox.isSelected());
        }
      });
      checkBoxesPanel.add(sliceCheckBox);

      uiPanel.add(checkBoxesPanel, gridConstraints);
      gridConstraints.gridx++;
    }

    // Slice Axis
    {
      JPanel sliceAxisPanel = new JPanel();
      sliceAxisPanel.setLayout(new BoxLayout(sliceAxisPanel, BoxLayout.X_AXIS));

      sliceAxisPanel.add(new JLabel("Slice Axis"));

      ButtonGroup sliceAxisButtonGroup = new ButtonGroup();
      final SoOrthoSlice.AxisType currentAxis = m_scene.getSliceAxis();

      JRadioButton sliceAxisX = new JRadioButton("X");
      sliceAxisX.setSelected(currentAxis == SoOrthoSlice.AxisType.X);
      sliceAxisPanel.add(sliceAxisX);
      sliceAxisButtonGroup.add(sliceAxisX);

      JRadioButton sliceAxisY = new JRadioButton("Y");
      sliceAxisY.setSelected(currentAxis == SoOrthoSlice.AxisType.Y);
      sliceAxisPanel.add(sliceAxisY);
      sliceAxisButtonGroup.add(sliceAxisY);

      JRadioButton sliceAxisZ = new JRadioButton("Z");
      sliceAxisZ.setSelected(currentAxis == SoOrthoSlice.AxisType.Z);
      sliceAxisPanel.add(sliceAxisZ);
      sliceAxisButtonGroup.add(sliceAxisZ);

      ChangeListener sliceAxisListener = new ChangeListener()
      {
        @Override
        public void stateChanged(ChangeEvent arg0)
        {
          if ( sliceAxisX.isSelected() )
            m_scene.setSliceAxis(SoOrthoSlice.AxisType.X);
          else if ( sliceAxisY.isSelected() )
            m_scene.setSliceAxis(SoOrthoSlice.AxisType.Y);
          else
            m_scene.setSliceAxis(SoOrthoSlice.AxisType.Z);
        }
      };
      sliceAxisX.addChangeListener(sliceAxisListener);
      sliceAxisY.addChangeListener(sliceAxisListener);
      sliceAxisZ.addChangeListener(sliceAxisListener);

      uiPanel.add(sliceAxisPanel, gridConstraints);
      gridConstraints.gridx++;
    }

    // Clipping Side
    {
      JPanel clippingSidePanel = new JPanel();
      clippingSidePanel.setLayout(new BoxLayout(clippingSidePanel, BoxLayout.X_AXIS));

      clippingSidePanel.add(new JLabel("Clipping Side"));

      ButtonGroup clippingSideButtonGroup = new ButtonGroup();
      SoOrthoSlice.ClippingSides currentClippingSide = m_scene.getClippingSide();

      JRadioButton clippingSideFront = new JRadioButton("Front");
      clippingSideFront.setSelected(currentClippingSide == SoOrthoSlice.ClippingSides.FRONT);
      clippingSidePanel.add(clippingSideFront);
      clippingSideButtonGroup.add(clippingSideFront);

      JRadioButton clippingSideBack = new JRadioButton("Back");
      clippingSideBack.setSelected(currentClippingSide == SoOrthoSlice.ClippingSides.BACK);
      clippingSidePanel.add(clippingSideBack);
      clippingSideButtonGroup.add(clippingSideBack);

      ChangeListener clippingSideListener = new ChangeListener()
      {
        @Override
        public void stateChanged(ChangeEvent arg0)
        {
          if ( clippingSideFront.isSelected() )
            m_scene.setClippingSide(SoOrthoSlice.ClippingSides.FRONT);
          else
            m_scene.setClippingSide(SoOrthoSlice.ClippingSides.BACK);
        }
      };
      clippingSideFront.addChangeListener(clippingSideListener);
      clippingSideBack.addChangeListener(clippingSideListener);

      uiPanel.add(clippingSidePanel, gridConstraints);
      gridConstraints.gridx++;
    }

    // Volume Shape
    {
      JPanel volumeShapePanel = new JPanel();
      volumeShapePanel.setLayout(new BoxLayout(volumeShapePanel, BoxLayout.X_AXIS));

      ButtonGroup volumeShapeButtonGroup = new ButtonGroup();
      final boolean volumeRenderDisplayed = m_scene.isVolumeRenderDisplayed();

      JRadioButton volumeRenderButton = new JRadioButton("Volume Render");
      volumeRenderButton.setSelected(volumeRenderDisplayed);
      volumeShapePanel.add(volumeRenderButton);
      volumeShapeButtonGroup.add(volumeRenderButton);

      JRadioButton volumeSkinButton = new JRadioButton("Volume Skin");
      volumeSkinButton.setSelected(!volumeRenderDisplayed);
      volumeShapePanel.add(volumeSkinButton);
      volumeShapeButtonGroup.add(volumeSkinButton);

      ChangeListener volumeShapeListener = new ChangeListener()
      {
        @Override
        public void stateChanged(ChangeEvent arg0)
        {
          if ( volumeRenderButton.isSelected() )
          {
            m_scene.displayVolumeRender(true);
            m_scene.displayVolumeSkin(false);
          }
          else
          {
            m_scene.displayVolumeRender(false);
            m_scene.displayVolumeSkin(true);
          }
        }
      };
      volumeRenderButton.addChangeListener(volumeShapeListener);
      volumeSkinButton.addChangeListener(volumeShapeListener);

      gridConstraints.ipadx = 30;
      uiPanel.add(volumeShapePanel, gridConstraints);
      gridConstraints.gridx++;
    }

    // Volume Opacity
    {
      SliderPanel volumeOpacitySlider = new SliderPanel(0.0f, 1.0f, m_scene.getVolumeOpacity(), 2, false);
      volumeOpacitySlider.addInfoText("Volume Opacity");
      volumeOpacitySlider.setSliderSize(new Dimension(200, 20));
      volumeOpacitySlider.addSliderPanelListener(new SliderPanel.Listener()
      {
        @Override
        public void stateChanged(float volumeOpacity)
        {
          m_scene.setVolumeOpacity(volumeOpacity);
        }
      });
      uiPanel.add(volumeOpacitySlider, gridConstraints);
    }

    return uiPanel;
  }
}
