///////////////////////////////////////////////////////////////////////
//
// This example shows how to do a custom shift for each vertex of a
// horizon geometry using a customized shader slot.
//
///////////////////////////////////////////////////////////////////////

package volumeviz.sample.horizonShiftProjection;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.io.File;

import javax.swing.BoxLayout;
import javax.swing.JPanel;

import com.openinventor.inventor.*;
import com.openinventor.inventor.SoSceneManager.AntialiasingModes;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.details.SoDetail;
import com.openinventor.inventor.engines.SoDecomposeVec3f;
import com.openinventor.inventor.events.SoLocation2Event;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.sensors.SoFieldSensor;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;
import com.openinventor.inventor.viewercomponents.nodes.SceneOrbiter;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.volumeviz.details.SoHeightFieldDetail;
import com.openinventor.volumeviz.nodes.SoHeightFieldGeometry;
import com.openinventor.volumeviz.nodes.SoHeightFieldRender;
import com.openinventor.volumeviz.nodes.SoVolumeShader;

import util.Example;
import util.ViewerComponentsFactory;

/**
 * Horizon geometry projected using a custom shader
 *
 * Demonstrates how to do a custom shift for each vertex of a horizon geometry
 * using a customized shader slot.
 */
public class Main extends Example
{
  private SoText2 m_pickInfoText;
  private SoCustomHeightFieldGeometry m_heightFieldGeom;
  private SoSwitch m_sphereSwitch;
  private IRenderAreaInteractive myViewer;

  private SoFieldSensor m_heightScaleSensor;
  private SoFieldSensor m_spherePositionSensor;

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain("Horizon Shift Projection");
  }

  // Custom HeightFieldGeometry to overload voxelToXYZ and XYZToVoxel methods
  // We overload these methods to get correct bounding boxes, tile loading and
  // picking values when a shift is applied.
  private class SoCustomHeightFieldGeometry extends SoHeightFieldGeometry
  {
    public SoTranslation sphereTranslation;
    public SoShaderParameter1f heightScaleParam;

    @Override
    public RenderModes getRenderEngineMode()
    {
      return RenderModes.OIV_OPENINVENTOR_RENDERING;
    }

    // CPU implementation of the vertex shift
    // This should be the same shift as the one defined by the
    // VVizTessVertexShift() function in the shift shader
    private SbVec3f vertexShift(SbVec3f position)
    {
      SbVec3f spherePosition = sphereTranslation.translation.getValue();
      SbVec2f XYorientation = new SbVec2f(position.getX(), position.getY())
          .minus(new SbVec2f(spherePosition.getX(), spherePosition.getY()));
      float angle = XYorientation.normalize() / spherePosition.getZ();
      XYorientation.multiply((float) Math.sin(angle));
      SbVec3f projectedFloor = new SbVec3f(XYorientation.getX(), XYorientation.getY(), 1.0f - (float) Math.cos(angle))
          .times(spherePosition.getZ());
      SbVec3f projectedFloorDir = projectedFloor.minus(spherePosition);
      projectedFloorDir.normalize();
      return spherePosition.plus(projectedFloorDir
          .times(Math.abs(spherePosition.getZ()) + heightScaleParam.value.getValue() * position.getZ()));
    }

    // The inverse vertex shift
    // Given a shifted vertex, this method yields the original vertex position.
    private SbVec3f vertexShiftInverse(SbVec3f position)
    {
      SbVec3f spherePosition = sphereTranslation.translation.getValue();
      SbVec3f projectedFloorDir = position.minus(spherePosition);
      float sphereRadius = Math.abs(spherePosition.getZ());
      float height = (projectedFloorDir.normalize() - sphereRadius) / heightScaleParam.value.getValue();
      float XYLength = sphereRadius * (float) Math.acos(projectedFloorDir.getZ());
      SbVec2f XYDir = new SbVec2f(projectedFloorDir.getX(), projectedFloorDir.getY());
      XYDir.normalize();
      XYDir.multiply(XYLength);
      return new SbVec3f(XYDir.getX(), XYDir.getY(), height);
    }

    // Used for bbox and tile loading
    // We apply the shift after calling the parent method
    @Override
    public SbVec3f voxelToXYZ(SbVec3f dataPosition)
    {
      return vertexShift(super.voxelToXYZ(dataPosition));
    }

    // Used for bbox and tile loading
    // We apply voxelToXYZ() to all 8 corners of the input box
    // then we compute the global min and max values for all 3 dimensions
    // and we create the converted box using this min and max.
    // This is an approximation because the projection is a non-linear
    // transformation, but it is a good enough heuristic here.
    @Override
    public SbBox3f voxelToXYZ(SbBox3f box)
    {
      float xmin, ymin, zmin, xmax, ymax, zmax;
      float[] bounds = box.getBounds();
      xmin = bounds[0];
      ymin = bounds[1];
      zmin = bounds[2];
      xmax = bounds[3];
      ymax = bounds[4];
      zmax = bounds[5];

      SbVec3f[] corners = new SbVec3f[] { new SbVec3f(xmin, ymin, zmin), new SbVec3f(xmin, ymin, zmax),
          new SbVec3f(xmin, ymax, zmin), new SbVec3f(xmin, ymax, zmax), new SbVec3f(xmax, ymin, zmin),
          new SbVec3f(xmax, ymin, zmax), new SbVec3f(xmax, ymax, zmin), new SbVec3f(xmax, ymax, zmax) };

      for ( int i = 0; i < corners.length; i++ )
      {
        corners[i] = voxelToXYZ(corners[i]);

        if ( i == 0 )
        {
          xmin = xmax = corners[i].getX();
          ymin = ymax = corners[i].getY();
          zmin = zmax = corners[i].getZ();
        }
        else
        {
          xmin = Math.min(xmin, corners[i].getX());
          ymin = Math.min(ymin, corners[i].getY());
          zmin = Math.min(zmin, corners[i].getZ());
          xmax = Math.max(xmax, corners[i].getX());
          ymax = Math.max(ymax, corners[i].getY());
          zmax = Math.max(zmax, corners[i].getZ());
        }
      }

      return new SbBox3f(xmin, ymin, zmin, xmax, ymax, zmax);
    }

    // Used during heightfield picking to fill IJK detail
    // We apply the inverse shift before calling the parent method
    @Override
    public SbVec3f XYZToVoxel(SbVec3f dataPosition)
    {
      return super.XYZToVoxel(vertexShiftInverse(dataPosition));
    }

    // Used for bbox and tile loading
    // We apply XYZToVoxel() to all 8 corners of the input box
    // then we compute the global min and max values for all 3 dimensions
    // and we create the converted box using this min and max.
    // This is an approximation because the projection is a non-linear
    // transformation, but it is a good enough heuristic here.
    @Override
    public SbBox3f XYZToVoxel(SbBox3f box)
    {
      float xmin, ymin, zmin, xmax, ymax, zmax;
      float[] bounds = box.getBounds();
      xmin = bounds[0];
      ymin = bounds[1];
      zmin = bounds[2];
      xmax = bounds[3];
      ymax = bounds[4];
      zmax = bounds[5];

      SbVec3f[] corners = new SbVec3f[] { new SbVec3f(xmin, ymin, zmin), new SbVec3f(xmin, ymin, zmax),
          new SbVec3f(xmin, ymax, zmin), new SbVec3f(xmin, ymax, zmax), new SbVec3f(xmax, ymin, zmin),
          new SbVec3f(xmax, ymin, zmax), new SbVec3f(xmax, ymax, zmin), new SbVec3f(xmax, ymax, zmax), };

      for ( int i = 0; i < corners.length; i++ )
      {
        corners[i] = XYZToVoxel(corners[i]);

        if ( i == 0 )
        {
          xmin = xmax = corners[i].getX();
          ymin = ymax = corners[i].getY();
          zmin = zmax = corners[i].getZ();
        }
        else
        {
          xmin = Math.min(xmin, corners[i].getX());
          ymin = Math.min(ymin, corners[i].getY());
          zmin = Math.min(zmin, corners[i].getZ());
          xmax = Math.max(xmax, corners[i].getX());
          ymax = Math.max(ymax, corners[i].getY());
          zmax = Math.max(zmax, corners[i].getZ());
        }
      }

      return new SbBox3f(xmin, ymin, zmin, xmax, ymax, zmax);
    }
  }

  // Callback used to update picking info
  private class MouseMoveEventHandler extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoLocation2Event evt = (SoLocation2Event) node.getEvent();
      if ( evt.getEventSource() == SoLocation2Event.EventSources.MOUSE_LEAVE )
        return;

      SoHeightFieldDetail hfDetail = null;
      SoPickedPoint pickedPoint = node.getPickedPoint();
      if ( pickedPoint != null )
      {
        SoDetail detail = pickedPoint.getDetail();
        if ( detail instanceof SoHeightFieldDetail )
          hfDetail = (SoHeightFieldDetail) detail;
      }

      if ( hfDetail != null )
      {
        SbVec3i32 ijk = hfDetail.getIjkPos();
        m_pickInfoText.string.set1Value(1, "IJK: (" + ijk.getX() + ", " + ijk.getY() + ", " + ijk.getZ() + ")");
        m_pickInfoText.string.set1Value(2, "Height: " + hfDetail.getHeight());
      }
      else
      {
        m_pickInfoText.string.set1Value(1, "IJK: ");
        m_pickInfoText.string.set1Value(2, "Height: ");
      }
    }
  }

  private SoNode buildScenegraph()
  {
    String pkgName = this.getClass().getPackage().getName();
    pkgName = pkgName.replace('.', File.separatorChar);

    SoSeparator root = new SoSeparator();

    // Background
    SoGradientBackground background = new SoGradientBackground();
    float greyValue = 0.96f;
    background.color0.setValue(new SbColor(greyValue, greyValue, greyValue));
    background.color1.connectFrom(background.color0);
    root.addChild(background);

    // Pick Info
    {
      SoSeparator pickInfoSep = new SoSeparator();
      root.addChild(pickInfoSep);

      // Callback to update pick info
      SoEventCallback eventCBNode = new SoEventCallback();
      pickInfoSep.addChild(eventCBNode);

      SoPickStyle pickInfoPickStyle = new SoPickStyle();
      pickInfoPickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
      pickInfoSep.addChild(pickInfoPickStyle);

      SoOrthographicCamera pickInfoCam = new SoOrthographicCamera();
      pickInfoCam.viewportMapping.setValue(SoCamera.ViewportMappings.LEAVE_ALONE);
      pickInfoCam.position.setValue(new SbVec3f(0.005f, 0.005f, 1.0f));
      pickInfoCam.height.setValue(0.01f);
      pickInfoSep.addChild(pickInfoCam);

      SoMaterial pickInfoMaterial = new SoMaterial();
      pickInfoMaterial.ambientColor.setValue(new SbColor(0.0f, 0.0f, 0.0f));
      pickInfoMaterial.diffuseColor.setValue(new SbColor(0.0f, 0.0f, 0.0f));
      pickInfoSep.addChild(pickInfoMaterial);

      SoTranslation pickInfoTranslation = new SoTranslation();
      pickInfoTranslation.translation.setValue(new SbVec3f(0.0002f, 0.0094f, 0.0f));
      pickInfoSep.addChild(pickInfoTranslation);

      SoFontStyle pickInfoFontStyle = new SoFontStyle();
      pickInfoFontStyle.size.setValue(20.0f);
      pickInfoFontStyle.family.setValue(SoFontStyle.Families.SANS);
      pickInfoSep.addChild(pickInfoFontStyle);

      m_pickInfoText = new SoText2();
      m_pickInfoText.string.setValues(0, new String[] { "Pick info:", "IJK: ", "Height: " });
      pickInfoSep.addChild(m_pickInfoText);

      eventCBNode.addEventCallback(SoLocation2Event.class, new MouseMoveEventHandler());
    }

    // Horizon geometry data node
    // We use a custom derived SoHeightFieldGeometry
    // with some overloaded methods (see SoCustomHeightFieldGeometry class
    // definition)
    m_heightFieldGeom = new SoCustomHeightFieldGeometry();

    // A node to represent the projection sphere translation
    // Its radius is defined as the absolute value of the Z component
    m_heightFieldGeom.sphereTranslation = new SoTranslation();
    m_heightFieldGeom.sphereTranslation.translation.setValue(new SbVec3f(0.0f, 0.0f, -1.0f));

    // a semi-transparent sphere to visualize the projection
    {
      m_sphereSwitch = new SoSwitch();
      m_sphereSwitch.whichChild.setValue(SoSwitch.WhichChild.SO_SWITCH_ALL.getValue());
      root.addChild(m_sphereSwitch);

      SoSeparator sphereSep = new SoSeparator();
      m_sphereSwitch.addChild(sphereSep);

      SoTranslation sphereTranslation = new SoTranslation();
      sphereTranslation.translation.connectFrom(m_heightFieldGeom.sphereTranslation.translation);
      sphereSep.addChild(sphereTranslation);

      SoDecomposeVec3f decomposeTranslation = new SoDecomposeVec3f();
      decomposeTranslation.vector.connectFrom(m_heightFieldGeom.sphereTranslation.translation);

      SoPickStyle spherePickStyle = new SoPickStyle();
      spherePickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE.getValue());
      sphereSep.addChild(spherePickStyle);

      SoMaterial sphereMaterial = new SoMaterial();
      sphereMaterial.transparency.setValue(0.9f);
      sphereSep.addChild(sphereMaterial);

      SoAlgebraicSphere sphere = new SoAlgebraicSphere();
      sphere.radius.connectFrom(decomposeTranslation.z);
      sphereSep.addChild(sphere);
    }

    // VolumeViz nodes used to render the horizon
    {
      SoSeparator horizonNodes = new SoSeparator();
      root.addChild(horizonNodes);

      m_heightFieldGeom.fileName.setValue(SoPreferences.getValue("OIVJHOME") + File.separator + "data" + File.separator
          + "volumeviz" + File.separator + "horizon.ldm");
      horizonNodes.addChild(m_heightFieldGeom);

      // A transfer function to add some color
      {
        SoTransferFunction transferFunction = new SoTransferFunction();
        transferFunction.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.STANDARD.getValue());
        horizonNodes.addChild(transferFunction);
      }

      // Horizon Z scale parameter
      {
        m_heightFieldGeom.heightScaleParam = new SoShaderParameter1f();
        m_heightFieldGeom.heightScaleParam.name.setValue("heightScale");
        m_heightFieldGeom.heightScaleParam.value.setValue(0.25f);
      }

      // Projection sphere position parameter
      SoShaderParameter3f spherePositionParam = new SoShaderParameter3f();
      {
        spherePositionParam.name.setValue("spherePosition");
        spherePositionParam.value.connectFrom(m_heightFieldGeom.sphereTranslation.translation);
      }

      // When the shift parameters change, the bounding boxes of the tiles may
      // need to be updated
      // but the VolumeData node has no way of knowing when to do the update, so
      // we need a trick to force the re-computation of those bounding boxes.
      // We use sensors to know when the shift parameters change.
      {
        Runnable projectionChangedCB = new Runnable()
        {
          // This callback is called when the shift parameters change and
          // performs a trick to force the update of the tiles bounding boxes.
          @Override
          public void run()
          {
            SbBox3f extentValue = m_heightFieldGeom.extent.getValue();
            SbVec3f extentMax = extentValue.getMax();

            // Modifying the extent of a SoDataSet triggers a re-computation of
            // the tiles bounding boxes. The Z component of the extent is unused
            // for HeightFields so we change it to trigger the re-computation
            // without affecting the rendering (we set the Z max value to 0 or 1
            // depending on the previous value)
            extentMax.setZ((extentMax.getZ() > 0.0f) ? 0.0f : 1.0f);

            m_heightFieldGeom.extent.setValue(extentValue.getMin(), extentMax);
          }
        };

        m_heightScaleSensor = new SoFieldSensor(projectionChangedCB);
        m_heightScaleSensor.attach(m_heightFieldGeom.heightScaleParam.value);

        m_spherePositionSensor = new SoFieldSensor(projectionChangedCB);
        m_spherePositionSensor.attach(spherePositionParam.value);
      }

      // Shift shader
      SoTessellationEvaluationShader tessShader = new SoTessellationEvaluationShader();
      tessShader.sourceProgram.setValue(SoPreferences.getValue("OIVJHOME") + File.separator + "examples"
          + File.separator + pkgName + File.separator + "shiftProjection_tess.glsl");

      // Send projection parameters to the shift shader
      tessShader.parameter.set1Value(0, m_heightFieldGeom.heightScaleParam);
      tessShader.parameter.set1Value(1, spherePositionParam);

      // Setup shift shader in TESS_VERTEX_SHIFT slot
      SoVolumeShader volumeShader = new SoVolumeShader();
      volumeShader.shaderObject.set1Value(SoVolumeShader.ShaderPositions.TESS_VERTEX_SHIFT.getValue(), tessShader);
      horizonNodes.addChild(volumeShader);

      // Finally, add the render node
      horizonNodes.addChild(new SoHeightFieldRender());
    }

    return root;
  }

  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createRenderAreaOrbiter();
    myViewer.setTransparencyType(SoGLRenderAction.TransparencyTypes.SORTED_PIXEL);
    myViewer.setAntialiasingMode(AntialiasingModes.AUTO);
    myViewer.setAntialiasingQuality(1);

    SceneOrbiter sceneOrbiter = (SceneOrbiter) myViewer.getSceneInteractor();
    sceneOrbiter.setUpAxis(Axis.Z);
    sceneOrbiter.setRotationMethod(SceneOrbiter.RotationMethods.TRACKBALL);

    // Set up viewer:
    myViewer.setSceneGraph(buildScenegraph());
    myViewer.viewAll(new SbViewportRegion());

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(1024, 768));
    setLayout(new BorderLayout());
    add(createSliderPanel(), BorderLayout.NORTH);
    add(component);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  // SWING part
  // Add GUI allowing to interact with uniform variables.
  private JPanel createSliderPanel()
  {
    SliderPanel curvatureSlider = new SliderPanel(0.0f, 1.0f, 0.5f, 2);
    curvatureSlider.addInfoText("Projection Curvature: ");
    curvatureSlider.setSliderSize(new Dimension(200, 20));
    curvatureSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float curvature)
      {
        float radius;
        if ( curvature == 0.0 )
        {
          // prevent division by zero and sphere rendering
          radius = 1000.0f;
          m_sphereSwitch.whichChild.setValue(SoSwitch.WhichChild.SO_SWITCH_NONE.getValue());
        }
        else
        {
          radius = 0.5f / curvature;
          m_sphereSwitch.whichChild.setValue(SoSwitch.WhichChild.SO_SWITCH_ALL.getValue());
        }
        m_heightFieldGeom.sphereTranslation.translation.setValue(new SbVec3f(0.0f, 0.0f, -radius));
      }
    });

    SliderPanel heightScaleSlider = new SliderPanel(0.0f, 1.0f, 0.2f, 2);
    heightScaleSlider.addInfoText("Height Scale: ");
    heightScaleSlider.setSliderSize(new Dimension(200, 20));
    heightScaleSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float heightScale)
      {
        m_heightFieldGeom.heightScaleParam.value.setValue(heightScale);
      }
    });

    JPanel sliderPanel = new JPanel();
    sliderPanel.setLayout(new BoxLayout(sliderPanel, BoxLayout.X_AXIS));
    sliderPanel.add(curvatureSlider);
    sliderPanel.add(heightScaleSlider);

    return sliderPanel;
  }
}
