package volumeviz.sample.physicalRendering;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.File;
import java.io.IOException;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;

import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec2i32;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.SoOffscreenRenderArea;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.fields.SoSFEnum;
import com.openinventor.inventor.fields.SoSFFloat;
import com.openinventor.inventor.helpers.SbFileHelper;
import com.openinventor.inventor.nodes.SoCamera;
import com.openinventor.inventor.nodes.SoEnvironmentMap;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoPhysicalMaterial;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoShadowGroup;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.sensors.SoFieldSensor;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;

import util.Example;
import util.ViewerComponentsFactory;

/**
 * This demo shows off physical rendering effects applied to volume rendering.
 *
 * Many parameters can be adjusted, including shadow opacity and quality,
 * material roughness, environment map texture and intensity, tone mapping and
 * depth of field (blur factor and focal distance).
 *
 * The demo also allows to take snapshots of up to 10 times the original
 * resolution.
 */
public class Main extends Example
{
  SoSwitch m_lightModelSwitch;
  SoMaterial m_lightingMaterial;
  SoPhysicalMaterial m_lightingPMaterial;
  SoEnvironmentMap m_envMap;
  SoShadowGroup m_shadowGroup;
  private IViewerExaminer m_viewer;
  private SoFieldSensor m_cameraFocalDistanceSensor;

  enum ShadingStyles
  {
    PHONG, PHYSICALLY_BASED,
  };

  enum EnvironmentMaps
  {
    SKY, GROVE, STPETER, GRACE, UFFIZI, STLAZARUS, NO_ENVIRONMENT,
  };

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain("Physical Rendering");
  }

  @Override
  public void start()
  {
    SoPreferences.setValue("IVVR_ALPHA_THRESHOLD_INTERACTIVE", "0.99");

    m_viewer = ViewerComponentsFactory.createViewerExaminer();

    String pkgName = this.getClass().getPackage().getName();
    pkgName = pkgName.replace('.', File.separatorChar);
    String prefix =
        SoPreferences.getValue("OIVJHOME") + File.separator + "examples" + File.separator + pkgName + File.separator;
    SoSeparator root = readSceneGraph(prefix + "scene_physicalRendering.iv");

    m_lightModelSwitch = (SoSwitch) SoNode.getByName("LIGHT_MODEL_SWITCH");
    m_lightingMaterial = (SoMaterial) SoNode.getByName("MATERIAL");
    m_lightingPMaterial = (SoPhysicalMaterial) SoNode.getByName("PHYSICAL_MATERIAL");
    m_envMap = (SoEnvironmentMap) SoNode.getByName("ENVIRONMENT_MAP");
    m_shadowGroup = (SoShadowGroup) SoNode.getByName("SHADOW_GROUP");

    m_viewer.getRenderArea().setTransparencyType(SoGLRenderAction.TransparencyTypes.NO_SORT);
    m_viewer.setSceneGraph(root);

    SoCamera cam = m_viewer.getRenderArea().getSceneInteractor().getCamera();
    cam.position.setValue(new SbVec3f(340.0f, 180.0f, 130.0f));
    cam.orientation.setValue(new SbRotation(new SbVec3f(-0.36f, -0.53f, -0.77f), 3.92f));
    cam.focalDistance.setValue(400.0f);
    cam.exposureMode.setValue(SoCamera.ExposureModes.HDR_NEUTRAL);
    cam.blur.setValue(0.25f);

    initGUI();
  }

  @Override
  public void stop()
  {
    m_viewer.dispose();
  }

  private static SoSeparator readSceneGraph(String fileName)
  {
    SoInput input = new SoInput();

    if ( !input.openFile(fileName) )
    {
      System.err.println("Cannot open file " + fileName);
      return null;
    }

    SoSeparator node = SoDB.readAll(input);
    if ( node == null )
    {
      System.err.println("Problem reading file");
      input.closeFile();
      return null;
    }

    input.closeFile();
    return node;
  }

  private void initGUI()
  {
    // ====================================
    // shadow panel
    JPanel shadow_panel = new JPanel();
    shadow_panel.setLayout(new BoxLayout(shadow_panel, BoxLayout.PAGE_AXIS));
    shadow_panel.setBorder(new TitledBorder(new EtchedBorder(), "Shadows"));

    final JCheckBox enable_shadows_box = new JCheckBox("Enable");
    enable_shadows_box.setSelected(true);

    final SliderPanel shadow_opacity_slider = new SliderPanel(0.f, 1.f, 0.8f, 3, false);
    shadow_opacity_slider.setSliderSize(new Dimension(300, 20));
    shadow_opacity_slider.addInfoText("opacity");
    shadow_opacity_slider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_shadowGroup.intensity.setValue(value);
      }
    });

    final SliderPanel shadow_quality_slider = new SliderPanel(0.f, 1.f, 0.5f, 3, false);
    shadow_quality_slider.setSliderSize(new Dimension(300, 20));
    shadow_quality_slider.addInfoText("quality");
    shadow_quality_slider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_shadowGroup.quality.setValue(value);
      }
    });

    enable_shadows_box.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        boolean selected = (e.getStateChange() == ItemEvent.SELECTED);
        m_shadowGroup.isActive.setValue(selected);
        shadow_opacity_slider.setEnabled(selected);
        shadow_quality_slider.setEnabled(selected);
      }
    });

    shadow_panel.add(enable_shadows_box);
    shadow_panel.add(shadow_opacity_slider);
    shadow_panel.add(shadow_quality_slider);

    // ====================================
    // lighting panel
    JPanel lighting_panel = new JPanel();
    lighting_panel.setLayout(new BoxLayout(lighting_panel, BoxLayout.PAGE_AXIS));
    lighting_panel.setBorder(new TitledBorder(new EtchedBorder(), "Physically-based Lighting"));

    // shading style combobox
    JPanel shadingStyle_panel = new JPanel();
    JLabel shadingStyle_label = new JLabel("Shading Style");
    final JComboBox<ShadingStyles> shadingStyle_box = new JComboBox<>(ShadingStyles.values());
    shadingStyle_box.setSelectedItem(ShadingStyles.PHYSICALLY_BASED);
    shadingStyle_box.setPreferredSize(new Dimension(145, 20));
    shadingStyle_panel.add(shadingStyle_label);
    shadingStyle_panel.add(shadingStyle_box);

    // roughness
    final SliderPanel roughness_slider = new SliderPanel(0.f, 1.f, 0.3f, 3, false);
    roughness_slider.setSliderSize(new Dimension(300, 20));
    roughness_slider.addInfoText("Roughness");
    roughness_slider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        if ( m_lightModelSwitch.whichChild.getValue() == 1 )
          m_lightingPMaterial.roughness.setValue(value);
        else
          m_lightingMaterial.shininess.setValue(0.05f * (-1.0f + 1.0f / Math.max(value, 0.001f)));
      }
    });

    // environment map combobox
    JPanel envmap_panel = new JPanel();
    JLabel envmap_label = new JLabel("Environment map");
    final JComboBox<EnvironmentMaps> envmap_box = new JComboBox<>(EnvironmentMaps.values());
    envmap_box.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent arg0)
      {
        String[][] filenames = new String[][] { new String[] { "$OIVJHOME/data/textures/cubemaps/sky/skyHDR.hdr" },
            new String[] { "$OIVJHOME/data/textures/cubemaps/rnl/rnl_cross.hdr" },
            new String[] { "$OIVJHOME/data/textures/cubemaps/stpeters/stpeters_cross.hdr" },
            new String[] { "$OIVJHOME/data/textures/cubemaps/grace/grace_cross.hdr" },
            new String[] { "$OIVJHOME/data/textures/cubemaps/uffizi/uffizi.hdr" },
            new String[] { "$OIVJHOME/data/textures/cubemaps/tenerife/posx.png",
                "$OIVJHOME/data/textures/cubemaps/tenerife/posy.png",
                "$OIVJHOME/data/textures/cubemaps/tenerife/posz.png",
                "$OIVJHOME/data/textures/cubemaps/tenerife/negx.png",
                "$OIVJHOME/data/textures/cubemaps/tenerife/negy.png",
                "$OIVJHOME/data/textures/cubemaps/tenerife/negz.png" },
            new String[] { "" }, };
        m_envMap.filenames.setValues(0, filenames[envmap_box.getSelectedIndex()]);
      }
    });
    envmap_box.setPreferredSize(new Dimension(145, 20));
    envmap_box.setSelectedItem(EnvironmentMaps.UFFIZI);
    envmap_panel.add(envmap_label);
    envmap_panel.add(envmap_box);

    // environment map intensity
    final SliderPanel envmap_intensity_slider = new SliderPanel(0.f, 2.f, 0.2f, 3, false);
    envmap_intensity_slider.setSliderSize(new Dimension(300, 20));
    envmap_intensity_slider.addInfoText("Environment Map Intensity");
    envmap_intensity_slider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_envMap.intensity.setValue(value);
      }
    });

    // tone mapping
    final JCheckBox enable_tonemapping_box = new JCheckBox("Tone Mapping");
    enable_tonemapping_box.setSelected(true);
    SoSFEnum<SoCamera.ExposureModes> cameraExposureMode =
        m_viewer.getRenderArea().getSceneInteractor().getCamera().exposureMode;
    enable_tonemapping_box.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        boolean selected = (e.getStateChange() == ItemEvent.SELECTED);
        cameraExposureMode.setValue(selected ? SoCamera.ExposureModes.HDR_NEUTRAL : SoCamera.ExposureModes.LDR_LINEAR);
      }
    });

    shadingStyle_box.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent arg0)
      {
        ShadingStyles newStyle = (ShadingStyles) shadingStyle_box.getSelectedItem();
        boolean phongShading = newStyle == ShadingStyles.PHONG;
        boolean enableButtons = !phongShading;
        envmap_box.setEnabled(enableButtons);
        enable_tonemapping_box.setEnabled(enableButtons);
        m_lightModelSwitch.whichChild.setValue(shadingStyle_box.getSelectedIndex());
      }
    });

    lighting_panel.add(shadingStyle_panel);
    lighting_panel.add(roughness_slider);
    lighting_panel.add(envmap_panel);
    lighting_panel.add(envmap_intensity_slider);
    lighting_panel.add(enable_tonemapping_box);

    // ====================================
    // depth of field panel
    JPanel dof_panel = new JPanel();
    dof_panel.setLayout(new BoxLayout(dof_panel, BoxLayout.PAGE_AXIS));
    dof_panel.setBorder(new TitledBorder(new EtchedBorder(), "Depth Of Field"));

    // blur factor slider
    SoSFFloat cameraBlur = m_viewer.getRenderArea().getSceneInteractor().getCamera().blur;
    final SliderPanel blurFactor_slider = new SliderPanel(0.f, 2.0f, 0.25f, 3, false);
    blurFactor_slider.setSliderSize(new Dimension(300, 20));
    blurFactor_slider.addInfoText("Blur Factor");
    blurFactor_slider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        cameraBlur.setValue(value);
      }
    });

    // focal distance slider
    SoSFFloat cameraFocalDistance = m_viewer.getRenderArea().getSceneInteractor().getCamera().focalDistance;
    float focalDistance = cameraFocalDistance.getValue();
    m_cameraFocalDistanceSensor = new SoFieldSensor();
    final SliderPanel focalDistance_slider = new SliderPanel(0.f, 2.0f * focalDistance, focalDistance, 3, false);
    focalDistance_slider.setSliderSize(new Dimension(300, 20));
    focalDistance_slider.addInfoText("Focal Distance");
    focalDistance_slider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        // Don't trigger sensor unless necessary
        m_cameraFocalDistanceSensor.detach();
        if ( cameraFocalDistance.getValue() != value )
          cameraFocalDistance.setValue(value);
        m_cameraFocalDistanceSensor.attach(cameraFocalDistance);
      }
    });

    // Schedule a sensor to update the slider when camera focal distance
    // changes.
    m_cameraFocalDistanceSensor.setTask(new Runnable()
    {
      @Override
      public void run()
      {
        // Update UI with actual focal distance
        float focalDistance = cameraFocalDistance.getValue();

        // Don't trigger an event unless necessary
        if ( focalDistance_slider.getSliderValue() != focalDistance )
        {
          focalDistance_slider.setSliderMax(2.0f * focalDistance);
          focalDistance_slider.setSliderValue(focalDistance);
        }
      }
    });
    m_cameraFocalDistanceSensor.attach(cameraFocalDistance);

    dof_panel.add(blurFactor_slider);
    dof_panel.add(focalDistance_slider);

    // ====================================
    // snapshot panel
    JPanel snapshot_panel = new JPanel();
    snapshot_panel.setLayout(new BoxLayout(snapshot_panel, BoxLayout.PAGE_AXIS));
    snapshot_panel.setBorder(new TitledBorder(new EtchedBorder(), "Snapshot"));

    final SliderPanel scale_slider = new SliderPanel(1.f, 10.f, 2.f, 3, false);
    scale_slider.setSliderSize(new Dimension(300, 20));
    scale_slider.addInfoText("Scale");

    JPanel snapshot_filename_panel = new JPanel();
    snapshot_filename_panel.setLayout(new FlowLayout());

    final JTextField snapshot_filename =
        new JTextField("$OIVJHOME/examples/volumeviz/sample/physicalRendering/snapshot.png");
    final JButton snapshot_filechooser = new JButton("...");
    snapshot_filechooser.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent arg0)
      {
        JFileChooser fileChooser = new JFileChooser();
        int option = fileChooser.showSaveDialog(snapshot_filechooser);

        if ( option == JFileChooser.APPROVE_OPTION )
        {
          try
          {
            snapshot_filename.setText(fileChooser.getSelectedFile().getCanonicalPath());
          }
          catch (IOException e)
          {
            e.printStackTrace();
          }
        }

      }
    });

    snapshot_filename_panel.add(snapshot_filename);
    snapshot_filename_panel.add(snapshot_filechooser);

    JButton snaphot_button = new JButton("Snapshot");
    snaphot_button.setAlignmentX(CENTER_ALIGNMENT);
    snaphot_button.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent arg0)
      {
        String filename = SbFileHelper.expandString(snapshot_filename.getText());
        float scale = scale_slider.getSliderValue();
        // Make image same size and shape as viewer image
        SbViewportRegion vpRegion = new SbViewportRegion(m_viewer.getRenderArea().getComponent().getSize());
        vpRegion.setWindowSize((short) (scale * vpRegion.getWindowSizei32().getX()),
            (short) (scale * vpRegion.getWindowSizei32().getY()));

        SoOffscreenRenderArea renderer = new SoOffscreenRenderArea();
        renderer.setViewportRegion(vpRegion);

        // Transfer viewer settings to renderer
        renderer.setSceneGraph(m_viewer.getRenderArea().getSceneInteractor());
        renderer.setTransparencyType(m_viewer.getRenderArea().getTransparencyType());
        renderer.setTile(new SbVec2i32(1024, 1024), 5);
        renderer.renderToFile(filename);
      }
    });

    snapshot_panel.add(scale_slider);
    snapshot_panel.add(snapshot_filename_panel);
    snapshot_panel.add(snaphot_button);

    // ====================================
    // main panel
    JPanel main_panel = new JPanel();
    main_panel.setLayout(new BoxLayout(main_panel, BoxLayout.PAGE_AXIS));
    main_panel.setBorder(BorderFactory.createRaisedBevelBorder());
    main_panel.add(shadow_panel);
    main_panel.add(lighting_panel);
    main_panel.add(dof_panel);
    main_panel.add(snapshot_panel);

    // ScrollPane
    JScrollPane scroll_pane = new JScrollPane(main_panel);

    // SplitPane
    JPanel iPanel = new JPanel(new BorderLayout());
    final Component component = m_viewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(500, 700));
    iPanel.add(component, BorderLayout.CENTER);
    iPanel.setMinimumSize(new Dimension(1, 1));

    JSplitPane split_pane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, true, iPanel, scroll_pane);
    split_pane.setOneTouchExpandable(true);
    split_pane.setResizeWeight(0.6);

    setLayout(new BorderLayout());
    add(split_pane);
  }

}
