package volumeviz.sample.segmentedInterpolation;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Panel;
import java.awt.event.ActionEvent;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Vector;

import javax.swing.JComboBox;
import javax.swing.JLabel;

import com.openinventor.inventor.Inventor;
import com.openinventor.inventor.SbDataType;
import com.openinventor.inventor.SbVec3d;
import com.openinventor.inventor.SbVec3i32;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.actions.SoSearchAction;
import com.openinventor.inventor.fields.SoSFArray;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.ldm.SoLDMGlobalResourceParameters;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeIsosurface;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  /**
   * This demonstrate how to use the segmentedInterpolation and
   * colorInterpolation fields in order to avoid doing interpolation on
   * segmented data.
   */

  private IViewerExaminer myViewer;

  static final String DATA_FILE = "$OIVJHOME/data/volumeviz/scene_segmentedInterpolation.iv";
  static final int VOLUME_SIZE = 256;

  SoVolumeRenderingQuality volumeRenderQuality;

  class Body
  {
    private final double posx, posy, posz;
    private final int radius;
    private final float value;

    public Body(SbVec3d pos, int radius, float value)
    {
      this.posx = pos.getX();
      this.posy = pos.getY();
      this.posz = pos.getZ();
      this.radius = radius;
      this.value = value;
    }
  }

  void generateBodies(SoVolumeData vd, int size, Vector<Body> bodies)
  {
    System.out.println("generate bodies ...");
    ByteBuffer data = ByteBuffer.allocateDirect(size * size * size + 1);
    data.order(ByteOrder.nativeOrder());

    double rx,ry,rz;
    for ( int k = 0; k < size; k++ )
      for ( int j = 0; j < size; j++ )
        for ( int i = 0; i < size; i++ )
        {
          byte value = 0;
          for ( Body body : bodies )
          {
            rx = i - body.posx;
            ry = j - body.posy;
            rz = k - body.posz;
            double dist = Math.sqrt(rx*rx+ ry*ry + rz*rz);
            if (dist != 0)
            {
              rx /= dist; ry/= dist; rz /= dist;
            }
            double radius = body.radius * Math.sin(15 * rx * ry * rz);

            if ( dist < radius )
              value += (byte)body.value;
          }
          data.put(value);
        }

    SbDataType dataType = new SbDataType(SbDataType.DataTypes.UNSIGNED_BYTE);
    vd.data.setValue(new SbVec3i32(size, size, size), dataType, 0, data, SoSFArray.CopyPolicies.NO_COPY);
    System.out.println("done");
  }

  private SoNode searchNode(SoNode scene, Class<? extends Inventor> c)
  {
    SoSearchAction sa = new SoSearchAction();
    sa.setNodeClass(c);
    sa.apply(scene);

    SoPath path = sa.getPath();
    if ( path == null )
    {
      System.err.println(c.getName() + " not found");
      return null;
    }

    return path.regular.getTail();
  }

  private static SoSeparator readSceneGraph(String fileName)
  {
    SoInput input = new SoInput();

    if ( !input.openFile(fileName) )
    {
      System.err.println("Cannot open file " + fileName);
      return null;
    }

    SoSeparator node = SoDB.readAll(input);
    if ( node == null )
    {
      System.err.println("Problem reading file");
      input.closeFile();
      return null;
    }

    input.closeFile();
    return node;
  }

  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    SoLDMGlobalResourceParameters.setViewCulling(false);
    SoLDMGlobalResourceParameters.setViewpointRefinement(false);
    SoLDMGlobalResourceParameters.setScreenResolutionCulling(false);

    SoSeparator scene = readSceneGraph(DATA_FILE);

    SoSeparator root = new SoSeparator();
    root.addChild(scene);

    SoVolumeData vd = (SoVolumeData) searchNode(scene, SoVolumeData.class);
    SoVolumeIsosurface iso = (SoVolumeIsosurface) searchNode(scene, SoVolumeIsosurface.class);

    if ( iso.isovalues.getNum() < 3 )
    {
      System.err.println("Not enough isovalues, 3 needed");
      System.exit(0);
    }
    float[] values = iso.isovalues.getValues(0);

    Vector<Body> bodies = new Vector<Body>();
    bodies.add(new Body(new SbVec3d(30, 7 * VOLUME_SIZE / 8, 30), 30, values[0]));
    bodies.add(new Body(new SbVec3d(VOLUME_SIZE / 3, VOLUME_SIZE / 3, VOLUME_SIZE / 3), 90, values[1]));
    bodies.add(new Body(new SbVec3d(3 * VOLUME_SIZE / 4, 3 * VOLUME_SIZE / 4, 3 * VOLUME_SIZE / 4), 40, values[2]));
    generateBodies(vd, VOLUME_SIZE, bodies);

    volumeRenderQuality = (SoVolumeRenderingQuality) searchNode(scene, SoVolumeRenderingQuality.class);
    // Enhance lighting
    volumeRenderQuality.lighting.setValue(true);
    volumeRenderQuality.deferredLighting.setValue(false);

    // Set up viewer:
    myViewer.setSceneGraph(root);
    myViewer.viewAll();

    buildUI();
  }

  void buildUI()
  {
    Panel menuPanel = new Panel();
    menuPanel.add(new JLabel("Interpolation method: "));
    final JComboBox<String> cmbMethod = new JComboBox<>(new String[] { "Segmented", "Gradient", "Color and Gradient" });
    cmbMethod.addActionListener(new java.awt.event.ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent evt)
      {
        if ( cmbMethod.getSelectedItem().equals("Segmented") )
        {
          volumeRenderQuality.segmentedInterpolation.setValue(true);
          volumeRenderQuality.colorInterpolation.setValue(true);
        }
        else if ( cmbMethod.getSelectedItem().equals("Gradient") )
        {
          volumeRenderQuality.segmentedInterpolation.setValue(false);
          volumeRenderQuality.colorInterpolation.setValue(false);
        }
        else
        {
          volumeRenderQuality.segmentedInterpolation.setValue(false);
          volumeRenderQuality.colorInterpolation.setValue(true);
        }
      }
    });
    menuPanel.add(cmbMethod);

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(menuPanel, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  public static void main(String argv[])
  {
    Main example = new Main();
    example.demoMain("Segmented interpolation");
  }
}
