package com.openinventor.inventor.viewercomponents.awt.newt.renderareas;

import java.awt.Canvas;
import java.util.List;

import javax.swing.SwingUtilities;

import com.jogamp.nativewindow.CapabilitiesImmutable;
import com.jogamp.nativewindow.NativeWindowException;
import com.jogamp.newt.awt.NewtCanvasAWT;
import com.jogamp.newt.opengl.GLWindow;
import com.jogamp.opengl.DefaultGLCapabilitiesChooser;
import com.jogamp.opengl.GLAutoDrawable;
import com.jogamp.opengl.GLCapabilities;
import com.jogamp.opengl.GLEventListener;
import com.jogamp.opengl.GLProfile;
import com.openinventor.inventor.SbVec2i32;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoSceneManager;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.devices.SoGLContext;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.viewercomponents.SoRawStereoParameters;
import com.openinventor.inventor.viewercomponents.SoRenderAreaCore;
import com.openinventor.inventor.viewercomponents.awt.IRenderArea;
import com.openinventor.inventor.viewercomponents.awt.Timer;

public class RenderArea extends NewtCanvasAWT implements GLEventListener, IRenderArea
{

  protected SoRenderAreaCore m_renderAreaCore;
  protected SoNode m_sceneGraph;
  protected SoGLRenderAction.TransparencyTypes m_transparencyType;
  private SoGLRenderAction m_glRenderAction;
  private SoSceneManager.AntialiasingModes m_antialiasingMode;
  private float m_antialiasingQuality;
  private int m_stillAntialiasingDelay;
  private float m_stillAntialiasingQuality;
  private boolean m_stereo;
  private boolean m_customGlRenderAction;

  // JOGL
  protected GLWindow m_glWindow;

  public RenderArea()
  {
    this(false);
  }

  public RenderArea(boolean enableStereo)
  {
    super();

    m_renderAreaCore = null;
    m_sceneGraph = null;

    // init the system timer with a javax.swing.Timer
    SoDB.setSystemTimer(new Timer());

    // default configuration
    GLProfile profile = GLProfile.getDefault();

    // define requested GL capabilities
    final GLCapabilities glCapabilities = new GLCapabilities(profile);
    glCapabilities.setDoubleBuffered(true);
    glCapabilities.setHardwareAccelerated(true);
    glCapabilities.setStereo(enableStereo);
    glCapabilities.setDepthBits(24);
    glCapabilities.setStencilBits(8);
    glCapabilities.setSampleBuffers(false);
    glCapabilities.setNumSamples(0);
    glCapabilities.setAlphaBits(8);

    // custom GL capabilities chooser
    OptionalStereoGLCapabilitiesChooser chooser = new OptionalStereoGLCapabilitiesChooser();

    // create a new GLWindow with default profile
    m_glWindow = GLWindow.create(glCapabilities);
    m_glWindow.setCapabilitiesChooser(chooser);
    m_glWindow.addGLEventListener(this);

    // define default values
    m_transparencyType = SoGLRenderAction.TransparencyTypes.NO_SORT;
    m_glRenderAction = null;
    m_antialiasingMode = SoSceneManager.AntialiasingModes.NO_ANTIALIASING;
    m_antialiasingQuality = 0;
    m_stillAntialiasingDelay = 0;
    m_stillAntialiasingQuality = 0;
    m_stereo = false;
    m_customGlRenderAction = false;

    setNEWTChild(m_glWindow);
  }

  @Override
  public void setTransparencyType(SoGLRenderAction.TransparencyTypes type)
  {
    m_transparencyType = type;
    if ( m_renderAreaCore != null )
      m_renderAreaCore.setTransparencyType(m_transparencyType);
  }

  @Override
  public SoGLRenderAction.TransparencyTypes getTransparencyType()
  {
    if ( m_renderAreaCore != null )
      return m_renderAreaCore.getTransparencyType();
    else
      return m_transparencyType;
  }

  @Override
  public void setGLRenderAction(SoGLRenderAction ra)
  {
    m_glRenderAction = ra;
    m_customGlRenderAction = true;
    if ( m_renderAreaCore != null )
      m_renderAreaCore.setGLRenderAction(ra);
  }

  @Override
  public SoGLRenderAction getGLRenderAction()
  {
    if ( m_renderAreaCore != null )
      return m_renderAreaCore.getGLRenderAction();
    else
      return m_glRenderAction;
  }

  @Override
  public float getAntialiasingQuality()
  {
    if ( m_renderAreaCore == null )
      return m_antialiasingQuality;
    return m_renderAreaCore.getAntialiasingQuality();
  }

  @Override
  public void setAntialiasingQuality(float quality)
  {
    m_antialiasingQuality = quality;
    if ( m_renderAreaCore != null )
      m_renderAreaCore.setAntialiasingQuality(quality);
  }

  @Override
  public SoSceneManager.AntialiasingModes getAntialiasingMode()
  {
    if ( m_renderAreaCore == null )
      return m_antialiasingMode;
    return m_renderAreaCore.getAntialiasingMode();
  }

  @Override
  public void setAntialiasingMode(SoSceneManager.AntialiasingModes mode)
  {
    m_antialiasingMode = mode;
    if ( m_renderAreaCore != null )
      m_renderAreaCore.setAntialiasingMode(mode);
  }

  @Override
  public float getStillSuperSamplingQuality()
  {
    if ( m_renderAreaCore == null )
      return m_antialiasingQuality;
    return m_renderAreaCore.getAntialiasingQuality();
  }

  @Override
  public void setStillSuperSamplingQuality(float quality)
  {
    m_stillAntialiasingQuality = quality;
    if ( m_renderAreaCore != null )
      m_renderAreaCore.setStillSuperSamplingQuality(quality);
  }

  @Override
  public int getStillSuperSamplingDelay()
  {
    if ( m_renderAreaCore == null )
      return m_stillAntialiasingDelay;
    return m_renderAreaCore.getStillSuperSamplingDelay();
  }

  @Override
  public void setStillSuperSamplingDelay(int delay)
  {
    m_stillAntialiasingDelay = delay;
    if ( m_renderAreaCore != null )
      m_renderAreaCore.setStillSuperSamplingDelay(delay);
  }

  @Override
  public Canvas getComponent()
  {
    return this;
  }

  @Override
  public void activateStereo(boolean activated)
  {
    m_stereo = activated;

    if ( m_renderAreaCore == null )
      return;

    if ( activated )
    {
      // activate raw stereo
      if ( !isRawStereoAvailable() )
        throw new UnsupportedOperationException("Stereo buffers are not enabled");

      SoRawStereoParameters params = new SoRawStereoParameters();
      m_renderAreaCore.setStereoParameters(params);
      m_renderAreaCore.activateStereo(true);
    }
    else
      m_renderAreaCore.activateStereo(false);
  }

  @Override
  public boolean isRawStereoAvailable()
  {
    return m_glWindow.getChosenGLCapabilities().getStereo();
  }

  @Override
  public void setSceneGraph(SoNode sceneGraph)
  {
    m_sceneGraph = sceneGraph;

    if ( m_renderAreaCore != null )
      m_renderAreaCore.setSceneGraph(sceneGraph);
  }

  @Override
  public void scheduleRedraw()
  {
    if ( m_renderAreaCore != null )
      m_renderAreaCore.getSceneManager().scheduleRedraw();
  }

  protected synchronized void render()
  {
    if ( m_renderAreaCore != null )
      m_renderAreaCore.render();
  }

  private synchronized void disposeRenderAreaCore()
  {
    m_renderAreaCore.dispose();
    m_renderAreaCore = null;
  }

  private synchronized void setSize(SbVec2i32 newSize)
  {
    if ( m_renderAreaCore != null )
      m_renderAreaCore.setSize(newSize);
  }

  @Override
  public void init(GLAutoDrawable drawable)
  {
    // let OpenInventor be aware of the current context created by JOGL
    SoGLContext context = SoGLContext.getCurrent(true);
    m_renderAreaCore = new SoRenderAreaCore(context);
    m_renderAreaCore.setSceneGraph(m_sceneGraph);
    m_renderAreaCore.setTransparencyType(m_transparencyType);
    if ( m_glRenderAction == null )
      m_glRenderAction = m_renderAreaCore.getGLRenderAction();
    else
      m_renderAreaCore.setGLRenderAction(m_glRenderAction);
    m_renderAreaCore.setAntialiasingMode(m_antialiasingMode);
    m_renderAreaCore.setAntialiasingQuality(m_antialiasingQuality);
    m_renderAreaCore.setStillSuperSamplingDelay(m_stillAntialiasingDelay);
    m_renderAreaCore.setStillSuperSamplingQuality(m_stillAntialiasingQuality);
    activateStereo(m_stereo);
    context.unbind();
  }

  @Override
  public void dispose(GLAutoDrawable drawable)
  {
    if ( !m_customGlRenderAction )
      m_glRenderAction = null;
	disposeRenderAreaCore();
  }

  @Override
  public void dispose()
  {
    destroy();
  }

  @Override
  public void display(GLAutoDrawable drawable)
  {
    // Render scene in EDT
    if ( SwingUtilities.isEventDispatchThread() )
    {
      // render Open Inventor scene graph
      render();
    }
    else
      SwingUtilities.invokeLater(new Runnable()
      {

        @Override
        public void run()
        {
          m_glWindow.display();
        }
      });
  }

  @Override
  public void reshape(GLAutoDrawable drawable, int x, int y, final int width, final int height)
  {
    // Update viewport region in EDT
    if ( SwingUtilities.isEventDispatchThread() )
      setSize(new SbVec2i32(width, height));
    else
      SwingUtilities.invokeLater(new Runnable()
      {

        @Override
        public void run()
        {
          setSize(new SbVec2i32(width, height));
        }
      });

  }

  /**
   * Custom implementation of GLCapabilitiesChooser.<br>
   * The DefaultGLCapabilitiesChooser does not allow to define an optional but
   * not mandatory GL capability. In particular, stereo capability is considered
   * as a mandatory capability.<br>
   * This implementation allows selecting an optional stereo capability: if no
   * stereo capability is available, it continues to search other requested
   * (mandatory) capabilities.
   */
  class OptionalStereoGLCapabilitiesChooser extends DefaultGLCapabilitiesChooser
  {
    @Override
    public int chooseCapabilities(CapabilitiesImmutable desired, List<? extends CapabilitiesImmutable> available,
        int windowSystemRecommendedChoice)
    {
      int chosen = -1;
      try
      {
        // launch default chooser
        chosen = super.chooseCapabilities(desired, available, windowSystemRecommendedChoice);
      }
      catch (NativeWindowException exc)
      {
        // chosen index is not valid, check if stereo capability has been
        // requested
        final GLCapabilities gldes = (GLCapabilities) desired.cloneMutable();
        if ( gldes.getStereo() )
        {
          // stereo capability has been requested but it is not available.
          System.err.println("Could not enable stereo buffers");

          // try to get the best config with no stereo
          // if we don't do that, the first available config will be chosen
          // without paying attention to others requested GL capabilities.
          gldes.setStereo(false);
          chosen = super.chooseCapabilities(gldes, available, windowSystemRecommendedChoice);
        }
        else
          // let JOGL do its job...
          throw exc;
      }

      return chosen;
    }
  }
}
