package com.openinventor.inventor.viewercomponents.awt.newt.renderareas;

import com.jogamp.opengl.GLAutoDrawable;
import javax.swing.SwingUtilities;

import com.jogamp.newt.event.KeyEvent;
import com.jogamp.newt.event.KeyListener;
import com.jogamp.newt.event.MouseEvent;
import com.jogamp.newt.event.MouseListener;
import com.openinventor.inventor.SbEventListener;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.events.SoLocation2Event;
import com.openinventor.inventor.events.SoMouseButtonEvent;
import com.openinventor.inventor.events.SoMouseWheelEvent;
import com.openinventor.inventor.nodes.SoCamera;
import com.openinventor.inventor.nodes.SoGroup;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoStereoCamera;
import com.openinventor.inventor.viewercomponents.SiRenderArea;
import com.openinventor.inventor.viewercomponents.SiRenderArea.RenderEventArg;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor.CameraMode;

public class RenderAreaInteractive extends RenderArea implements MouseListener, KeyListener, IRenderAreaInteractive
{

  protected SceneInteractor m_rootSceneGraph;
  private SoGroup m_appSceneGraph;

  private ClippingPlanesAdjustMode m_clippingMode;

  public RenderAreaInteractive()
  {
    this(false);
  }

  public RenderAreaInteractive(boolean enableStereo)
  {
    this(true, enableStereo);
  }

  protected RenderAreaInteractive(boolean buildRootSceneGraph, boolean enableStereo)
  {
    super(enableStereo);

    m_clippingMode = ClippingPlanesAdjustMode.AUTO;

    if ( buildRootSceneGraph )
    {
      m_rootSceneGraph = new SceneInteractor();
      buildSceneGraph();
    }

    m_glWindow.addMouseListener(this);
    m_glWindow.addKeyListener(this);
  }

  protected void buildSceneGraph()
  {
    m_appSceneGraph = new SoGroup();
    m_rootSceneGraph.addChild(m_appSceneGraph);
    super.setSceneGraph(m_rootSceneGraph);
  }

  @Override
  public void setSceneGraph(SoNode sceneGraph)
  {
    m_appSceneGraph.removeAllChildren();
    if ( sceneGraph != null )
      m_appSceneGraph.addChild(sceneGraph);
  }

  @Override
  protected synchronized void render()
  {
    if ( m_clippingMode == ClippingPlanesAdjustMode.AUTO && m_renderAreaCore != null )
    {
      // Update camera clipping planes before each rendering.
      m_rootSceneGraph.adjustClippingPlanes(m_renderAreaCore.getSceneManager().getViewportRegion());
    }

    // render scene
    super.render();
  }

  @Override
  public void init(GLAutoDrawable drawable)
  {
    super.init(drawable);

    m_renderAreaCore.onStartRender().addEventListener(new SbEventListener<SiRenderArea.RenderEventArg>()
    {
      @Override
      public void onEvent(RenderEventArg arg0)
      {
        m_glWindow.display();
      }
    });
  }

  /**
   * Get the camera clipping planes adjustment mode.
   */
  @Override
  public ClippingPlanesAdjustMode getClippingPlanesAdjustMode()
  {
    return m_clippingMode;
  }

  /**
   * Set the camera clipping planes adjustment mode. <br>
   * When adjustment mode is set to {@code AUTO}, the camera near and far planes
   * are dynamically adjusted to be as tight as possible (least amount of stuff
   * is clipped) before each render traversal.<br>
   * When adjustment mode is set to {@code MANUAL}, the user is expected to
   * manually set those planes. Updating clipping planes after a camera move is
   * not enough, if a dragger or a rendered shape is moved, they can disappear
   * or become partially clipped.<br>
   * Default is {@code AUTO}.
   */
  @Override
  public void setClippingPlanesAdjustMode(ClippingPlanesAdjustMode mode)
  {
    if ( m_clippingMode != mode )
    {
      m_clippingMode = mode;
      scheduleRedraw();
    }
  }

  /*
   * @deprecated As of Open Inventor 10.5.0, use {@link #getSceneInteractor} instead.
  */
  @Deprecated
  @Override
  public SceneInteractor getRootSceneGraph()
  {
    return m_rootSceneGraph;
  }

  @Override
  public SceneInteractor getSceneInteractor()
  {
    return m_rootSceneGraph;
  }

  @Override
  public void viewAll(SbViewportRegion viewport)
  {
    m_rootSceneGraph.viewAll(viewport);
  }

  @Override
  public void viewAxis(SbVec3f direction, SbVec3f up)
  {
    m_rootSceneGraph.viewAxis(direction, up);
  }

  @Override
  public void saveCamera()
  {
    m_rootSceneGraph.getCameraInteractor().pushCamera();
  }

  @Override
  public void restoreCamera()
  {
    m_rootSceneGraph.getCameraInteractor().popCamera();
  }

  @Override
  public void setCameraType(CameraMode mode)
  {
    m_rootSceneGraph.setCameraMode(mode);
  }

  @Override
  public void activateStereo(boolean activated)
  {
    if ( activated )
    {
      // check camera
      if ( !isStereoSupported() )
        throw new UnsupportedOperationException("Cannot activate stereo: current camera is not a SoStereoCamera!");
    }
    super.activateStereo(activated);
  }

  @Override
  public boolean isStereoSupported()
  {
    // check camera
    SoCamera camera = m_rootSceneGraph.getCamera();
    return camera instanceof SoStereoCamera;
  }

  @Override
  public void setStereoCameraOffset(float offset)
  {
    SoCamera camera = m_rootSceneGraph.getCamera();
    try
    {
      SoStereoCamera stereoCamera = (SoStereoCamera) camera;
      stereoCamera.offset.setValue(offset);
    }
    catch (ClassCastException e)
    {
      throw new UnsupportedOperationException("Current camera is not a SoStereoCamera!");
    }
  }

  @Override
  public void setStereoCameraBalance(float balance)
  {
    SoCamera camera = m_rootSceneGraph.getCamera();
    try
    {
      SoStereoCamera stereoCamera = (SoStereoCamera) camera;
      stereoCamera.balance.setValue(balance);
    }
    catch (ClassCastException e)
    {
      throw new UnsupportedOperationException("Current camera is not a SoStereoCamera!");
    }
  }

  @Override
  public void mousePressed(MouseEvent e)
  {
    SoMouseButtonEvent evt = NewtEventToSoEvent.getMousePressEvent(e, this);
    processEventInEDT(evt);
  }

  @Override
  public void mouseWheelMoved(MouseEvent e)
  {
    SoMouseWheelEvent evt = NewtEventToSoEvent.getMouseWheelEvent(e, this);
    processEventInEDT(evt);
  }

  @Override
  public void mouseDragged(MouseEvent e)
  {
    SoLocation2Event evt = NewtEventToSoEvent.getMouseMoveEvent(e, this);
    processEventInEDT(evt);
  }

  @Override
  public void mouseMoved(MouseEvent e)
  {
    SoLocation2Event evt = NewtEventToSoEvent.getMouseMoveEvent(e, this);
    processEventInEDT(evt);
  }

  @Override
  public void mouseClicked(MouseEvent e)
  {
    if ( e.getClickCount() == 2 )
    {
      SoMouseButtonEvent evt = NewtEventToSoEvent.getMouseDoubleClickEvent(e, this);
      processEventInEDT(evt);
    }
  }

  @Override
  public void mouseReleased(MouseEvent e)
  {
    SoMouseButtonEvent evt = NewtEventToSoEvent.getMouseReleaseEvent(e, this);
    processEventInEDT(evt);
  }

  @Override
  public void mouseEntered(MouseEvent e)
  {
    SoLocation2Event evt = NewtEventToSoEvent.getMouseEnterEvent(e, this);
    processEventInEDT(evt);
  }

  @Override
  public void mouseExited(MouseEvent e)
  {
    SoLocation2Event evt = NewtEventToSoEvent.getMouseLeaveEvent(e, this);
    processEventInEDT(evt);
  }

  @Override
  public void keyPressed(KeyEvent e)
  {
    SoKeyboardEvent evt = NewtEventToSoEvent.getKeyPressEvent(e, this);
    processEventInEDT(evt);
  }

  @Override
  public void keyReleased(KeyEvent e)
  {
    SoKeyboardEvent evt = NewtEventToSoEvent.getKeyReleaseEvent(e, this);
    processEventInEDT(evt);
  }

  private synchronized void processEvent(SoEvent evt)
  {
    if ( m_renderAreaCore != null )
      m_renderAreaCore.processEvent(evt);
  }

  private void processEventInEDT(final SoEvent evt)
  {
    if ( SwingUtilities.isEventDispatchThread() )
    {
      processEvent(evt);
    }
    else
      SwingUtilities.invokeLater(new Runnable()
      {

        @Override
        public void run()
        {
          processEvent(evt);
        }
      });
  }
}
