package com.openinventor.inventor.viewercomponents.nodes;

import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.fields.SoSFTime;
import com.openinventor.inventor.sensors.SoFieldSensor;

/**
 * Base class for real-time animation.
 * <p>
 * The real-time animation is based on a field sensor connected to the <b>
 * realTime </b> global field.<br>
 * The abstract method {@link #animate()} is called whenever the <b> realTime
 * </b> global field changes.
 */
abstract class Animator
{

  private static final SoSFTime REAL_TIME_FIELD = (SoSFTime) SoDB.getGlobalField("realTime");

  private SoFieldSensor m_animationSensor;
  private AnimatorListener m_listener;

  protected Animator()
  {
    m_listener = null;

    m_animationSensor = new SoFieldSensor();
    m_animationSensor.setTask(new Runnable()
    {
      @Override
      public void run()
      {
        animate();
      }
    });
  }

  /**
   * Method called whenever the <b> realTime </b> global field changes. Override
   * this method to create your animation.
   */
  protected abstract void animate();

  /**
   * Set the listener to receive notifications of animation changes.
   */
  public void setListener(AnimatorListener listener)
  {
    m_listener = listener;
  }

  /**
   * Start the animation and notify the animation listener if any.
   */
  public void start()
  {
    if ( m_animationSensor.getAttachedField() == null )
    {
      m_animationSensor.attach(REAL_TIME_FIELD);
      m_animationSensor.schedule();

      if ( m_listener != null )
        m_listener.animationStarted();
    }
  }

  /**
   * Stop the animation and notify the animation listener if any.
   */
  public void stop()
  {
    if ( m_animationSensor.getAttachedField() != null )
    {
      m_animationSensor.detach();
      m_animationSensor.unschedule();

      if ( m_listener != null )
        m_listener.animationStopped();
    }
  }

}
