package com.openinventor.inventor.viewercomponents.nodes;

import com.openinventor.inventor.Axis;
import com.openinventor.inventor.IntegerValuedEnum;
import com.openinventor.inventor.SbMatrix;
import com.openinventor.inventor.SbTime;
import com.openinventor.inventor.SbVec2f;
import com.openinventor.inventor.SbVec2s;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.actions.SoHandleEventAction;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.events.SoLocation2Event;
import com.openinventor.inventor.events.SoMouseButtonEvent;
import com.openinventor.inventor.events.SoMouseWheelEvent;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.sensors.SoAlarmSensor;
import com.openinventor.inventor.sensors.SoFieldSensor;
import com.openinventor.inventor.viewercomponents.SoCameraInteractor;

/**
 * <strong>(Preview Feature) </strong> Tool class for building a basic
 * interactive OpenInventor application with mode-less scene "orbiter" viewing
 * behavior.
 *
 * <p style="background-color: #c7c7c7;">
 * Preview Feature means this class is fully supported and can be used in Open
 * Inventor applications. Being tagged as a Preview Feature just means that the
 * implementation is still subject to API changes and adjustments based on
 * feedback from early adopters. Please be also aware that source compatibility
 * might be broken regardless of the Open Inventor compatibility changes policy
 * due to our commitment to bring needed changes to  make sure that the specifications
 * of this Preview Feature match the expectations of our customers.
 * </p>
 *
 * <p>
 * The SceneOrbiter is an extension of the {@link SceneInteractor} node that
 * provides camera and headlight manipulations like panning, zooming and
 * orbiting similar to the SceneExaminer (in Orbit navigation mode).
 * <p>
 * The SceneOrbiter provides a "mode-less" viewer which is very convenient for
 * users. The SceneExaminer viewers are always in either viewing mode (mouse
 * events control the camera) or in selection mode (mouse events are sent to the
 * scene graph and objects can be selected). This often requires the user to
 * frequently switch between viewing and selection mode (for example by pressing
 * the ESC key) while interacting with the scene. SceneOrbiter does not have
 * modes. For example, a mouse click without moving the mouse is interpreted as
 * a selection and the event is sent to the scene graph, but a mouse click and
 * "drag" is interpreted as controlling the camera.
 * <p>
 * The SceneOrbiter provides two different methods to control the rotation of
 * the camera around the scene : trackball and turntable, see
 * {@link RotationMethods} enum.<br>
 * The trackball method allows the camera to move along any circular orbit around the
 * scene and look in the direction of its center. The orbit orientation can
 * change at any time while interpreting a mouse drag. Thus the trackball is
 * known as an unconstrained orbiting mode. The trackball is the unique method
 * used by the historical examiner viewer and by the SceneExaminer.<br>
 * The additional turntable method is a more constrained orbiting mode. It allows
 * the camera to move only along vertical orbits such as the earth meridians, and
 * along horizontal circles such as the earth parallels. The vertical orbits and
 * horizontal circles are related to an up axis, that can be changed calling
 * {@link SceneOrbiter#setUpAxis(Axis)}. The horizontal circles are centered
 * around this up axis, and the vertical orbits share their south and north pole
 * on this up axis. As the camera cannot move along any other circle, this mode
 * is more constrained compared to the trackball mode. However, that gives a more
 * natural navigation experience when viewing a scene that has strong vertical and
 * horizontal references. For instance, when orbiting in turntable mode around a
 * scene representing the Eiffel tower, this one always looks vertical on your
 * screen. Compared to the trackball mode, the turntable mode does not need any
 * tedious small mousemove correction to keep the scene well aligned with the
 * borders of the screen.
 * <p>
 * The 'headlight', an SoDirectionalLight node, is automatically aligned with
 * the camera's view direction.
 * <p>
 * An SoViewingCube node is automatically added to the scene. The viewing cube
 * can be hidden by calling the {@link SceneOrbiter#enableViewingCube(boolean)}
 * method.<br>
 * Note that the up axis of the viewing cube is synchronized with
 * the up axis of the SceneOrbiter.
 * <p>
 * See parent class {@link SceneInteractor} for more details about the structure
 * of the internal scene graph. <br>
 * The SceneOrbiter uses an instance of {@link SoCameraInteractor} to manipulate
 * the camera in response to OpenInventor events.
 *
 * <p>
 * <b>Notes:</b>
 * <ul>
 * <li>Window system integration<br>
 * The SceneOrbiter needs a component that integrates the Open Inventor 3D
 * rendering window with the native window system. System dependent tasks
 * include creating a window, placing the window in the application's user
 * interface, initializing OpenGL and processing the native event/message loop.
 * System independent support for this is provided by the
 * {@link com.openinventor.inventor.viewercomponents.SoRenderAreaCore} class.
 * Example components are provided for AWT and SWT toolkits.
 * <li>Event handling<br>
 * The SceneOrbiter needs a component that builds OpenInventor events (SoEvent)
 * from native system events. System independent support for this is provided by
 * the {@link com.openinventor.inventor.viewercomponents.SoEventBuilder} class.
 * Example components are provided for AWT and SWT toolkits: AWTEventToSoEvent
 * and SWTEventToSoEvent.
 * <li>Library<br>
 * A basic version of SceneOrbiter is a supported part of the Open Inventor API
 * and a prebuilt jar is provided.
 * <li>Source code<br>
 * The basic version of SceneOrbiter is also provided as source code in the
 * sources folder to allow applications to customize and build their own
 * interactive tool class.<br>
 * See $OIVJHOME/source/com/openinventor/inventor/viewercomponents/nodes.
 * <li>Scene graph<br>
 * The application scene graph should be the last child of the SceneOrbiter. The
 * initial application scene graph can be added by simply calling the inherited
 * method addChild(). But note that if you need to replace the application scene
 * graph, for example loading a new data set, do <i>not</i> call
 * removeAllChildren(). That would also remove the SceneOrbiter's camera,
 * headlight and event handler nodes. Add an SoSeparator to the SceneOrbiter to
 * serve as the "scene graph holder", then add and remove the application scene
 * graph from this node.
 * <li>Clip planes<br>
 * SceneOrbiter automatically adjusts the 'near' and 'far' clipping planes when
 * events modifying the camera are handled. This adjustment, based on the
 * bounding box of the scene, ensures that shapes will not be clipped as the
 * camera orbits and also that depth buffer precision is maximized.<br>
 * <b>Note</b>: Updating clipping planes after a camera move can be 
 * insufficient. If the scene graph is modified or if a dragger or a rendered
 * shape is moved, they can disappear or become partially clipped. A classic
 * implementation of a render area must adjust clipping planes before each
 * rendering by calling the provided method
 * {@link #adjustClippingPlanes(SbViewportRegion)}. See render area's
 * implementations available in $OIVJHOME/examples/inventor/viewercomponents/awt
 * and $OIVJHOME/examples/inventor/viewercomponents/swt folders for examples of
 * {@code adjustClippingPlanes} use.
 * <li>Compatibility with classical viewers<br>
 * Please note that some interaction behaviors are different than the classic
 * Open Inventor viewer classes :
 * <ul>
 * <li><b> Left Mouse + Shift </b> does Zoom in/out.</li>
 * <li><b> Mouse wheel</b> performs a dolly relative to the <i> cursor
 * position</i>, not the center of the viewport.</li> <br>
 * <li>The classic <b>ESC key</b> behavior is not implemented.<br>
 * (Not necessary because viewing and selection are supported without separate
 * modes.)</li>
 * <li>The classic <b>Alt key</b> behavior is not implemented. This key is
 * reserved for application use.</li>
 * <li>The <b>Right Mouse</b> button does not display a pop-up menu. This button
 * is reserved for application use.</li>
 * <li>Seek mode is not supported.</li>
 * </ul>
 * <li>Compatibility with SceneExaminer
 * <ul>
 * <li>Only the left mouse button has defined behaviors.<br>
 * Pan and zoom behaviors require pressing a keyboard key while dragging the
 * mouse.
 * <li>Touch events are not currently handled.
 * </ul>
 * </ul>
 * <p>
 * <b>Usage:</b>
 * <ul>
 *
 * <li><b>Left Mouse:</b> Rotate the scene</li>
 * <li><b>Left Mouse + Ctrl:</b> Pan the scene.</li>
 * <li><b>Left Mouse + Shift:</b> Zoom in/out the scene.</li>
 * <li><b>Mouse Wheel:</b> Zoom in/out (zoom center is the mouse cursor
 * location).</li>
 * </ul>
 * </ul>
 * </ul>
 */
public class SceneOrbiter extends SceneInteractor
{

  /**
   * Defines how the mouse events or touch events control the
   * rotations of the scene.
   */
  public enum RotationMethods implements IntegerValuedEnum
  {
    /**
     * Constrained mode which forces the camera to follow meridians and parallel
     * circles related to the up-axis. This mode is very useful to help the user
     * to see the up-axis of the scene vertical, i.e. parallel to the left/right
     * borders of the screen.
     */
    TURNTABLE(0),

    /**
     * Unconstrained mode which allows the camera to follow any circular orbit.
     * Thus, this mode allows to render the scene with any orientation.
     */
    TRACKBALL(1);

    private int value;

    private RotationMethods(int val)
    {
      this.value = val;
    }

    @Override
    public int getValue()
    {
      return this.value;
    }
  }

  private InteractionOrbiter m_interaction;
  private SoViewingCube m_viewingCube;
  private SoSwitch m_viewingCubeSwitch;

  private SoFieldSensor m_upAxisSensor;

  public SceneOrbiter()
  {
    super();

    m_interaction = new InteractionOrbiter(m_cameraInteractor);
    m_viewingCubeSwitch = new SoSwitch();
    m_viewingCube = new SoViewingCube();
    m_viewingCube.sceneCamera.setValue(getCamera());
    m_viewingCubeSwitch.addChild(m_viewingCube);
    m_viewingCubeSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);

    // TURNTABLE by default
    m_interaction.m_orbitMethod = RotationMethods.TURNTABLE;
    // up axis = Y by default
    m_interaction.m_upAxis = Axis.Y;
    m_viewingCube.upAxis.setValue(Axis.Y);

    m_upAxisSensor = new SoFieldSensor(new Runnable()
    {
      @Override
      public void run()
      {
        // when upAxis of viewing cube changes, update upAxis of orbiter
        m_interaction.m_upAxis = m_viewingCube.upAxis.getValue(Axis.class);
      }
    });
    m_upAxisSensor.attach(m_viewingCube.upAxis);

    addChild(m_viewingCubeSwitch);
  }

  /**
   * Sets the constraint level to limit the turntable rotation.
   * <p>
   * The level is a value between [0,1] used to adjust the constraint on the
   * turntable rotation according to the speed of movement :
   * <ul>
   * <li>0 means no constraint on the rotation, regardless of the interaction speed
   * <li>1 means very strongly constrained even if interaction is slow.
   * </ul>
   * This prevents unexpected changes in the axis of rotation during fast
   * movements.<br>
   * Default is 0 which means no constraint on the rotation.
   *
   * @param level
   *          the constraint level.
   */
  public void setConstraintLevel(float level)
  {
    m_interaction.m_turntableConstraintLevel = level;
  }

  /**
   * Returns the current constraint level which limits the
   * turntable rotation.<br>
   *
   * @return the constraint level.
   */
  public float getConstraintLevel()
  {
    return m_interaction.m_turntableConstraintLevel;
  }

  /**
   * Sets the scene rotation method.
   * <p>
   * Default is TURNTABLE.
   *
   * @param method
   *          the new scene rotation method.
   */
  public void setRotationMethod(RotationMethods method)
  {
    m_interaction.m_orbitMethod = method;
  }

  /**
   * Returns the current scene rotation method.
   *
   * @return the scene rotation method
   */
  public RotationMethods getRotationMethod()
  {
    return m_interaction.m_orbitMethod;
  }

  /**
   * Sets the up axis of the scene.
   * <p>
   * Default is Y.<br>
   * Note that the up axis of the viewing cube is synchronized with the up axis
   * of the SceneOrbiter.
   *
   * @param axis
   *          the new up axis of the scene
   */
  public void setUpAxis(Axis axis)
  {
    m_interaction.m_upAxis = axis;
    // update upAxis of viewing cube if needed
    if ( m_viewingCube.upAxis.getValue(Axis.class) != axis )
      m_viewingCube.upAxis.setValue(axis);
  }

  /**
   * Returns the current up axis of the scene.
   *
   * @return the up axis of the scene.
   */
  public Axis getUpAxis()
  {
    return m_interaction.m_upAxis;
  }

  /**
   * Enables or disables the viewing cube. Default is true.
   */
  public void enableViewingCube(boolean enabled)
  {
    m_viewingCubeSwitch.whichChild.setValue(enabled ? SoSwitch.SO_SWITCH_ALL : SoSwitch.SO_SWITCH_NONE);
  }

  /**
   * Returns if viewing cube is enabled.
   */
  public boolean isViewingCubeEnabled()
  {
    return m_viewingCubeSwitch.whichChild.getValue() == SoSwitch.SO_SWITCH_ALL;
  }

  /**
   * Returns the viewing cube object.
   */
  public SoViewingCube getViewingCube()
  {
    return m_viewingCube;
  }

  @Override
  public void setCameraMode(CameraMode mode)
  {
    super.setCameraMode(mode);

    if ( m_interaction != null )
      m_interaction.setCameraInteractor(m_cameraInteractor);

    if ( m_viewingCube != null )
      m_viewingCube.sceneCamera.setValue(getCamera());
  }

  @Override
  protected void keyPressed(SoKeyboardEvent event, SoHandleEventAction action)
  {
    m_interaction.beginAction(event, action);
  }

  @Override
  protected void keyReleased(SoKeyboardEvent event, SoHandleEventAction action)
  {
    m_interaction.beginAction(event, action);
  }

  @Override
  protected void mousePressed(SoMouseButtonEvent event, SoHandleEventAction action)
  {
    m_interaction.beginAction(event, action);
  }

  @Override
  protected void mouseReleased(SoMouseButtonEvent event, SoHandleEventAction action)
  {
    m_interaction.beginAction(event, action);
  }

  @Override
  protected void mouseWheelMoved(SoMouseWheelEvent event, SoHandleEventAction action)
  {
    m_interaction.doAction(event, action);
  }

  @Override
  protected void mouseMoved(SoLocation2Event event, SoHandleEventAction action)
  {
    m_interaction.doAction(event, action);
  }
}

class InteractionOrbiter
{
  private class InteractionEndSensorTask implements Runnable
  {
    @Override
    public void run()
    {
      // stop interaction
      interactiveCountDec();
    };
  }

  private static final float SELECTION_EPSILON = 0.05f;

  private SoCameraInteractor m_cameraInteractor;
  private boolean m_selectionAllowed;
  private boolean m_isButton1Down;
  private SbVec2f m_mouseNormPosition;
  private SbVec2s m_mousePosition;
  private int m_mouseWheelDelta;
  private long m_eventTime;

  protected SceneOrbiter.RotationMethods m_orbitMethod;
  protected Axis m_upAxis;
  protected float m_turntableConstraintLevel;

  // Interaction
  private int m_interactiveCount;
  private SoHandleEventAction m_currentAction;
  private SoAlarmSensor m_interactionEndSensor;
  // waiting time for a new interaction
  private static final double INTERACTION_TIME = 0.5;
  // Speed value to define what we consider as a fast movement.
  // Default is 3 pixels / ms.
  private static final double MAX_INTERACTION_SPEED = 3.0;

  public InteractionOrbiter(SoCameraInteractor cameraInteractor)
  {
    m_cameraInteractor = cameraInteractor;
    m_selectionAllowed = false;
    m_isButton1Down = false;
    m_mouseNormPosition = new SbVec2f();
    m_mousePosition = new SbVec2s();
    m_mouseWheelDelta = SoPreferences.getInteger("OIV_WHEEL_DELTA", 120);
    m_eventTime = 0;
    m_orbitMethod = SceneOrbiter.RotationMethods.TURNTABLE;
    m_upAxis = Axis.Y;
    // no constraint by default
    m_turntableConstraintLevel = 0.f;

    m_interactiveCount = 0;
    m_currentAction = null;
    m_interactionEndSensor = new SoAlarmSensor(new InteractionEndSensorTask());
  }

  public void setCameraInteractor(SoCameraInteractor cameraInteractor)
  {
    m_cameraInteractor = cameraInteractor;
  }

  public void beginAction(SoMouseButtonEvent event, SoHandleEventAction action)
  {
    SoMouseButtonEvent.Buttons button = event.getButton();
    SoMouseButtonEvent.States state = event.getState();

    if ( button == SoMouseButtonEvent.Buttons.BUTTON1 )
    {
      m_isButton1Down = state == SoMouseButtonEvent.States.DOWN;

      boolean ctrlDown = event.wasCtrlDown();
      boolean shiftDown = event.wasShiftDown();
      if ( !m_isButton1Down && m_selectionAllowed && !ctrlDown && !shiftDown )
      {
        // we let pass the event
        return;
      }

      SbViewportRegion vpRegion = action.getViewportRegion();
      m_mouseNormPosition = event.getNormalizedPosition(vpRegion);
      beginAction(action, event.wasCtrlDown(), event.wasShiftDown());

      if ( !m_isButton1Down && !m_selectionAllowed )
      {
        action.setHandled();
      }
    }
  }

  public void beginAction(SoKeyboardEvent event, SoHandleEventAction action)
  {
    SoKeyboardEvent.Keys key = event.getKey();
    SoKeyboardEvent.States state = event.getState();

    if ( key == SoKeyboardEvent.Keys.LEFT_CONTROL || key == SoKeyboardEvent.Keys.LEFT_SHIFT )
    {
      m_selectionAllowed = false;
    }

    switch ( key )
    {
    case LEFT_CONTROL :
      beginAction(action, state == SoKeyboardEvent.States.DOWN, event.wasShiftDown());
      break;
    case LEFT_SHIFT :
      beginAction(action, event.wasCtrlDown(), state == SoKeyboardEvent.States.DOWN);
      break;
    default:
      break;
    }
  }

  protected void beginAction(SoHandleEventAction action, boolean ctrlDown, boolean shiftDown)
  {
    m_currentAction = action;

    if ( m_isButton1Down && ctrlDown )
    {
      beginPan(action);
    }
    else if ( m_isButton1Down && !shiftDown )
    {
      beginSelection(action);
    }
  }

  public void doAction(SoMouseWheelEvent event, SoHandleEventAction action)
  {
    m_currentAction = action;

    // Zoom
    int wheelDelta = event.getDelta() / m_mouseWheelDelta;
    float scaleFactor = (float) Math.pow(2., wheelDelta * Math.PI / 180.);

    doDollyWithCenter(m_mouseNormPosition, scaleFactor, action);
  }

  public void doAction(SoLocation2Event event, SoHandleEventAction action)
  {
    m_currentAction = action;

    boolean ctrlDown = event.wasCtrlDown();
    boolean shiftDown = event.wasShiftDown();

    if ( m_isButton1Down && ctrlDown )
    {
      // BUTTON 1 + CTRL = pan
      doPan(event, action);
    }
    else if ( m_isButton1Down && shiftDown )
    {
      // BUTTON 1 + SHIFT = dolly
      doDolly(event, action);
    }
    else if ( m_isButton1Down )
    {
      if ( m_selectionAllowed )
      {
        doIdentifyGesture(event, action);
      }
      else
      {
        // BUTTON 1 without modifier = orbit
        doOrbit(event, action);
      }
    }

    if ( !action.isHandled() )
    {
      // update mouse cursor's position
      m_mouseNormPosition = event.getNormalizedPosition(action.getViewportRegion());
    }

  }

  protected void beginPan(SoHandleEventAction action)
  {
    m_cameraInteractor.activatePanning(m_mouseNormPosition, action.getViewportRegion());
    action.setHandled();
  }

  protected void beginSelection(SoHandleEventAction action)
  {
    m_selectionAllowed = true;
  }

  protected void beginOrbit(SoHandleEventAction action)
  {
    switch ( m_orbitMethod )
    {
    case TRACKBALL :
      m_cameraInteractor.activateOrbiting(m_mouseNormPosition);
      m_cameraInteractor.setRotationCenter(m_cameraInteractor.getFocalPoint());
      break;
    case TURNTABLE :
      m_mousePosition = action.getEvent().getPosition();
      m_eventTime = action.getEvent().getTime().getMsecValue();
      break;
    default:
      // do nothing
      break;
    }
    action.setHandled();
  }

  protected void doIdentifyGesture(SoEvent event, SoHandleEventAction action)
  {
    SbViewportRegion vpRegion = action.getViewportRegion();
    SbVec2f current_mousePositionNorm = event.getNormalizedPosition(vpRegion);
    SbVec2f delta = current_mousePositionNorm.minus(m_mouseNormPosition);

    float length = delta.length();

    if ( length > SELECTION_EPSILON )
    {
      m_selectionAllowed = false;
      m_mouseNormPosition = event.getNormalizedPosition(vpRegion);
      beginOrbit(action);
    }
    action.setHandled();
  }

  protected void doDolly(SoEvent event, SoHandleEventAction action)
  {
    startInteraction();

    SbViewportRegion vpRegion = action.getViewportRegion();
    SbVec2f newLocator = event.getNormalizedPosition(vpRegion);
    float dollyScaleFactor = (float) Math.pow(2, 10 * (newLocator.getY() - m_mouseNormPosition.getY()));
    m_cameraInteractor.dolly(dollyScaleFactor);

    m_cameraInteractor.adjustClippingPlanes(action.getNodeAppliedTo(), vpRegion);
    m_mouseNormPosition = newLocator;
    action.setHandled();
  }

  protected void doDollyWithCenter(SbVec2f center, float scaleFactor, SoHandleEventAction action)
  {
    startInteraction();

    SbViewportRegion vpRegion = action.getViewportRegion();
    m_cameraInteractor.dollyWithZoomCenter(center, scaleFactor, vpRegion);

    m_cameraInteractor.adjustClippingPlanes(action.getNodeAppliedTo(), vpRegion);
    action.setHandled();
  }

  protected void doOrbit(SoEvent event, SoHandleEventAction action)
  {
    startInteraction();

    switch ( m_orbitMethod )
    {
    case TURNTABLE :
      doTurntableOrbit(event, action);
      break;
    case TRACKBALL :
    default:
      SbViewportRegion vpRegion = action.getViewportRegion();
      m_mouseNormPosition = event.getNormalizedPosition(vpRegion);
      m_cameraInteractor.orbit(m_mouseNormPosition);
      break;
    }
    action.setHandled();
  }

  /**
   * Rotates the view keeping the horizon horizontal.
   */
  private void doTurntableOrbit(SoEvent event, SoHandleEventAction action)
  {
    SbViewportRegion vpRegion = action.getViewportRegion();

    // step 1 : Calculate the amount of rotation given the mouse movement.
    // a movement from left to right or top to bottom = 2*PI
    double twoPIperPixel = 2 * Math.PI / Math.min(vpRegion.getViewportSizePixels().getX(), vpRegion.getViewportSizePixels().getY()) * .8;


    SbVec2s newPos = event.getPosition();
    long newEventTime = event.getTime().getMsecValue();

    int xDiff = m_mousePosition.getX() - newPos.getX();
    int yDiff = newPos.getY() - m_mousePosition.getY();

    double xAngle = xDiff * twoPIperPixel;
    double yAngle = yDiff * twoPIperPixel;

    float dt = newEventTime - m_eventTime;
    double distance = Math.sqrt(xDiff * xDiff + yDiff * yDiff);
    // interaction speed: pixels/ms
    double speed = dt == 0 ? 0 : distance / dt;
    // apply constraint level
    double level = speed > MAX_INTERACTION_SPEED ? 0 : 1 - speed / MAX_INTERACTION_SPEED;
    boolean constrained = level < m_turntableConstraintLevel;

    SbMatrix mx = m_cameraInteractor.getCamera().orientation.getValue().getMatrix();
    SbVec3f focalPt = m_cameraInteractor.getFocalPoint();

    xDiff = Math.abs(xDiff);
    yDiff = Math.abs(yDiff);

    // if it's a fast movement, do this rotation only if it's an "X" rotation
    if ( !constrained || xDiff >= yDiff )
    {
      // step 2: Rotate the camera around the pivot point on the up axis.
      SbVec3f rotAxis;
      switch ( m_upAxis )
      {
      case X :
        rotAxis = new SbVec3f(mx.getElement(0, 0), mx.getElement(0, 1), mx.getElement(0, 2));
        break;
      case Z :
        rotAxis = new SbVec3f(mx.getElement(2, 0), mx.getElement(2, 1), mx.getElement(2, 2));
        break;
      case Y :
      default:
        rotAxis = new SbVec3f(mx.getElement(1, 0), mx.getElement(1, 1), mx.getElement(1, 2));
        break;
      }

      m_cameraInteractor.setRotationAxis(rotAxis);
      m_cameraInteractor.setRotationCenter(focalPt);
      m_cameraInteractor.rotate((float) xAngle);
    }
    // if it's a fast movement, do this rotation only if it's an "Y" rotation
    if ( !constrained || yDiff >= xDiff )
    {
      // step 3: Rotate the camera around the pivot point on the second axis.
      SbVec3f horiz = new SbVec3f(1, 0, 0);
      m_cameraInteractor.setRotationAxis(horiz);
      m_cameraInteractor.setRotationCenter(focalPt);
      m_cameraInteractor.rotate((float) yAngle);
    }

    // Update the mouse position for the next rotation
    m_mousePosition = newPos;
    m_eventTime = newEventTime;
  }

  protected void doPan(SoEvent event, SoHandleEventAction action)
  {
    startInteraction();

    SbViewportRegion vpRegion = action.getViewportRegion();
    m_mouseNormPosition = event.getNormalizedPosition(vpRegion);
    m_cameraInteractor.pan(m_mouseNormPosition, vpRegion);
    action.setHandled();
  }

  protected void interactiveCountInc()
  {
    m_interactiveCount++;
    if ( m_interactiveCount == 1 && m_currentAction != null )
    {
      // an interaction starts, set interactive mode in scene manager
      m_currentAction.getSceneManager().setInteractive(true);
      // force a redraw, so graph will see the change in
      // the SoInteractionElement
      m_currentAction.getSceneManager().scheduleRedraw();
    }
  }

  protected void interactiveCountDec()
  {
    if ( m_interactiveCount > 0 )
    {
      m_interactiveCount--;

      if ( m_interactiveCount == 0 && m_currentAction != null )
      {
        // interaction is finished, disable interactive mode in scene manager
        m_currentAction.getSceneManager().setInteractive(false);
        // force a redraw, so graph will see the change in
        // the SoInteractionElement
        m_currentAction.getSceneManager().scheduleRedraw();
      }
    }
  }

  private void startInteraction()
  {
    if ( m_interactionEndSensor.isScheduled() )
    {
      // already interacting, unschedule previous timer
      m_interactionEndSensor.unschedule();
    }
    else
    {
      // interaction starts
      interactiveCountInc();
    }
    // wait 0.5s... if scene is still, consider interaction is finished
    m_interactionEndSensor.setTimeFromNow(new SbTime(INTERACTION_TIME));
    m_interactionEndSensor.schedule();
  }
}
