package com.openinventor.inventor.viewercomponents.nodes;

import com.openinventor.inventor.SbMatrix;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.nodes.SoCamera;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.viewercomponents.SoCameraInteractor;

/**
 * Seek animation.
 * <p>
 * This class allows animating a camera by moving it to a desired seek point.
 * This point is specified by the {@link #setUp(SbVec3f, SbViewportRegion)}
 * method. <br>
 * Duration of the animation and relative distance from the viewpoint to the
 * specified point can be adjusted.
 */
class SeekAnimator extends Animator
{

  /**
   * Duration of the seek animation in seconds (default is 2).
   */
  private float m_duration;

  /**
   * Relative distance from the viewpoint to the seek point (default is 50%).
   */
  private float m_relativeDistance;

  private SoCameraInteractor m_cameraInteractor;
  private SoNode m_sceneGraph;

  private SbViewportRegion m_vpRegion;
  private SbVec3f m_saveCameraPosition;
  private SbRotation m_saveCameraOrientation;
  private SbRotation m_newCameraOrientation;
  private SbVec3f m_newCameraPosition;
  private long m_startTime;

  private final boolean m_isAutoClippingPlanes;

  public SeekAnimator(SoCameraInteractor cameraInteractor, SoNode sceneGraph)
  {
    super();

    m_cameraInteractor = cameraInteractor;
    m_sceneGraph = sceneGraph;

    // set default values
    m_duration = 2;
    m_relativeDistance = 50;

    m_isAutoClippingPlanes = SoPreferences.getBoolean("OIV_SCENE_EXAMINER_AUTO_CLIPPING_PLANES", true);
  }

  /**
   * Set the camera interactor.
   */
  public void setCameraInteractor(SoCameraInteractor cameraInteractor)
  {
    m_cameraInteractor = cameraInteractor;
  }

  /**
   * Set the scene graph.
   */
  public void setSceneGraph(SoNode sceneGraph)
  {
    m_sceneGraph = sceneGraph;
  }

  /**
   * Get the duration of the seek animation in seconds (default is 2).
   */
  public float getDuration()
  {
    return m_duration;
  }

  /**
   * Set the duration of the seek animation in seconds (default is 2).
   *
   * @param duration
   *          duration of the seek animation in seconds
   */
  public void setDuration(float duration)
  {
    m_duration = duration;
  }

  /**
   * Get the relative distance from the viewpoint to the seek point (default is
   * 50%).
   */
  public float getRelativeDistance()
  {
    return m_relativeDistance;
  }

  /**
   * Set the relative distance from the viewpoint to the seek point. The given
   * value must be in range [0, 100%] (default is 50%).
   *
   * @param distance
   *          relative distance in range [0, 100%]
   */
  public void setRelativeDistance(float distance)
  {
    m_relativeDistance = distance;
  }

  /**
   * Set up the seek animation.
   * <p>
   * Given the seek point, this routine will compute the new camera's position
   * and orientation. Then call {@link #start()} to start the animation.
   */
  public void setUp(SbVec3f seekPoint, SbViewportRegion vpRegion)
  {
    m_vpRegion = vpRegion;

    // compute new camera position/orientation
    SbVec3f viewVector = new SbVec3f();
    SbMatrix matrix = new SbMatrix();
    // save camera starting point
    SoCamera camera = m_cameraInteractor.getCamera();
    m_saveCameraPosition = new SbVec3f(camera.position.getValue());
    m_saveCameraOrientation = new SbRotation(camera.orientation.getValue());

    // compute the distance the camera will be from the seek point
    // and update the camera focal distance
    SbVec3f seekVec = seekPoint.minus(camera.position.getValue());
    float distance = seekVec.length() * (m_relativeDistance / 100);
    camera.focalDistance.setValue(distance);

    // get the camera new orientation
    matrix.setRotate(camera.orientation.getValue());
    float[] col = matrix.getColumn(2);
    viewVector.setValue(-col[0], -col[1], -col[2]);
    SbRotation changeOrient = new SbRotation();
    SbVec3f tmpVec = seekPoint.minus(camera.position.getValue());
    changeOrient.setValue(viewVector, tmpVec);
    m_newCameraOrientation = camera.orientation.getValue().times(changeOrient);

    // find the camera final position based on orientation and distance
    matrix.setRotate(m_newCameraOrientation);
    col = matrix.getColumn(2);
    viewVector.setValue(-col[0], -col[1], -col[2]);
    tmpVec = viewVector.times(distance);
    m_newCameraPosition = seekPoint.minus(tmpVec);
  }

  @Override
  public void start()
  {
    // now check if animation sensor needs to be scheduled
    if ( m_duration == 0 )
      // jump to new location, no animation needed
      interpolateSeekAnimation(1);
    else
    {
      // start animation
      super.start();
      m_startTime = System.currentTimeMillis();
    }
  }

  @Override
  protected void animate()
  {
    // get the time difference
    long currentTime = System.currentTimeMillis();
    long millisec = currentTime - m_startTime;

    float t = millisec / (1000.f * m_duration);
    if ( t > 1 )
      t = 1;
    else if ( (1 - t) < 0.0001 )
      t = 1; // this will be the last one..

    // interpolate the animation
    interpolateSeekAnimation(t);

    // stops seek if this was the last interval
    if ( t == 1 )
      stop();

  }

  private void interpolateSeekAnimation(float t)
  {
    // now position the camera according to the animation time

    // use and ease-in ease-out approach
    float cos_t = (float) (0.5f - 0.5f * Math.cos(t * Math.PI));

    // get camera new rotation
    m_cameraInteractor.setOrientation(SbRotation.slerp(m_saveCameraOrientation, m_newCameraOrientation, cos_t));

    // get camera new position
    SbVec3f cameraPos = m_newCameraPosition.minus(m_saveCameraPosition);
    cameraPos.multiply(cos_t);
    cameraPos.add(m_saveCameraPosition);
    m_cameraInteractor.setPosition(cameraPos);

    if ( m_isAutoClippingPlanes )
      m_cameraInteractor.adjustClippingPlanes(m_sceneGraph, m_vpRegion);
  }

}
