package com.openinventor.inventor.viewercomponents.swt.glcanvas.renderareas;

import java.util.List;

import org.eclipse.swt.widgets.Composite;

import com.jogamp.nativewindow.CapabilitiesImmutable;
import com.jogamp.nativewindow.NativeWindowException;
import com.jogamp.opengl.DefaultGLCapabilitiesChooser;
import com.jogamp.opengl.GLAutoDrawable;
import com.jogamp.opengl.GLCapabilities;
import com.jogamp.opengl.GLCapabilitiesChooser;
import com.jogamp.opengl.GLCapabilitiesImmutable;
import com.jogamp.opengl.GLEventListener;
import com.jogamp.opengl.swt.GLCanvas;
import com.openinventor.inventor.SbVec2i32;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoSceneManager;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.devices.SoGLContext;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.viewercomponents.SoRawStereoParameters;
import com.openinventor.inventor.viewercomponents.SoRenderAreaCore;
import com.openinventor.inventor.viewercomponents.swt.Timer;

public class RenderArea extends GLCanvas implements GLEventListener
{

  protected SoRenderAreaCore m_renderAreaCore;
  protected SoNode m_sceneGraph;
  private SoGLRenderAction m_glRenderAction;
  private SoSceneManager.AntialiasingModes m_antialiasingMode;
  private float m_antialiasingQuality;
  private int m_stillAntialiasingDelay;
  private float m_stillAntialiasingQuality;
  private boolean m_stereo;
  private boolean m_customGlRenderAction;

  public RenderArea(Composite parent, int style)
  {
    this(parent, style, null, new OptionalStereoGLCapabilitiesChooser());
  }

  public RenderArea(Composite parent, int style, GLCapabilities caps)
  {
    this(parent, style, caps, new OptionalStereoGLCapabilitiesChooser());
  }

  private RenderArea(final Composite parent, final int style, final GLCapabilitiesImmutable capsReqUser,
      final GLCapabilitiesChooser capsChooser)
  {
    super(parent, style, capsReqUser, capsChooser);

    m_renderAreaCore = null;
    m_sceneGraph = null;
    m_stereo = false;
    m_customGlRenderAction = false;

    // init the system timer with an SWT timer
    SoDB.setSystemTimer(new Timer());

    addGLEventListener(this);

    // define default values
    m_glRenderAction = null;
    m_antialiasingMode = SoSceneManager.AntialiasingModes.NO_ANTIALIASING;
    m_antialiasingQuality = 0;
    m_stillAntialiasingDelay = 0;
    m_stillAntialiasingQuality = 0;
  }

  /**
   * Defines the render action used for rendering.
   */
  public void setGLRenderAction(SoGLRenderAction ra)
  {
    m_glRenderAction = ra;
    m_customGlRenderAction = true;
    if ( m_renderAreaCore != null )
      m_renderAreaCore.setGLRenderAction(ra);
  }

  /**
   * Gets the render action used for rendering.
   */
  public SoGLRenderAction getGLRenderAction()
  {
    if ( m_renderAreaCore != null )
      return m_renderAreaCore.getGLRenderAction();
    else
      return m_glRenderAction;
  }

  /**
   * Gets the antialiasing quality value.
   *
   * @return the antialiasing quality, factor in the range [0.0,1.0].
   *
   */
  public float getAntialiasingQuality()
  {
    if ( m_renderAreaCore == null )
      return m_antialiasingQuality;
    return m_renderAreaCore.getAntialiasingQuality();
  }

  /**
   * Sets the antialiasing quality value.
   * <p>
   * Default value is 0.0. The value 0.0 turns off antialiasing.
   *
   * @param quality
   *          The quality is a factor in the range [0.0,1.0].
   *
   */
  public void setAntialiasingQuality(float quality)
  {
    m_antialiasingQuality = quality;
    if ( m_renderAreaCore != null )
      m_renderAreaCore.setAntialiasingQuality(quality);
  }

  /**
   * Gets the antialiasing mode.
   *
   * @return the antialiasing algorithm
   */
  public SoSceneManager.AntialiasingModes getAntialiasingMode()
  {
    if ( m_renderAreaCore == null )
      return m_antialiasingMode;
    return m_renderAreaCore.getAntialiasingMode();
  }

  /**
   * Sets the antialiasing mode.
   * <p>
   * Default value is NO_ANTIALIASING which turns off antialiasing.
   *
   * @param mode
   *          the antialiasing algorithm.
   */
  public void setAntialiasingMode(SoSceneManager.AntialiasingModes mode)
  {
    m_antialiasingMode = mode;
    if ( m_renderAreaCore != null )
      m_renderAreaCore.setAntialiasingMode(mode);
  }

  /**
   * Gets quality for supersampling when "still" (not interacting).
   *
   * @return the quality for supersampling, factor in the range [0.0,1.0].
   */
  public float getStillSuperSamplingQuality()
  {
    if ( m_renderAreaCore == null )
      return m_antialiasingQuality;
    return m_renderAreaCore.getAntialiasingQuality();
  }

  /**
   * Sets quality for supersampling when "still" (not interacting).
   * <p>
   * When quality is greater than 0, still images will be automatically
   * supersampled. Default value is 0.0. Use the value 0.0 to turn off still
   * supersampling. 0.5 is a typical value.
   *
   * @param quality
   *          The quality is a factor in the range [0.0,1.0].
   */
  public void setStillSuperSamplingQuality(float quality)
  {
    m_stillAntialiasingQuality = quality;
    if ( m_renderAreaCore != null )
      m_renderAreaCore.setStillSuperSamplingQuality(quality);
  }

  /**
   * Gets delay for supersampling when "still" (not interacting).
   *
   * @return the delay for supersampling in milliseconds.
   */
  public int getStillSuperSamplingDelay()
  {
    if ( m_renderAreaCore == null )
      return m_stillAntialiasingDelay;
    return m_renderAreaCore.getStillSuperSamplingDelay();
  }

  /**
   * Sets delay for supersampling when "still" (not interacting).
   * <p>
   * If greater than 0, images will be supersampled after the specified delay.
   * Default value is 0. The delay is in milliseconds.
   *
   * @param delay
   *          The delay for supersampling in milliseconds.
   */
  public void setStillSuperSamplingDelay(int delay)
  {
    m_stillAntialiasingDelay = delay;
    if ( m_renderAreaCore != null )
      m_renderAreaCore.setStillSuperSamplingDelay(delay);
  }

  public void activateStereo(boolean activated)
  {
    m_stereo = activated;

    if ( m_renderAreaCore == null )
      return;

    if ( activated )
    {
      // activate raw stereo
      if ( !isRawStereoAvailable() )
        throw new UnsupportedOperationException("Stereo buffers are not enabled");

      SoRawStereoParameters params = new SoRawStereoParameters();
      m_renderAreaCore.setStereoParameters(params);
      m_renderAreaCore.activateStereo(true);
    }
    else
      m_renderAreaCore.activateStereo(false);
  }

  public boolean isRawStereoAvailable()
  {
    return getChosenGLCapabilities().getStereo();
  }

  public void setSceneGraph(SoNode sceneGraph)
  {
    m_sceneGraph = sceneGraph;

    if ( m_renderAreaCore != null )
      m_renderAreaCore.setSceneGraph(sceneGraph);
  }

  /**
   * Schedules a redraw for some time in the near future.
   */
  public void scheduleRedraw()
  {
    if ( m_renderAreaCore != null )
      m_renderAreaCore.getSceneManager().scheduleRedraw();
  }

  protected void render()
  {
    m_renderAreaCore.render();
  }

  @Override
  public void init(GLAutoDrawable drawable)
  {
    // let OpenInventor be aware of the current context created by JOGL
    SoGLContext context = SoGLContext.getCurrent(true);
    m_renderAreaCore = new SoRenderAreaCore(context);
    m_renderAreaCore.setSceneGraph(m_sceneGraph);
    if ( m_glRenderAction == null )
      m_glRenderAction = m_renderAreaCore.getGLRenderAction();
    else
      m_renderAreaCore.setGLRenderAction(m_glRenderAction);
    m_renderAreaCore.setAntialiasingMode(m_antialiasingMode);
    m_renderAreaCore.setAntialiasingQuality(m_antialiasingQuality);
    m_renderAreaCore.setStillSuperSamplingDelay(m_stillAntialiasingDelay);
    m_renderAreaCore.setStillSuperSamplingQuality(m_stillAntialiasingQuality);
    activateStereo(m_stereo);
  }

  @Override
  public void dispose(GLAutoDrawable drawable)
  {
    if ( !m_customGlRenderAction )
      m_glRenderAction = null;
    m_renderAreaCore.dispose();
  }

  @Override
  public void display(GLAutoDrawable drawable)
  {
    // render Open Inventor scene graph
    render();
  }

  @Override
  public void reshape(GLAutoDrawable drawable, int x, int y, int width, int height)
  {
    m_renderAreaCore.setSize(new SbVec2i32(width, height));
  }
}

/**
 * Custom implementation of GLCapabilitiesChooser.<br>
 * The DefaultGLCapabilitiesChooser does not allow to define an optional but not
 * mandatory GL capability. In particular, stereo capability is considered as a
 * mandatory capability.<br>
 * This implementation allows selecting an optional stereo capability: if no
 * stereo capability is available, it continues to search other requested
 * (mandatory) capabilities.
 */
class OptionalStereoGLCapabilitiesChooser extends DefaultGLCapabilitiesChooser
{
  @Override
  public int chooseCapabilities(CapabilitiesImmutable desired, List<? extends CapabilitiesImmutable> available,
      int windowSystemRecommendedChoice)
  {
    int chosen = -1;
    try
    {
      // launch default chooser
      chosen = super.chooseCapabilities(desired, available, windowSystemRecommendedChoice);
    }
    catch (NativeWindowException exc)
    {
      // chosen index is not valid, check if stereo capability has been
      // requested
      final GLCapabilities gldes = (GLCapabilities) desired.cloneMutable();
      if ( gldes.getStereo() )
      {
        // stereo capability has been requested but it is not available.
        System.err.println("Could not enable stereo buffers");

        // try to get the best config with no stereo
        // if we don't do that, the first available config will be chosen
        // without paying attention to others requested GL capabilities.
        gldes.setStereo(false);
        chosen = super.chooseCapabilities(gldes, available, windowSystemRecommendedChoice);
      }
      else
        // let JOGL do its job...
        throw exc;
    }

    return chosen;
  }
}
