package com.openinventor.inventor.viewercomponents.swt.tools;

import java.text.DecimalFormat;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;

import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.NavigationMode;
import com.openinventor.inventor.viewercomponents.swt.glcanvas.renderareas.RenderAreaInteractive.ClippingPlanesAdjustMode;

public class PreferencesDialog extends Dialog
{

  public static interface StereoListener
  {
    public void activateRawStereo(boolean activated);

    public void balanceValueChanged(float value);

    public void offsetValueChanged(float value);
  }

  public static interface GeneralPreferencesListener
  {
    public void navigationModeChanged(NavigationMode newMode);

    public void clippingModeChanged(ClippingPlanesAdjustMode newMode);
  }

  private static PreferencesDialog m_instance = null;

  private Shell shell;
  private StereoListener stereoListener;
  private GeneralPreferencesListener generalPreferencesListener;

  private Button btnActivateRawStereo;
  private Button balanceCheckBox;
  private Slider balanceSlider;
  private Label balanceValueLabel;
  private Button offsetCheckBox;
  private Slider offsetSlider;
  private Label offsetValueLabel;
  private Button orbitButton;
  private Button planeButton;
  private Label clippingModeLabel;
  private Combo clippingPlanesCombo;
  private Label lblNavigationMode;

  /**
   * Create the dialog.
   *
   * @param parent
   * @param style
   */
  private PreferencesDialog(Shell parent)
  {
    super(parent, SWT.NONE);
    setText("Preferences");

    stereoListener = null;
    generalPreferencesListener = null;
    createContents();
  }

  public static PreferencesDialog getInstance(Shell parent)
  {
    if ( m_instance == null )
      m_instance = new PreferencesDialog(parent);

    return m_instance;
  }

  public void setStereoListener(StereoListener listener)
  {
    stereoListener = listener;
  }

  public StereoListener getStereoListener()
  {
    return stereoListener;
  }

  public void setGeneralPreferencesListener(GeneralPreferencesListener listener)
  {
    generalPreferencesListener = listener;
  }

  public GeneralPreferencesListener getGeneralPreferencesListener()
  {
    return generalPreferencesListener;
  }

  public void enableStereoPreferences(boolean enabled)
  {
    if ( !enabled && btnActivateRawStereo.getSelection() )
    {
      // deactivate raw stereo
      if ( stereoListener != null )
        stereoListener.activateRawStereo(false);
      btnActivateRawStereo.setSelection(false);
    }
    btnActivateRawStereo.setEnabled(enabled);
    balanceCheckBox.setEnabled(enabled);
    balanceSlider.setEnabled(enabled);
    balanceValueLabel.setEnabled(enabled);
    offsetCheckBox.setEnabled(enabled);
    offsetSlider.setEnabled(enabled);
    offsetValueLabel.setEnabled(enabled);
  }

  public void setNavigationMode(NavigationMode mode)
  {
    switch ( mode )
    {
    case ORBIT :
      orbitButton.setSelection(true);
      break;
    case PLANE :
      planeButton.setSelection(true);
      break;
    default:
      break;
    }
  }

  public void setClippingPlanesAdjustMode(ClippingPlanesAdjustMode mode)
  {
    clippingPlanesCombo.select(mode.ordinal());
  }

  /**
   * Open the dialog.
   *
   */
  public void open()
  {
    shell.open();
    shell.layout();
    Display display = getParent().getDisplay();
    while ( !shell.isDisposed() )
    {
      if ( !display.readAndDispatch() )
      {
        display.sleep();
      }
    }
  }

  /**
   * Create contents of the dialog.
   */
  private void createContents()
  {
    shell = new Shell(getParent(), SWT.SHELL_TRIM);
    shell.addListener(SWT.Close, new Listener()
    {

      @Override
      public void handleEvent(Event event)
      {
        // just hide the dialog
        shell.setVisible(false);
        event.doit = false;
      }
    });
    shell.setImage(SWTResourceManager.getImage(PreferencesDialog.class,
        "/com/openinventor/inventor/viewercomponents/icons/preferencesMenu.png"));
    shell.setSize(450, 200);
    shell.setText(getText());
    shell.setLayout(new FillLayout(SWT.HORIZONTAL));

    TabFolder tabFolder = new TabFolder(shell, SWT.NONE);

    TabItem tbtmGeneral = new TabItem(tabFolder, SWT.NONE);
    tbtmGeneral.setText("General");

    Composite navigationComposite = new Composite(tabFolder, SWT.NONE);
    tbtmGeneral.setControl(navigationComposite);
    navigationComposite.setLayout(new GridLayout(3, false));

    lblNavigationMode = new Label(navigationComposite, SWT.NONE);
    lblNavigationMode.setText("Navigation mode");

    orbitButton = new Button(navigationComposite, SWT.RADIO);
    orbitButton.setText("Orbit");
    orbitButton.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent e)
      {
        if ( generalPreferencesListener != null && orbitButton.getSelection() )
          generalPreferencesListener.navigationModeChanged(NavigationMode.ORBIT);
      }
    });

    planeButton = new Button(navigationComposite, SWT.RADIO);
    planeButton.setText("Plane");
    planeButton.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent e)
      {
        if ( generalPreferencesListener != null && planeButton.getSelection() )
          generalPreferencesListener.navigationModeChanged(NavigationMode.PLANE);
      }
    });

    clippingModeLabel = new Label(navigationComposite, SWT.CHECK);
    clippingModeLabel.setText("Clipping planes mode");

    final ClippingPlanesAdjustMode[] clippingModes = ClippingPlanesAdjustMode.values();
    String[] clippingItems = new String[clippingModes.length];
    for ( int i = 0; i < clippingModes.length; i++ )
    {
      clippingItems[i] = clippingModes[i].toString();
    }
    clippingPlanesCombo = new Combo(navigationComposite, SWT.READ_ONLY);
    clippingPlanesCombo.setItems(clippingItems);
    clippingPlanesCombo.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false, 2, 1));
    clippingPlanesCombo.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent e)
      {
        if ( generalPreferencesListener != null )
          generalPreferencesListener
              .clippingModeChanged(ClippingPlanesAdjustMode.valueOf(clippingPlanesCombo.getText()));
      }
    });

    TabItem tbtmStereo = new TabItem(tabFolder, SWT.NONE);
    tbtmStereo.setImage(SWTResourceManager.getImage(PreferencesDialog.class,
        "/com/openinventor/inventor/viewercomponents/icons/ViewerStereo24.png"));
    tbtmStereo.setText("Stereo");

    Composite stereoComposite = new Composite(tabFolder, SWT.NONE);
    tbtmStereo.setControl(stereoComposite);
    stereoComposite.setLayout(new GridLayout(1, false));

    btnActivateRawStereo = new Button(stereoComposite, SWT.CHECK);
    btnActivateRawStereo.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent e)
      {
        if ( stereoListener != null )
          stereoListener.activateRawStereo(btnActivateRawStereo.getSelection());
      }
    });
    btnActivateRawStereo.setText("Activate raw stereo");

    Group grpStereoAdjustment = new Group(stereoComposite, SWT.NONE);
    grpStereoAdjustment.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 1, 1));
    grpStereoAdjustment.setText("Stereo Adjustment");
    grpStereoAdjustment.setLayout(new GridLayout(4, false));

    Label lblZeroParallaxBalance = new Label(grpStereoAdjustment, SWT.NONE);
    lblZeroParallaxBalance.setText("Zero Parallax Balance:");

    balanceCheckBox = new Button(grpStereoAdjustment, SWT.CHECK);

    float defaultBalanceValue = 1.f;
    balanceSlider = new Slider(grpStereoAdjustment, SWT.NONE);
    balanceSlider.setMinimum(-300);
    balanceSlider.setMaximum(300);
    balanceSlider.setSelection((int) defaultBalanceValue * 100);
    balanceSlider.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 1, 1));

    balanceValueLabel = new Label(grpStereoAdjustment, SWT.BORDER);
    GridData gd_balanceValueLabel = new GridData(SWT.CENTER, SWT.CENTER, false, false, 1, 1);
    gd_balanceValueLabel.widthHint = 30;
    balanceValueLabel.setLayoutData(gd_balanceValueLabel);
    balanceValueLabel.setText(DecimalFormat.getInstance().format(defaultBalanceValue));

    balanceSlider.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent e)
      {
        if ( balanceCheckBox.getSelection() || e.detail != SWT.DRAG )
        {
          float value = balanceSlider.getSelection();
          value /= 100.f;
          balanceValueLabel.setText(DecimalFormat.getInstance().format(value));
          if ( stereoListener != null )
            stereoListener.balanceValueChanged(value);
        }
      }
    });

    Label lblCameraOffset = new Label(grpStereoAdjustment, SWT.NONE);
    lblCameraOffset.setText("Camera Offset:");

    offsetCheckBox = new Button(grpStereoAdjustment, SWT.CHECK);

    float defaultOffsetValue = 0.7f;
    offsetSlider = new Slider(grpStereoAdjustment, SWT.NONE);
    offsetSlider.setMinimum(0);
    offsetSlider.setMaximum(200);
    offsetSlider.setSelection((int) defaultOffsetValue * 100);
    offsetSlider.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 1, 1));

    offsetValueLabel = new Label(grpStereoAdjustment, SWT.BORDER);
    GridData gd_offsetValueLabel = new GridData(SWT.CENTER, SWT.CENTER, false, false, 1, 1);
    gd_offsetValueLabel.widthHint = 30;
    offsetValueLabel.setLayoutData(gd_offsetValueLabel);
    offsetValueLabel.setText(DecimalFormat.getInstance().format(defaultOffsetValue));

    offsetSlider.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent e)
      {
        if ( offsetCheckBox.getSelection() || e.detail != SWT.DRAG )
        {
          float value = offsetSlider.getSelection();
          value /= 100.f;
          offsetValueLabel.setText(DecimalFormat.getInstance().format(value));
          if ( stereoListener != null )
            stereoListener.offsetValueChanged(value);
        }
      }
    });
  }
}
