///////////////////////////////////////////////////////////////////////////////
//
// This code is part of the Open Inventor Medical Edition utility library.
//
// Open Inventor customers may use this source code to create or enhance Open
// Inventor-based applications.
//
// The medical utility classes are provided as a jar named com.openinventor.medical.jar,
// that can be used directly in an Open Inventor application. The classes in this jar
// are documented and supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
///////////////////////////////////////////////////////////////////////////////
package com.openinventor.medical.nodes;

import com.openinventor.inventor.fields.SoField;
import com.openinventor.inventor.fields.SoSFFilePathString;
import com.openinventor.volumeviz.readers.SoVRDicomData;

//@formatter:off
/**
 * <strong>(Preview Feature) </strong>Text node to display DICOM information on screen.
 *
 * <p style="background-color: #c7c7c7;">Preview Feature means this class is fully supported and can be used in Open Inventor applications.
 * Being tagged as a Preview Feature just means that the implementation is still
 * subject to API changes and adjustments based on feedback from early adopters.
 * Please be also aware that source compatibility might be broken regardless of
 * the Open Inventor compatibility changes policy due to our commitment to bring
 * needed changes to be sure the specifications of this Preview Feature match
 * the expectations of our customers.
 * </p>
 *
 * This node is a convenience class derived from TextBox for displaying
 * DICOM annotation text on screen.
 * <p>
 * This node displays a text box at a fixed location in the viewer window.
 * The position is specified in normalized device coordinates -1 to 1.
 * The DICOM file is specified using the 'fileName' field.
 * <p>
 * Alignment options allow the box to be positioned, for example, in a
 * corner of the window. For example position -0.98, -0.98, 0 with default
 * alignment is the upper left corner of the window.
 * The text box automatically expands or shrinks when lines of text are
 * added to or deleted from the box.  Positioning a text box in the lower
 * left corner creates a sort of 'console' output overlaying the scene.
 * Note that the alignment options control the positioning of the text box.
 * The text inside the box is currently always left justified.
 * <p>
 * In order to have convenient default values for font size, the font
 * settings are <i>not</i> inherited from the scene graph. By default the text
 * is rendered using an SoText2 node with font name 'Arial', font size 16 and
 * line spacing 1.1. The application can modify these values using the
 * getFontNode() and getTextNode() methods. Lighting and picking are disabled.
 * Text will be rendered on top of whatever is rendered in the main scene graph.
 * The application can modify the text strings directly, but this class also
 * provides some convenience methods that are very useful. For example, the
 * addLine() method appends a new string to the end of the list.
 * <p>
 * <b>File format/default:</b>
 * <p>
 * DicomInfo {
 * <ul><table>
 * <tr><td> fileName       </td><td> ""            </td></tr>
 * <tr><td> position       </td><td> 0 0 0         </td></tr>
 * <tr><td> alignmentH     </td><td> LEFT          </td></tr>
 * <tr><td> alignmentV     </td><td> TOP           </td></tr>
 * <tr><td> textAlignH     </td><td> LEFT          </td></tr>
 * <tr><td> fontName       </td><td> Arial:Bold    </td></tr>
 * <tr><td> fontSize       </td><td> 15            </td></tr>
 * <tr><td> border         </td><td> FALSE         </td></tr>
 * <tr><td> borderColor    </td><td> 1 1 1         </td></tr>
 * </table></ul> }
 *
 * @see TextBox
 * @see Gnomon
 * @see Magnifier
 * @see Ruler
 *
 */
//@formatter:on
public class DicomInfo extends TextBox
{

  /**
   * Path to the DICOM file used to extract DICOM info. Default is empty.
   */
  public SoSFFilePathString fileName;

  /**
   * Add a line to the DICOM annotation.
   *
   * @param title
   *          A string to prefix the DICOM info
   * @param group
   *          Group tag of requested DICOM info
   * @param element
   *          Element tag of requested DICOM info
   * @return Index of the next insertion
   */
  public int displayDicomInfo(String title, short group, short element)
  {
    checkFilename();
    String str = _dicomReader.getDicomInfo(group, element);
    // Check for dates
    // TODO: Need to get access to the DICOM "value representation" code
    boolean isDate = false;
    if ( group == 0x08 )
    {
      if ( element == 0x12 || (element >= 0x20 && element <= 0x23) )
        isDate = true;
    }
    else if ( group == 0x10 && element == 0x30 )
      isDate = true;

    // Date info is available
    if ( isDate && str != null && str.length() > 0 )
    {
      String info = new String(str.substring(0, 4) + " - " + str.substring(4, 6) + " - " + str.substring(6, 8));
      str = info;
    }

    addLine(title + str);
    return (getNumLines() - 1);
  }

  /**
   * Constructor
   */
  public DicomInfo()
  {
    setName("DicomInfo");
    fileName = new SoSFFilePathString(this, "FileName", SoField.FieldTypes.EXPOSED_FIELD, "");
    _filename = "";

    // DICOM reader
    _dicomReader = new SoVRDicomData();
  }

  protected String _filename;
  protected SoVRDicomData _dicomReader;

  /**
   * Check that we have data from the correct file.
   */
  protected boolean checkFilename()
  {
    boolean rc = true;
    if ( !_filename.equals(fileName.getValue()) )
    {
      _filename = fileName.getValue();
      rc = false;
      if ( _filename.length() > 0 )
        rc = _dicomReader.readDicomHeader(_filename);
    }
    return rc;
  }

}
