///////////////////////////////////////////////////////////////////////////////
//
// This code is part of the Open Inventor Medical Edition utility library.
//
// Open Inventor customers may use this source code to create or enhance Open
// Inventor-based applications.
//
// The medical utility classes are provided as a jar named com.openinventor.medical.jar,
// that can be used directly in an Open Inventor application. The classes in this jar
// are documented and supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
///////////////////////////////////////////////////////////////////////////////
package com.openinventor.medical.nodes;

import java.util.Arrays;

import com.openinventor.inventor.IntegerValuedEnum;
import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.actions.SoGetBoundingBoxAction;
import com.openinventor.inventor.actions.SoHandleEventAction;
import com.openinventor.inventor.actions.SoSearchAction;
import com.openinventor.inventor.elements.SoViewportRegionElement;
import com.openinventor.inventor.fields.SoField;
import com.openinventor.inventor.fields.SoSFBool;
import com.openinventor.inventor.fields.SoSFColor;
import com.openinventor.inventor.fields.SoSFEnum;
import com.openinventor.inventor.fields.SoSFFloat;
import com.openinventor.inventor.fields.SoSFString;
import com.openinventor.inventor.fields.SoSFVec3f;
import com.openinventor.inventor.misc.SoState;
import com.openinventor.inventor.nodes.SoAnnotation;
import com.openinventor.inventor.nodes.SoBBox;
import com.openinventor.inventor.nodes.SoCallback;
import com.openinventor.inventor.nodes.SoCamera;
import com.openinventor.inventor.nodes.SoFont;
import com.openinventor.inventor.nodes.SoLightModel;
import com.openinventor.inventor.nodes.SoLineSet;
import com.openinventor.inventor.nodes.SoOrthographicCamera;
import com.openinventor.inventor.nodes.SoPickStyle;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.nodes.SoText2;
import com.openinventor.inventor.nodes.SoTextProperty;
import com.openinventor.inventor.nodes.SoTranslation;
import com.openinventor.inventor.nodes.SoVertexProperty;
import com.openinventor.inventor.sensors.SoNodeSensor;

//@formatter:off
/**
 * <strong>(Preview Feature) </strong>Shape node to display a text box in window coordinates.
 *
 * <p style="background-color: #c7c7c7;">Preview Feature means this class is fully supported and can be used in Open Inventor applications.
 * Being tagged as a Preview Feature just means that the implementation is still
 * subject to API changes and adjustments based on feedback from early adopters.
 * Please be also aware that source compatibility might be broken regardless of
 * the Open Inventor compatibility changes policy due to our commitment to bring
 * needed changes to be sure the specifications of this Preview Feature match
 * the expectations of our customers.
 * </p>
 *
 * <p>
 * This node displays a text box at a fixed location in the viewer window.
 * <p>
 * The {@link #position} is specified in normalized device coordinates -1 to 1.
 * Horizontal {@link #alignmentH} and vertical {@link #alignmentV} can be specified. For example
 * position -0.98, -0.98, 0 with TOP/LEFT alignment (the default) puts a text
 * box in the upper left corner of the window. The text box automatically
 * expands or shrinks when lines of text are added to or deleted from the box.
 * Positioning a text box in the lower left corner creates a sort of 'console'
 * output overlaying the scene. Note that the alignment options control the
 * positioning of the text box. The text can be left, center or right justified
 * inside the box using the {@link #alignmentH} field.
 * <p>
 * In order to have convenient default values for the font properties, by
 * default the font properties are <i>not</i> inherited from the scene graph. By
 * default the text is rendered using an SoText2 node with font name
 * 'Arial:Bold', font size 15 and line spacing 1.1. The application can modify
 * the {@link #fontName} and {@link #fontSize} fields or modify text properties directly using
 * the getFontNode() and getTextNode() methods.
 * <p>
 * Lighting and picking are disabled. A {@link #border} can be drawn around the box.
 * Text will be rendered on top of whatever is rendered in the main scene graph.
 * The application can modify the text strings directly, but this class also
 * provides some convenience methods that are very useful. For example, the
 * addLine() method appends a new string to the end of the list.
 * <p>
 * The bounding box of this node is ignored, i.e. it does not contribute to an
 * SoGetBoundingBoxAction traversal and it does not affect a "viewAll" call on
 * the camera or viewer.
 *
 * <p>
 * <b>File format/default:</b>
 * <p>
 * TextBox {
 * <ul><table>
 * <tr><td> position          </td><td> 0 0 0        </td></tr>
 * <tr><td> alignmentH        </td><td> LEFT         </td></tr>
 * <tr><td> alignmentV        </td><td> TOP          </td></tr>
 * <tr><td> textAlignH        </td><td> LEFT         </td></tr>
 * <tr><td> fontName          </td><td> Arial:Bold   </td></tr>
 * <tr><td> fontSize          </td><td> 15           </td></tr>
 * <tr><td> border            </td><td> FALSE        </td></tr>
 * <tr><td> borderColor       </td><td> 1 1 1        </td></tr>
 * </table></ul> }
 * <p>
 * @see DicomInfo
 * @see Magnifier
 * @see Gnomon
 * @see Ruler
 *
 */
//@formatter:on
public class TextBox extends SoAnnotation
{

  /**
   * Horizontal alignment values.
   */
  public enum AlignmentH implements IntegerValuedEnum
  {
    /** Left edge */
    LEFT(SoTextProperty.AlignmentHs.LEFT.getValue()),
    /** Center */
    CENTER(SoTextProperty.AlignmentHs.CENTER.getValue()),
    /** Right edge */
    RIGHT(SoTextProperty.AlignmentHs.RIGHT.getValue());

    private AlignmentH(int ali)
    {
      _alignmentH = ali;
    }

    @Override
    public int getValue()
    {
      return _alignmentH;
    }

    private final int _alignmentH;
  };

  /**
   * Vertical alignment values.
   */
  public enum AlignmentV implements IntegerValuedEnum
  {
    /** Top edge */
    TOP(SoTextProperty.AlignmentVs.TOP.getValue()),
    /** Middle */
    MIDDLE(SoTextProperty.AlignmentVs.HALF.getValue()),
    /** Bottom edge */
    BOTTOM(SoTextProperty.AlignmentVs.BOTTOM.getValue());

    private AlignmentV(int ali)
    {
      _alignmentV = ali;
    }

    @Override
    public int getValue()
    {
      return _alignmentV;
    }

    private final int _alignmentV;
  };

  /**
   * Position of the text box in normalized screen coordinates (-1 to 1).
   * Default is 0,0,0.
   */
  public SoSFVec3f position;

  /**
   * Horizontal alignment of the text box (default is LEFT).
   */
  public SoSFEnum<AlignmentH> alignmentH;

  /**
   * Vertical alignment of the text box (default is TOP).
   */
  public SoSFEnum<AlignmentV> alignmentV;

  /**
   * Horizontal alignment of the text inside the box (default is LEFT).
   */
  public SoSFEnum<AlignmentH> textAlignH;

  /**
   * Specify the font name (default is "Arial:Bold"). See SoFont.name for
   * details
   */
  public SoSFString fontName;

  /**
   * Specify the font size in pixels (default is 15).
   */
  public SoSFFloat fontSize;

  /**
   * Enable drawing a border around the text box (default is false).
   */
  public SoSFBool border;

  /**
   * Border color (default is 1,1,1).
   */
  public SoSFColor borderColor;

  /**
   * Set the contents of the specified line of text (convenience method).
   */
  public void setLine(String text, int line)
  {
    if ( line < 0 )
      return;
    if ( text.length() <= 0 )
      _textNode.string.set1Value(line, "");
    else
      _textNode.string.set1Value(line, text);
  }

  /**
   * Add a line of text at the bottom of the box.
   */
  public void addLine(String text)
  {
    // Note: SoText2 may crash if one of the string is 'empty'
    // (which is different from having 0 characters).
    int n = _textNode.string.getNum();
    if ( text.length() <= 0 )
      _textNode.string.set1Value(n, "");
    else
      _textNode.string.set1Value(n, text);
  }

  /**
   * Get number of lines of text currently in the box
   */
  public int getNumLines()
  {
    return _textNode.string.getNum();
  }

  /**
   * Constructor
   */
  public TextBox()
  {
    position = new SoSFVec3f(this, "position", SoField.FieldTypes.EXPOSED_FIELD, new SbVec3f(0.0f, 0.0f, 0.0f));
    fontName = new SoSFString(this, "fontName", SoField.FieldTypes.EXPOSED_FIELD, "Arial:Bold");
    alignmentH = new SoSFEnum<AlignmentH>(this, "alignmentH", SoField.FieldTypes.EXPOSED_FIELD, AlignmentH.class,
        TextBox.AlignmentH.LEFT);
    alignmentV = new SoSFEnum<AlignmentV>(this, "alignmentV", SoField.FieldTypes.EXPOSED_FIELD, AlignmentV.class,
        TextBox.AlignmentV.TOP);
    textAlignH = new SoSFEnum<AlignmentH>(this, "textAlignH", SoField.FieldTypes.EXPOSED_FIELD, AlignmentH.class,
        TextBox.AlignmentH.LEFT);
    fontSize = new SoSFFloat(this, "fontSize", SoField.FieldTypes.EXPOSED_FIELD, 15.0f);
    border = new SoSFBool(this, "border", SoField.FieldTypes.EXPOSED_FIELD, false);
    borderColor = new SoSFColor(this, "borderColor", SoField.FieldTypes.EXPOSED_FIELD, new SbColor(1.0f, 1.0f, 1.0f));

    _isModified = true; // Update on next traversal
    // Initial viewport is unknown (get this during traversal)
    _curViewport = new SbViewportRegion((short) 0, (short) 0);
    _curBBox = new SbBox3f(); // ditto

    setName("Tooltip");

    buildSceneGraph();

    // Detect changes to our fields or children
    _nodeSensor = new SoNodeSensor(new Runnable()
    {
      @Override
      public void run()
      {
        _isModified = true;
        enableNotify(false);

        // Border
        int which = border.getValue() ? 0 : -1;
        if ( _borderSwitch.whichChild.getValue() != which )
          _borderSwitch.whichChild.setValue(which);

        // Border Color
        int packedColor = borderColor.getValue().getPackedValue();
        SoVertexProperty vprop = (SoVertexProperty) _borderGeom.vertexProperty.getValue();
        if ( vprop.orderedRGBA.getValueAt(0) != packedColor )
          vprop.orderedRGBA.set1Value(0, packedColor);

        enableNotify(true);
      }
    });

    _nodeSensor.setPriority(0);
    _nodeSensor.attach(this);
  }

  /**
   * Builds the scene graph to represent text box object.
   */
  protected void buildSceneGraph()
  {
    // --------------------------------------------------------------------
    // Build internal scene graph
    // Callback allows us to update state at traversal time
    SoCallback nodeCB = new SoCallback();
    nodeCB.setCallback(new SoCallback.CB()
    {
      @Override
      public void invoke(com.openinventor.inventor.actions.SoAction action)
      {
        // Ignore actions that (probably) don't require adjusting position.
        if ( action instanceof SoHandleEventAction || action instanceof SoSearchAction
            || action instanceof SoGetBoundingBoxAction )
          return;

        // We have to check the viewport because the bounding box of an SoText2
        // depends on the current window size.
        // Changes to actual fields should be handled by the node sensor.
        SoState state = action.getState();
        SbViewportRegion vport = SoViewportRegionElement.get(state);
        if ( !_isModified && vport.equals(_curViewport) )
          return;

        // Something changed...
        _curViewport = vport;
        updatePosAndGeom(state);
      }
    });

    this.addChild(nodeCB);

    // Set to NO_BOUNDING_BOX to prevent text box geometry from affecting
    // the bounding box of the application's scene.
    SoBBox bboxNode = new SoBBox();
    bboxNode.mode.setValue(SoBBox.Modes.NO_BOUNDING_BOX);
    this.addChild(bboxNode);

    SoLightModel lmodel = new SoLightModel();
    lmodel.model.setValue(SoLightModel.Models.BASE_COLOR);
    this.addChild(lmodel);

    SoPickStyle pstyle = new SoPickStyle();
    pstyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
    this.addChild(pstyle);

    SoOrthographicCamera camera = new SoOrthographicCamera();
    camera.viewportMapping.setValue(SoCamera.ViewportMappings.LEAVE_ALONE);
    this.addChild(camera);

    _textSep = new SoSeparator();
    this.addChild(_textSep);

    // Note we can't simply connect from the position field.
    // Justification for multi-string text nodes doesn't work right (#43882),
    // so we have to adjust the text position to "fake it".
    // However translation and position are the same for default LEFT/TOP
    // justification.
    _tranNode = new SoTranslation();
    _tranNode.translation.setValue(position.getValue());
    _textSep.addChild(_tranNode);

    // Since OIV doesn't handle alignment of text blocks (multi-string nodes)
    // the way we want, force a predictable mode, so we can work around it.
    _prop = new SoTextProperty();
    _prop.alignmentH.connectFrom(textAlignH);
    _prop.alignmentV.setValue(SoTextProperty.AlignmentVs.TOP);
    _textSep.addChild(_prop);

    // Switch to control whether font properties are inherited or not.
    // TODO: How to expose this?
    _fontSwitch = new SoSwitch();
    _fontSwitch.whichChild.setValue(0);
    _textSep.addChild(_fontSwitch);

    _fontNode = new SoFont();
    _fontNode.size.connectFrom(this.fontSize);
    _fontNode.name.connectFrom(this.fontName);
    _fontNode.renderStyle.setValue(SoFont.RenderStyles.TEXTURE);
    _fontSwitch.addChild(_fontNode);

    _textNode = new SoText2();
    _textNode.string.deleteValues(0);
    _textNode.spacing.setValue(1.1f);
    _textNode.justification.setValue(SoText2.Justifications.INHERITED);
    _textSep.addChild(_textNode);

    // We don't want notification because this node will be modified during
    // traversal.
    // Do NOT disable notification on the children or changes to the border
    // geometry won't have any visible effect (render cache will not be broken).
    SoSeparator geomSep = new SoSeparator();
    this.addChild(geomSep);

    // Switch to control visibility of the border.
    _borderSwitch = new SoSwitch();
    _borderSwitch.whichChild.setValue(border.getValue() ? 0 : -1);
    geomSep.addChild(_borderSwitch);

    SoVertexProperty vprop = new SoVertexProperty();
    int packedColor = borderColor.getValue().getPackedValue();
    vprop.orderedRGBA.set1Value(0, packedColor);

    _borderGeom = new SoLineSet();
    _borderGeom.vertexProperty.setValue(vprop);
    _borderSwitch.addChild(_borderGeom);
  }

  @Override
  public RenderModes getRenderEngineMode()
  {
    return RenderModes.OIV_OPENINVENTOR_RENDERING;
  }

  protected SoFont _fontNode;
  protected SoLineSet _borderGeom;
  protected SoSwitch _fontSwitch;
  protected SoSwitch _borderSwitch;
  protected SoText2 _textNode;
  protected SoTranslation _tranNode;
  protected SoSeparator _textSep;

  protected boolean _isModified;
  protected SbViewportRegion _curViewport;
  protected SbBox3f _curBBox;

  protected SoNodeSensor _nodeSensor;
  protected SoTextProperty _prop;

  protected void updatePosAndGeom(SoState state)
  {
    // We may be called during traversal with a valid SoState
    // That normally indicates the viewport has changed.
    // We may be called from a node/field sensor with no SoState.
    // That normally indicates a string has been added or deleted.
    // In this case we just use the last viewport we saved.
    //
    // Note we have to adjust the actual position of the text box to simulate
    // some alignment/justification modes. OIV does not currently justify a
    // multi-string text node as a "block" of text.
    SbViewportRegion vport = _curViewport;
    if ( state != null )
      vport = SoViewportRegionElement.get(state);

    // Update bounding box.
    // Note1: This node has "probably" been added to a group node, but... we
    // could arrive here through the node sensor without ever ref'ing it. So
    // avoid crashing.
    // Note2: SoText2 considers clipping at the window edge when it computes its
    // bounding box. Sigh. So we need to temporarily move the string to a "safe"
    // location while we compute the actual bounding. (Even this will break if
    // there are too many strings but that doesn't seem like a useful use case.)
    // Remember to back this out before updating border geometry...
    SbVec3f SAFE_POS = new SbVec3f(-0.99f, 0.99f, 0.0f);

    boolean saveNotTrans = _tranNode.enableNotify(false);
    SbVec3f oldPos = _tranNode.translation.getValue();
    _tranNode.translation.setValue(SAFE_POS);

    SoGetBoundingBoxAction gbba = new SoGetBoundingBoxAction(vport);
    gbba.apply(_textSep);
    SbBox3f bbox = gbba.getBoundingBox();

    // Update alignment
    SbVec3f pos = position.getValue(); // App specified text box position
    SbVec3f size = bbox.getSize();
    SbVec3f delta = new SbVec3f(0.0f, 0.0f, 0.0f);

    // Horizontal alignment adjustment
    if ( alignmentH.getValue() == AlignmentH.LEFT.getValue() )
    {
      if ( textAlignH.getValue() == AlignmentH.CENTER.getValue() )
        delta.setX(0.5f * size.getX());
      else if ( textAlignH.getValue() == AlignmentH.RIGHT.getValue() )
        delta.setX(size.getX());
    }
    else if ( alignmentH.getValue() == AlignmentH.CENTER.getValue() )
    {
      if ( textAlignH.getValue() == AlignmentH.LEFT.getValue() )
        delta.setX(-0.5f * size.getX());
      else if ( textAlignH.getValue() == AlignmentH.RIGHT.getValue() )
        delta.setX(0.5f * size.getX());
    }
    else if ( alignmentH.getValue() == AlignmentH.RIGHT.getValue() )
    {
      if ( textAlignH.getValue() == AlignmentH.LEFT.getValue() )
        delta.setX(-size.getX());
      else if ( textAlignH.getValue() == AlignmentH.CENTER.getValue() )
        delta.setX(-0.5f * size.getX());
    }

    // Vertical alignment adjustment
    if ( alignmentV.getValue() != AlignmentV.TOP.getValue() )
    {
      if ( alignmentV.getValue() == AlignmentV.MIDDLE.getValue() )
        delta.setY(0.5f * size.getY());
      else if ( alignmentV.getValue() == AlignmentV.BOTTOM.getValue() )
        delta.setY(size.getY());
    }

    // Update our actual position
    pos.add(delta);
    _tranNode.translation.setValue(pos);
    _tranNode.enableNotify(saveNotTrans);
    if ( !pos.equals(oldPos) )
      // notify position changed
      _tranNode.touch();

    // Update border geometry
    SAFE_POS.add(pos);
    SbVec3f bmin = bbox.getMin().minus(SAFE_POS.plus(pos));
    SbVec3f bmax = bbox.getMax().minus(SAFE_POS.plus(pos));
    bbox.setBounds(bmin, bmax);

    SoVertexProperty vprop = (SoVertexProperty) _borderGeom.vertexProperty.getValue();
    updateBorderGeom(vprop, bbox);

    // Remember current state
    _curBBox = bbox;
    _curViewport = vport;
    _isModified = false;
  }

  protected void updateBorderGeom(SoVertexProperty vprop, SbBox3f bbox)
  {
    SbVec3f p0 = bbox.getMin();
    SbVec3f p1 = bbox.getMax();
    SbVec3f size = bbox.getSize();
    // Leave some slack
    p0.setX(p0.getX() - 0.02f * size.getX());
    p1.setX(p1.getX() + 0.02f * size.getX());
    p1.setY(p1.getY() + 0.02f * size.getY());

    SbVec3f[] vertices = new SbVec3f[5];
    vertices[0] = new SbVec3f(p0.getX(), p0.getY(), p0.getZ());
    vertices[1] = new SbVec3f(p1.getX(), p0.getY(), p0.getZ());
    vertices[2] = new SbVec3f(p1.getX(), p1.getY(), p1.getZ());
    vertices[3] = new SbVec3f(p0.getX(), p1.getY(), p0.getZ());
    vertices[4] = new SbVec3f(p0.getX(), p0.getY(), p0.getZ());

    if ( !Arrays.equals(vertices, vprop.vertex.getValues(0)) )
      vprop.vertex.setValues(0, vertices);
  }
}
