///////////////////////////////////////////////////////////////////////
//
// Open Inventor example for Multitexturing.
// Multitexturing is used in this example to
// realize a light mapping.
//
///////////////////////////////////////////////////////////////////////
package inventor.advanced.MultiTexturing;

import java.awt.BorderLayout;
import java.awt.Component;

import com.openinventor.inventor.SbMatrix;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec2f;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewVolume;
import com.openinventor.inventor.SbViewVolume.Matrices;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.actions.SoGetBoundingBoxAction;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.sensors.SoNodeSensor;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;


public class Main extends Example {

  public static final float DEFAULT_LIGHT_FOV = 1.04720F;
  public static final SbVec3f DEFAULT_SPOT_DIR = new SbVec3f(0, -1, 0);
  public static final SbRotation DEFAULT_SPOT_ORIENTATION =
      new SbRotation(0.0f, 0.0f, 0.382683f, 0.923880f);
  public static final String TITLE =
      new String("Multitexturing, Light Mapping and Projective texture");
  public final String TEXTURE_PATH = new String("$OIVJHOME/data/textures/");

  public static final String ProjectedTextures[] = new String[] {
      "jpg/Spot2.jpg",
      "jpg/Spot3.jpg",
      "jpg/Spot4.jpg",
      "jpg/Spot5.jpg",
      "jpg/Spot6.jpg",
      "jpg/Spot7.jpg",
      "jpg/Spot8.jpg",
  };

  // Scale and bias [-1,1]^3 clip space into [0,1]^3 texture space.
  static SbMatrix Smatrix = new SbMatrix(0.5f, 0.0f, 0.0f, 0.0f,
                                         0.0f, 0.5f, 0.0f, 0.0f,
                                         0.0f, 0.0f, 0.5f, 0.0f,
                                         0.5f, 0.5f, 0.5f, 1.0f);

  private IViewerExaminer myViewer;
  private SoRotor m_spotRotor;
  private SoTextureMatrix m_projectiveTexMat;
  private SoSeparator m_spotRotorSep;
  private SoBlinker m_blinkerProjTex;
  private SoNodeSensor m_spotRotorSensor;

  /*---------------------------------------------------------------------------*/
  public static void main(String[] argv) {
    Main example = new Main();
    example.demoMain("Multi texturing");
  }

  /*---------------------------------------------------------------------------*/
  @Override
  public void start() {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    SoSeparator scene_graph = makeSceneGraph();

    myViewer.setSceneGraph(scene_graph);
    myViewer.getRenderArea().getSceneInteractor().enableHeadlight(false);
    myViewer.viewAll();
    updateProjectiveTexture();

    UserInterface ui = new UserInterface(this);

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(ui.getMainPanel(), BorderLayout.WEST);
  }

  @Override
  public void stop()
  {
    m_spotRotorSensor.detach();
    myViewer.dispose();
  }

  /*---------------------------------------------------------------------------*/
  SoTexture2 buildProjectiveTex(String texName) {
    SoTexture2 projectiveTex = new SoTexture2();
    projectiveTex.wrapS.setValue(SoTexture.WrapType.CLAMP);
    projectiveTex.wrapT.setValue(SoTexture.WrapType.CLAMP);
    projectiveTex.model.setValue(SoTexture.Models.BLEND);
    projectiveTex.filename.setValue(TEXTURE_PATH + texName);

    return projectiveTex;
  }

  /*----------------------------------------------------------------------------*/
  // start/stop the rotation of the spot
  public void animSpot(boolean do_anim) {
    m_spotRotor.on.setValue(do_anim);
  }

  // start/stop the automatic changment of the projected texture
  public void blinkProjectedTexture(boolean do_blink) {
    m_blinkerProjTex.on.setValue(do_blink);
  }

  // choose manually the projected texture
  public void selectProjectedTexure(int texture_id) {
    m_blinkerProjTex.on.setValue(false);
    if (texture_id == -1)
      m_blinkerProjTex.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
    else
      m_blinkerProjTex.whichChild.setValue(texture_id);
  }

  /*---------------------------------------------------------------------------*/
  void updateProjectiveTexture() {
    // Compute a view volume of the spot
    SbViewVolume viewVol = new SbViewVolume();
    SbMatrix viewingMatrix = new SbMatrix();
    SbMatrix projectionMatrix = new SbMatrix();

    viewVol.perspective(DEFAULT_LIGHT_FOV, 1, 1, 40);

    SoGetBoundingBoxAction bboxAction = new SoGetBoundingBoxAction(new SbViewportRegion()) ;
    bboxAction.apply(m_spotRotorSep);

    SbVec3f spotDir = new SbVec3f();
    spotDir = DEFAULT_SPOT_ORIENTATION.multVec(DEFAULT_SPOT_DIR);

    SbRotation orientation = new SbRotation();
    orientation.setValue(new SbVec3f(0, 0, -1), spotDir);
    orientation.multiply(m_spotRotor.rotation.getValue());

    viewVol.rotateCamera(orientation);
    viewVol.translateCamera(bboxAction.getCenter());

    Matrices matrices = viewVol.getMatrices();
    viewingMatrix = matrices.affine;
    projectionMatrix = matrices.proj;

    // Update the texture matrix
    SbMatrix pmat = new SbMatrix(viewingMatrix);
    pmat.multiply(projectionMatrix);
    pmat.multiply(Smatrix);

    m_projectiveTexMat.matrix.setValue(pmat);
  }

  /*---------------------------------------------------------------------------*/
  public SoSeparator makeSceneGraph() {

    SoSeparator root = new SoSeparator();

    SoSeparator scene3D = new SoSeparator();

    SoComplexity complexity = new SoComplexity();
    complexity.value.setValue(0.8f);
    complexity.textureQuality.setValue(1.0f);

    //************** Build texture unit 0 with a light map texture ***********

     // The default texture unit is 0, so it is not
     // necessary to specify it.
    SoTextureUnit texUnit0 = new SoTextureUnit();
    texUnit0.unit.setValue(0);

    SoTexture2 lightMapTex = new SoTexture2();
    lightMapTex.model.setValue(SoTexture.Models.REPLACE);
    lightMapTex.filename.setValue(TEXTURE_PATH + "png/Lightmap.png");

    // Texture coordinates are defined by a function

    //************** Build texture unit 2 with projective textures ***********
    SoGroup projectiveTexGroup = new SoGroup();

    SoTextureUnit texUnit2 = new SoTextureUnit();
    texUnit2.unit.setValue(2);

    m_blinkerProjTex = new SoBlinker();
    m_blinkerProjTex.speed.setValue(0.02f);

    m_projectiveTexMat = new SoTextureMatrix();

    //************** Build texture unit 1 with an oxidated texture ***********

    SoTextureUnit texUnit1 = new SoTextureUnit();
    texUnit1.unit.setValue(1);

    // We let Open Inventor compute texture coordinates for texture unit 1
    SoTexture2 oxidatedTex = new SoTexture2();
    oxidatedTex.filename.setValue(TEXTURE_PATH + "jpg/Oxidated.jpg");
    oxidatedTex.model.setValue(SoTexture.Models.BLEND);
    oxidatedTex.blendColor.setValue(0.5f, 0.2f, 0.f);

    // Make a sphere, cone and cube with :
    //    - light map texture in texture unit 0
    //    - oxidated texture in texture unit 1
    //    - projective texture in texture unit 2
    SoSeparator objectsSep = new SoSeparator();

    // Sphere
    SoSeparator sphereSep = new SoSeparator();

    SoSeparator sphereShapeSep = new SoSeparator();
    SoSphere sphere = new SoSphere();
    sphere.radius.setValue(2);

    // Cone
    SoSeparator coneSep = new SoSeparator();

    SoSeparator coneShapeSep = new SoSeparator();
    SoTranslation transCone = new SoTranslation();
    transCone.translation.setValue(4, 0, 4);
    SbMatrix transMatrix = new SbMatrix();
    transMatrix.setTranslate(new SbVec3f(4, 0, 4));
    SoTextureMatrix projConeTexMat = new SoTextureMatrix();
    projConeTexMat.matrix.setValue(transMatrix);

    SoCone cone = new SoCone();
    cone.bottomRadius.setValue(2);
    cone.height.setValue(4);

    // Cube
    SoSeparator cubeSep = new SoSeparator();

    SoSeparator cubeShapeSep = new SoSeparator();
    SoTranslation transCube = new SoTranslation();
    transCube.translation.setValue( -5, 0, -5);

    SbMatrix.identity();
    transMatrix.setTranslate(new SbVec3f( -5, 0, -5));
    SoTextureMatrix projCubeTexMat = new SoTextureMatrix();
    projCubeTexMat.matrix.setValue(transMatrix);

    SoCube cube = new SoCube();
    cube.width.setValue(4);
    cube.depth.setValue(4);
    cube.height.setValue(4);

    // Make Floor ground with
    //    - light map texture in texture unit 0
    //    - oxidated texture in texture unit 1
    //    - projective texture in texture unit 2
    SoSeparator floorGroundSep = new SoSeparator();
    SoTranslation transFloorGround = new SoTranslation();
    transFloorGround.translation.setValue(0, -2, 0);
    SoTexture2Transform projGroundTexMat = new SoTexture2Transform();
    projGroundTexMat.translation.setValue(0, -2);

    SoTexture2Transform floorGroundTransTex = new SoTexture2Transform();
    floorGroundTransTex.scaleFactor.setValue(2, 2);

    SoTexture2 floorGroundTex = new SoTexture2();
    floorGroundTex.model.setValue(SoTexture.Models.REPLACE);
    floorGroundTex.filename.setValue(TEXTURE_PATH + "jpg/MarbleTiles.jpg");

    SoCube floorGround = new SoCube();
    floorGround.width.setValue(25);
    floorGround.depth.setValue(25);
    floorGround.height.setValue(0.125f);

    // Build a spot light
    SoSeparator spotSep = new SoSeparator();

    m_spotRotor = new SoRotor();
    m_spotRotor.rotation.setValue(new SbVec3f(0.0f, 1.0f, 0.0f),
                                  (float) Math.PI / 32.0f); // y axis
    m_spotRotor.speed.setValue(0.05f);

    m_spotRotorSensor = new SoNodeSensor(new Runnable() {
      @Override
      public void run() {
        updateProjectiveTexture() ;
      }
    });
    m_spotRotorSensor.attach(m_spotRotor);

    SoTransform transfSpot = new SoTransform();
    transfSpot.rotation.setValue(DEFAULT_SPOT_ORIENTATION);
    transfSpot.translation.setValue(new SbVec3f( -16.5f, 7f, 0.f));
    transfSpot.scaleFactor.setValue(new SbVec3f(0.75f, 0.75f, 0.75f));

    SoCone spotPart1 = new SoCone();
    spotPart1.removePart(SoCone.PartType.BOTTOM);

    SoCylinder spotPart2 = new SoCylinder();
    spotPart2.radius.setValue(0.5f);
    spotPart2.height.setValue(1.95f);

    SoTexture2 spotTex1 = new SoTexture2();
    spotTex1.filename.setValue(TEXTURE_PATH + "jpg/Ste06.jpg");

    SoTexture2Transform transTexSpot3 = new SoTexture2Transform();
    transTexSpot3.translation.setValue(new SbVec2f(0.5f, -0.10f));
    transTexSpot3.scaleFactor.setValue(new SbVec2f(0.8f, 0.8f));
    transTexSpot3.rotation.setValue( (float) Math.PI / 4.0f);

    SoCone spotPart3 = new SoCone();
    spotPart3.removePart(SoCone.PartType.SIDES);

    m_spotRotorSep = new SoSeparator();

    // assemble scene graph
    root.addChild(scene3D);
    {
      scene3D.addChild(complexity);
      scene3D.addChild(texUnit0);
      scene3D.addChild(new SoTextureCoordinateEnvironment());
      scene3D.addChild(lightMapTex);
      scene3D.addChild(texUnit1);
      scene3D.addChild(objectsSep);
      {
        objectsSep.addChild(oxidatedTex);
        objectsSep.addChild(projectiveTexGroup);
        {
          projectiveTexGroup.addChild(texUnit2);
          projectiveTexGroup.addChild(new SoTextureCoordinateObject());
          projectiveTexGroup.addChild(m_projectiveTexMat);
          projectiveTexGroup.addChild(m_blinkerProjTex);
          {
            m_blinkerProjTex.addChild(buildProjectiveTex(ProjectedTextures[0]));
            m_blinkerProjTex.addChild(buildProjectiveTex(ProjectedTextures[1]));
            m_blinkerProjTex.addChild(buildProjectiveTex(ProjectedTextures[2]));
            m_blinkerProjTex.addChild(buildProjectiveTex(ProjectedTextures[3]));
            m_blinkerProjTex.addChild(buildProjectiveTex(ProjectedTextures[4]));
            m_blinkerProjTex.addChild(buildProjectiveTex(ProjectedTextures[5]));
            m_blinkerProjTex.addChild(buildProjectiveTex(ProjectedTextures[6]));
          }
        }
        objectsSep.addChild(coneSep);
        {
          coneSep.addChild(projConeTexMat);
          coneSep.addChild(coneShapeSep);
          {
            coneShapeSep.addChild(transCone);
            coneShapeSep.addChild(cone);
          }
        }
        objectsSep.addChild(cubeSep);
        {
          cubeSep.addChild(projCubeTexMat);
          cubeSep.addChild(cubeShapeSep);
          {
            cubeShapeSep.addChild(transCube);
            cubeShapeSep.addChild(cube);
          }
        }
        objectsSep.addChild(sphereSep);
        {
          sphereSep.addChild(sphereShapeSep);
          {
            sphereShapeSep.addChild(sphere);
          }
        }
      }
      scene3D.addChild(floorGroundSep);
      {
        floorGroundSep.addChild(transFloorGround);
        // Build another texture unit 1 with a wood texture
        floorGroundSep.addChild(floorGroundTransTex);
        floorGroundSep.addChild(floorGroundTex);
        floorGroundSep.addChild(projectiveTexGroup);
        floorGroundSep.addChild(projGroundTexMat);
        floorGroundSep.addChild(floorGround);
      }
      scene3D.addChild(m_spotRotorSep);
      {
        m_spotRotorSep.addChild(m_spotRotor);
        m_spotRotorSep.addChild(spotSep);
        {
          spotSep.addChild(transfSpot);
          spotSep.addChild(spotTex1);
          spotSep.addChild(spotPart1);
          spotSep.addChild(spotPart2);
          spotSep.addChild(m_blinkerProjTex);
          spotSep.addChild(transTexSpot3);
          spotSep.addChild(spotPart3);
        }
      }

      return root;
    }

  }

}
