package inventor.advanced.collision.sceneCollider;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.File;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;
import javax.swing.filechooser.FileFilter;

import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.actions.SoGetBoundingBoxAction;
import com.openinventor.inventor.collision.SoDualSceneCollider;
import com.openinventor.inventor.manips.SoHandleBoxManip;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{

  private IViewerExaminer m_viewer;

  private DualSceneCollider m_sceneCollider;
  private int m_numCollision;
  private int m_maxStaticT;
  private int m_maxMovingT;
  private boolean m_isActive;

  private SoHandleBoxManip m_manip;
  private SoSeparator m_root;
  private SoSeparator m_movingRoot;
  private SoGroup m_movingScene;
  private SoPath m_movingScenePath;
  private SbBox3f m_movingBbox;
  private SoNode m_staticScene;
  private SoPath m_staticScenePath;
  private SbBox3f m_staticBbox;
  private SoCoordinate3 m_trianglesCoord;
  private SoCoordinate3 m_commonPtsCoord;
  private SoSwitch m_collidingSwitch;

  private JLabel m_staticSceneValueLabel;
  private JLabel m_movingSceneValueLabel;
  private JLabel m_collisionValueLabel;

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("Scene Collider");
  }

  @Override
  public void start()
  {
    m_viewer = ViewerComponentsFactory.createViewerExaminer();

    m_sceneCollider = new DualSceneCollider();
    m_numCollision = 0;
    m_maxStaticT = 1000;
    m_maxMovingT = 100;
    m_isActive = true;

    m_movingBbox = new SbBox3f(0, 0, 0, 1, 1, 1);
    m_staticBbox = new SbBox3f(0, 0, 0, 1, 1, 1);

    buildUI();
    buildSceneGraph();
    enableCollisions(m_isActive);

    // Set up viewer
    m_viewer.setSceneGraph(m_root);
    m_viewer.viewAll();
  }

  @Override
  public void stop()
  {
    m_viewer.dispose();
  }

  private void buildSceneGraph()
  {
    m_manip = new SoHandleBoxManip();

    // build the group that highlight the colliding triangle pair
    SoFaceSet faceSet = new SoFaceSet();
    faceSet.numVertices.set1Value(0, 3);
    faceSet.numVertices.set1Value(1, 3);
    m_trianglesCoord = new SoCoordinate3();
    m_collidingSwitch = new SoSwitch();
    m_collidingSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
    SoMaterial collidingMat = new SoMaterial();
    collidingMat.diffuseColor.setValue(1, 1, 0);
    SoSeparator collidingSep = new SoSeparator();

    // build the common point of current pair of intersecting triangle
    m_commonPtsCoord = new SoCoordinate3();
    SoMarkerSet markerSet = new SoMarkerSet();
    markerSet.markerIndex.setValue(SoMarkerSet.MarkerTypes.SATELLITE_FILLED_9_9.getValue());

    m_root = new SoSeparator();
    m_movingRoot = new SoSeparator();
    m_movingScene = new SoGroup();
    m_staticScene = null;

    // assemble the scene graph
    m_root.addChild(m_movingRoot);
    {
      m_movingRoot.addChild(collidingSep);
      {
        collidingSep.addChild(m_collidingSwitch);
        {
          m_collidingSwitch.addChild(m_trianglesCoord);
          m_collidingSwitch.addChild(collidingMat);
          m_collidingSwitch.addChild(new SoPolygonOffset());
          m_collidingSwitch.addChild(faceSet);
          m_collidingSwitch.addChild(m_commonPtsCoord);
          m_collidingSwitch.addChild(markerSet);
        }
      }
      m_movingRoot.addChild(m_manip);
      m_movingRoot.addChild(m_movingScene);
    }

    m_staticScenePath = new SoPath();
    setStaticScene(getDefaultScene());

    m_movingScenePath = new SoPath(m_root);
    setMovingScene(getDefaultScene());
  }

  private SoGroup getDefaultScene()
  {
    SoSphere m_objectSphere = new SoSphere();
    SoMaterial sphereMat = new SoMaterial();
    sphereMat.diffuseColor.setValue(1, 0, 0);
    SoComplexity cps = new SoComplexity();
    cps.value.setValue(1);

    SoGroup parent = new SoGroup();
    {
      parent.addChild(sphereMat);
      parent.addChild(cps);
      parent.addChild(m_objectSphere);
    }

    return parent;
  }

  /**
   * Define a new moving scene
   */
  private void setMovingScene(SoNode scene)
  {
    m_movingScene.removeAllChildren();
    m_movingScene.addChild(scene);

    m_movingScenePath.regular.truncate(0);
    m_movingScenePath.regular.append(m_root);
    m_movingScenePath.regular.append(m_movingRoot);
    m_movingScenePath.regular.append(m_movingScene);

    // update the position and size of the moving object according to the static
    // scene
    SoGetBoundingBoxAction act = new SoGetBoundingBoxAction(new SbViewportRegion());
    act.apply(m_movingScene);
    m_movingBbox = act.getBoundingBox();

    updateManip(m_movingBbox, m_staticBbox);

    m_sceneCollider.setMovingScene(m_movingScenePath, m_maxMovingT);
  }

  /**
   * Define a new static scene
   */
  private void setStaticScene(SoNode scene)
  {
    SoNode last_static_scene = m_staticScene;
    m_staticScene = scene;

    if ( last_static_scene == null )
      m_root.addChild(m_staticScene);
    else
      m_root.replaceChild(last_static_scene, m_staticScene);

    m_staticScenePath.regular.truncate(0);
    m_staticScenePath.regular.append(m_root);
    m_staticScenePath.regular.append(m_staticScene);

    m_sceneCollider.setStaticScene(m_staticScenePath, m_maxStaticT);

    // update the position and size of the moving object according to the static
    // scene
    SoGetBoundingBoxAction act = new SoGetBoundingBoxAction(new SbViewportRegion());
    act.apply(m_staticScene);
    m_staticBbox = act.getBoundingBox();

    updateManip(m_movingBbox, m_staticBbox);
  }

  private void updateManip(SbBox3f movingBbox, SbBox3f staticBbox)
  {
    // scale the "moving" object according to the static scene size
    SbVec3f size = staticBbox.getSize();
    float smax = size.array[0];
    if ( size.array[1] > smax )
      smax = size.array[1];
    if ( size.array[2] > smax )
      smax = size.array[2];
    smax /= 2;

    size = movingBbox.getSize();
    float mmax = size.array[0];
    if ( size.array[1] > mmax )
      mmax = size.array[1];
    if ( size.array[2] > mmax )
      mmax = size.array[2];

    // translate the "moving" object at the lower corner of the static scene
    SbVec3f translation = new SbVec3f(staticBbox.getMin());
    translation.negate();
    translation.substract(movingBbox.getCenter());

    m_manip.translation.setValue(translation);
    m_manip.scaleFactor.setValue(new SbVec3f(smax / mmax, smax / mmax, smax / mmax));
    m_manip.center.setValue(movingBbox.getCenter());
  }

  private void enableCollisions(boolean check)
  {
    if ( check )
    {
      m_sceneCollider.activate(m_manip);
      messageCollision(m_numCollision + "");
      m_isActive = true;
    }
    else
    {
      m_sceneCollider.activate(null);
      messageCollision("detection disabled !");
      m_isActive = false;
    }
  }

  private void setMaxTriangleForMoving(int maxT)
  {
    m_sceneCollider.setMovingScene(m_movingScenePath, maxT);
    m_maxMovingT = maxT;
  }

  private void setMaxTriangleForStatic(int maxT)
  {
    m_sceneCollider.setStaticScene(m_staticScenePath, maxT);
    m_maxStaticT = maxT;
  }

  private static SoNode readSceneGraph(String fileName)
  {
    SoInput input = new SoInput();

    if ( !input.openFile(fileName) )
    {
      System.err.println("Cannot open file " + fileName);
      return null;
    }

    SoSeparator node = SoDB.readAll(input);
    if ( node == null )
    {
      System.err.println("Problem reading file");
      input.closeFile();
      return null;
    }

    input.closeFile();
    return node;
  }

  private void buildUI()
  {
    final FileFilter ivFileFilter = new FileFilter()
    {

      @Override
      public String getDescription()
      {
        return "Open-Inventor files";
      }

      @Override
      public boolean accept(File f)
      {
        if ( f == null )
          return false;
        if ( f.isDirectory() )
          return true;
        return f.getName().toLowerCase().endsWith(".iv");
      }
    };

    final JFileChooser ivFileChooser = new JFileChooser(
        new File(SoPreferences.getValue("OIVJHOME") + File.separator + "data" + File.separator + "models"));
    ivFileChooser.setFileFilter(ivFileFilter);
    ivFileChooser.removeChoosableFileFilter(ivFileChooser.getAcceptAllFileFilter());

    // Collision
    JPanel collisionPanel = new JPanel(new GridBagLayout());
    collisionPanel.setBorder(new TitledBorder(new EtchedBorder(), "Collision"));

    final JCheckBox activateBox = new JCheckBox("Activate collision");
    activateBox.setSelected(m_isActive);
    activateBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        enableCollisions(activateBox.isSelected());
      }
    });
    GridBagConstraints activateBoxGbc = new GridBagConstraints();
    activateBoxGbc.anchor = GridBagConstraints.WEST;
    activateBoxGbc.insets = new Insets(0, 0, 5, 5);
    activateBoxGbc.gridx = 0;
    activateBoxGbc.gridy = 0;
    collisionPanel.add(activateBox, activateBoxGbc);

    m_collisionValueLabel = new JLabel();
    GridBagConstraints collisionLabelGbc = new GridBagConstraints();
    collisionLabelGbc.anchor = GridBagConstraints.WEST;
    collisionLabelGbc.insets = new Insets(5, 5, 5, 5);
    collisionLabelGbc.gridx = 0;
    collisionLabelGbc.gridy = 1;
    collisionLabelGbc.weightx = 1;
    collisionPanel.add(m_collisionValueLabel, collisionLabelGbc);

    // Static scene
    JPanel staticScenePanel = new JPanel(new GridBagLayout());
    staticScenePanel.setBorder(new TitledBorder(new EtchedBorder(), "Static scene"));

    JButton newStaticSceneButton = new JButton("Open file...");
    newStaticSceneButton.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        if ( ivFileChooser.showOpenDialog(null) != JFileChooser.APPROVE_OPTION )
          return;
        String fileName = ivFileChooser.getSelectedFile().getAbsolutePath();
        SoNode node = readSceneGraph(fileName);
        setStaticScene(node);
      }
    });
    GridBagConstraints newStaticButtonGbc = new GridBagConstraints();
    newStaticButtonGbc.anchor = GridBagConstraints.WEST;
    newStaticButtonGbc.insets = new Insets(0, 5, 5, 0);
    newStaticButtonGbc.gridx = 0;
    newStaticButtonGbc.gridy = 0;
    staticScenePanel.add(newStaticSceneButton, newStaticButtonGbc);

    m_staticSceneValueLabel = new JLabel();
    GridBagConstraints staticLabelGbc = new GridBagConstraints();
    staticLabelGbc.anchor = GridBagConstraints.WEST;
    staticLabelGbc.insets = new Insets(0, 10, 5, 5);
    staticLabelGbc.gridx = 1;
    staticLabelGbc.gridy = 0;
    staticScenePanel.add(m_staticSceneValueLabel, staticLabelGbc);

    final SliderPanel maxStaticSliderPanel = new SliderPanel(100, 10000, m_maxStaticT);
    maxStaticSliderPanel.setTextFieldColumns(4);
    maxStaticSliderPanel.addInfoText("Max triangle/leaf in static scene");
    maxStaticSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        setMaxTriangleForStatic((int) value);
      }
    });
    GridBagConstraints staticSliderPanelGbc = new GridBagConstraints();
    staticSliderPanelGbc.gridwidth = 2;
    staticSliderPanelGbc.weightx = 1.0;
    staticSliderPanelGbc.anchor = GridBagConstraints.WEST;
    staticSliderPanelGbc.fill = GridBagConstraints.VERTICAL;
    staticSliderPanelGbc.gridx = 0;
    staticSliderPanelGbc.gridy = 1;
    staticScenePanel.add(maxStaticSliderPanel, staticSliderPanelGbc);

    // Moving scene
    JPanel movingScenePanel = new JPanel(new GridBagLayout());
    movingScenePanel.setBorder(new TitledBorder(new EtchedBorder(), "Moving scene"));

    JButton newMovingSceneButton = new JButton("Open file...");
    newMovingSceneButton.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        if ( ivFileChooser.showOpenDialog(null) != JFileChooser.APPROVE_OPTION )
          return;
        String fileName = ivFileChooser.getSelectedFile().getAbsolutePath();
        SoNode node = readSceneGraph(fileName);
        setMovingScene(node);
      }
    });
    GridBagConstraints newMovingButtonGbc = new GridBagConstraints();
    newMovingButtonGbc.anchor = GridBagConstraints.WEST;
    newMovingButtonGbc.insets = new Insets(0, 5, 5, 0);
    newMovingButtonGbc.gridx = 0;
    newMovingButtonGbc.gridy = 0;
    movingScenePanel.add(newMovingSceneButton, newMovingButtonGbc);

    m_movingSceneValueLabel = new JLabel();
    GridBagConstraints movingLabelGbc = new GridBagConstraints();
    movingLabelGbc.anchor = GridBagConstraints.WEST;
    movingLabelGbc.insets = new Insets(0, 10, 5, 5);
    movingLabelGbc.gridx = 1;
    movingLabelGbc.gridy = 0;
    movingScenePanel.add(m_movingSceneValueLabel, movingLabelGbc);

    final SliderPanel maxMovingSliderPanel = new SliderPanel(100, 10000, m_maxMovingT);
    maxMovingSliderPanel.setTextFieldColumns(4);
    maxMovingSliderPanel.addInfoText("Max triangle/leaf in moving scene");
    maxMovingSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        setMaxTriangleForMoving((int) value);
      }
    });
    GridBagConstraints movingSliderPanelGbc = new GridBagConstraints();
    movingSliderPanelGbc.gridwidth = 2;
    movingSliderPanelGbc.weightx = 1.0;
    movingSliderPanelGbc.anchor = GridBagConstraints.WEST;
    movingSliderPanelGbc.fill = GridBagConstraints.VERTICAL;
    movingSliderPanelGbc.gridx = 0;
    movingSliderPanelGbc.gridy = 1;
    movingScenePanel.add(maxMovingSliderPanel, movingSliderPanelGbc);

    JPanel optionsPanel = new JPanel();
    optionsPanel.setLayout(new GridLayout(0, 1, 0, 0));
    optionsPanel.add(collisionPanel);
    optionsPanel.add(staticScenePanel);
    optionsPanel.add(movingScenePanel);

    final Component component = m_viewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(optionsPanel, BorderLayout.SOUTH);
  }

  private void messageCollision(String message)
  {
    m_collisionValueLabel.setText("# collisions: " + message);
  }

  class DualSceneCollider extends SoDualSceneCollider
  {

    private SoPath m_lastPath;
    private SoGroup m_lastHighLightGroup;
    private SoMaterial m_highlight;

    private int m_numST, m_numst, m_percentS;
    private int m_numMT, m_nummt, m_percentM;

    public DualSceneCollider()
    {
      super();

      m_lastPath = null;
      m_lastHighLightGroup = null;

      // define an highlight node (special color attr) that will be inserted
      // just before the colliding shape
      m_highlight = new SoMaterial();
      m_highlight.diffuseColor.setValue(0, 1, 0);

      m_numST = m_numst = m_percentS = 0;
      m_numMT = m_nummt = m_percentM = 0;
    }

    @Override
    public void setStaticScene(SoPath scene, int max_triangles_per_leaf)
    {
      super.setStaticScene(scene, max_triangles_per_leaf);

      m_lastPath = null;
    }

    @Override
    public boolean checkCollision()
    {
      boolean result = super.checkCollision();
      if ( !result )
      {
        if ( m_lastPath != null )
        {
          // a new object was colliding during last checkCollision
          m_lastHighLightGroup.removeChild(m_highlight);
          m_collidingSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
          messageCollision("");
        }

        m_lastPath = null;
      }
      return result;
    }

    @Override
    public boolean searchNextIntersection()
    {
      // a new intersection occurs
      m_numCollision++;
      messageCollision(m_numCollision + "");

      // get the path in the static scene of the current colliding shape
      SoPath path = getCollidingStaticPath();
      int shapeIndex = path.regular.getIndexFromTail(0);
      if ( m_lastPath != path )
      {
        // a new object is colliding, highlight it by a inserting the given
        // m_highlight node
        if ( m_lastPath != null )
          m_lastHighLightGroup.removeChild(m_highlight);
        SoNode parent = path.regular.getNodeFromTail(1);
        if ( parent instanceof SoGroup )
        {
          m_lastHighLightGroup = (SoGroup) parent;
          m_lastHighLightGroup.insertChild(m_highlight, shapeIndex);
        }
        else
        {
          path = null;
        }
      }

      m_collidingSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
      SbVec3f[] vecs = getCollidingStaticTriangle();
      m_trianglesCoord.point.set1Value(0, vecs[0]);
      m_trianglesCoord.point.set1Value(1, vecs[1]);
      m_trianglesCoord.point.set1Value(2, vecs[2]);
      vecs = getCollidingMovingTriangle();
      m_trianglesCoord.point.set1Value(3, vecs[0]);
      m_trianglesCoord.point.set1Value(4, vecs[1]);
      m_trianglesCoord.point.set1Value(5, vecs[2]);

      vecs = getCommonPoints();
      m_commonPtsCoord.point.set1Value(0, vecs[0]);
      m_commonPtsCoord.point.set1Value(1, vecs[1]);

      m_lastPath = path;
      return false;
    }

    @Override
    protected void staticTriangleListBuilt(int numTriangles)
    {
      m_numST = numTriangles;
      m_percentS = 5;
      m_numst = 0;
      messageStaticScene(0);
    }

    @Override
    protected void staticLeafBuilt(int numTriangles)
    {
      m_numst += numTriangles;
      if ( m_numst >= (m_numST * m_percentS) / 100 )
      {
        messageStaticScene(m_percentS);
        m_percentS += 5;
      }
    }

    @Override
    protected void movingTriangleListBuilt(int numTriangles)
    {
      m_numMT = numTriangles;
      m_percentM = 5;
      m_nummt = 0;
      messageMovingScene(0);
    }

    @Override
    protected void movingLeafBuilt(int numTriangles)
    {
      m_nummt += numTriangles;
      if ( m_nummt >= (m_numMT * m_percentM) / 100 )
      {
        messageMovingScene(m_percentM);
        m_percentM += 5;
      }
    }

    private void messageStaticScene(int percentDone)
    {
      m_staticSceneValueLabel.setText(m_numST + " triangles found, " + percentDone + "% processed");
    }

    private void messageMovingScene(int percentDone)
    {
      m_movingSceneValueLabel.setText(m_numMT + " triangles found, " + percentDone + "% processed");
    }

  }

}
