package inventor.advanced.hiddenLine;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;

import javax.swing.JFileChooser;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.filechooser.FileFilter;

import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.nodes.SoColorMask;
import com.openinventor.inventor.nodes.SoDepthOffset;
import com.openinventor.inventor.nodes.SoDrawStyle;
import com.openinventor.inventor.nodes.SoFile;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoLightModel;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;

/**
 * Demonstrates how to use the SoColorMask node in order to do "Hidden Line"
 * rendering.
 *
 * The SoColorMask node is used to render only the depth of the model faces. In
 * a second pass, the model is rendered again in wireframe mode using the style
 * field of SoDrawStyle and setting it to LINES.
 */
public class Main extends Example
{
  private IViewerExaminer m_viewer;
  private SoFile m_modelFile;

  @Override
  public void start()
  {
    setLayout(new BorderLayout());

    JMenuBar menubar = buildMenu();
    add(menubar, BorderLayout.NORTH);

    m_viewer = ViewerComponentsFactory.createViewerExaminer();
    m_viewer.setSceneGraph(createSceneGraph());
    m_viewer.viewAll();

    final Component component = m_viewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(1280, 720));
    add(component, BorderLayout.CENTER);
  }

  @Override
  public void stop()
  {
    m_viewer.dispose();
  }

  private JMenuBar buildMenu()
  {
    JPopupMenu.setDefaultLightWeightPopupEnabled(false);

    // File menu
    JMenu fileMenu = new JMenu("File");

    final FileFilter ivFileFilter = new FileFilter()
    {
      @Override
      public String getDescription()
      {
        return "Open Inventor files";
      }

      @Override
      public boolean accept(File f)
      {
        if ( f == null )
          return false;
        if ( f.isDirectory() )
          return true;
        return f.getName().toLowerCase().endsWith(".iv");
      }
    };

    final JFileChooser ivFileChooser = new JFileChooser(
        new File(SoPreferences.getValue("OIVJHOME") + File.separator + "data" + File.separator + "models"));
    ivFileChooser.setFileFilter(ivFileFilter);
    ivFileChooser.removeChoosableFileFilter(ivFileChooser.getAcceptAllFileFilter());

    JMenuItem openIvFileItem = new JMenuItem("Open");

    openIvFileItem.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        if ( ivFileChooser.showOpenDialog(null) != JFileChooser.APPROVE_OPTION )
          return;
        m_modelFile.name.setValue(ivFileChooser.getSelectedFile().getAbsolutePath());
        m_viewer.viewAll();
      }
    });

    fileMenu.add(openIvFileItem);

    JMenuItem closeDemoItem = new JMenuItem("Exit");
    closeDemoItem.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        System.exit(0);
      }
    });
    fileMenu.add(closeDemoItem);

    JMenuBar menubar = new JMenuBar();
    menubar.add(fileMenu);
    return menubar;
  }

  private SoSeparator createSceneGraph()
  {
    String pkgName = this.getClass().getPackage().getName();
    pkgName = pkgName.replace('.', File.separatorChar);

    SoSeparator sceneGraph = new SoSeparator();
    sceneGraph.addChild(new SoGradientBackground());

    SoSeparator firstPassSeparator = new SoSeparator();
    sceneGraph.addChild(firstPassSeparator);

    // For the first pass, use a ColorMask with all fields to FALSE in order to
    // draw only into the depth buffer
    SoColorMask colorMask = new SoColorMask();
    colorMask.red.setValue(false);
    colorMask.green.setValue(false);
    colorMask.blue.setValue(false);
    colorMask.alpha.setValue(false);
    firstPassSeparator.addChild(colorMask);

    m_modelFile = new SoFile();
    m_modelFile.name.setValue(SoPreferences.getValue("OIVJHOME") + File.separator + "examples" + File.separator
        + pkgName + File.separator + "hiddenLine_model.iv");
    firstPassSeparator.addChild(m_modelFile);

    SoSeparator secondPassSeparator = new SoSeparator();
    sceneGraph.addChild(secondPassSeparator);

    // For the second pass, draw the model in BASE_COLOR mode and DrawStyle
    // LINES
    SoLightModel lightModel = new SoLightModel();
    lightModel.model.setValue(SoLightModel.Models.BASE_COLOR);
    secondPassSeparator.addChild(lightModel);

    SoDrawStyle drawStyle = new SoDrawStyle();
    drawStyle.style.setValue(SoDrawStyle.Styles.LINES);
    secondPassSeparator.addChild(drawStyle);

    // Add a DepthOffset to make sure the lines will always be drawn in front
    // of the model
    SoDepthOffset depthOffset = new SoDepthOffset();
    secondPassSeparator.addChild(depthOffset);

    secondPassSeparator.addChild(m_modelFile);

    return sceneGraph;
  }

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("Hidden Line Rendering");
  }
}
