package inventor.advanced.highlightingStyles;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JColorChooser;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.filechooser.FileNameExtensionFilter;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.SoSceneManager.AntialiasingModes;
import com.openinventor.inventor.actions.SoBoxHighlightRenderAction;
import com.openinventor.inventor.actions.SoColorHighlightRenderAction;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.actions.SoGLRenderAction.TransparencyTypes;
import com.openinventor.inventor.actions.SoHaloHighlightRenderAction;
import com.openinventor.inventor.actions.SoLineHighlightRenderAction;
import com.openinventor.inventor.misc.callbacks.SoSelectionClassCB;
import com.openinventor.inventor.nodes.SoFile;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoSelection;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;
import com.openinventor.inventor.viewercomponents.nodes.SceneOrbiter;
import com.openinventor.inventor.viewercomponents.nodes.SoViewingCube;

import util.Example;
import util.ViewerComponentsFactory;

/**
 * Demonstrates how to use the SoHighlightRenderAction actions to highlight
 * OpenInventor Shapes.
 *
 * Press the ESC key to switch between NAVIGATION and SELECTION mode.
 *
 * This demo uses a SoSelection with a policy set to SHIFT, which means multiple
 * objects can be selected by maintaining the SHIFT key down.
 */
public class Main extends Example
{
  private IRenderAreaInteractive m_viewer;

  private SoFile m_modelFile;
  private Color m_highlightColor;
  private SoSelection m_rootSelection;
  private SoGLRenderAction m_renderAction;

  private SliderPanel m_widthSliderPanel;
  private SliderPanel m_smoothSliderPanel;
  private JCheckBox m_visibilityCheckBox;
  private JComboBox<String> m_highlightTypesCombo;

  private static final SbColor BKG_COLOR = new SbColor(0.92f, 0.92f, 0.92f);

  @Override
  public void start()
  {
    setLayout(new BorderLayout());

    JMenuBar menubar = buildMenu();
    add(menubar, BorderLayout.NORTH);

    m_highlightColor = new SbColor(1.0f, 0.0f, 0.0f).toAWTColor();
    add(buildOptionsPanel(), BorderLayout.CENTER);

    m_viewer = ViewerComponentsFactory.createRenderAreaOrbiter();
    m_viewer.setSceneGraph(buildSceneGraph());
    m_viewer.viewAll(new SbViewportRegion());

    // ViewingCube custom
    SoViewingCube vcube = ((SceneOrbiter) m_viewer.getSceneInteractor()).getViewingCube();
    vcube.facePosX.setValue("$OIVJHOME/data/examples/ViewingCube/Faces/xyz/xpos.png");
    vcube.faceNegX.setValue("$OIVJHOME/data/examples/ViewingCube/Faces/xyz/xneg.png");
    vcube.facePosY.setValue("$OIVJHOME/data/examples/ViewingCube/Faces/xyz/ypos.png");
    vcube.faceNegY.setValue("$OIVJHOME/data/examples/ViewingCube/Faces/xyz/yneg.png");
    vcube.facePosZ.setValue("$OIVJHOME/data/examples/ViewingCube/Faces/xyz/zpos.png");
    vcube.faceNegZ.setValue("$OIVJHOME/data/examples/ViewingCube/Faces/xyz/zneg.png");
    vcube.edgeSize.setValue(0.3f);
    vcube.edgeStyle.setValue(SoViewingCube.EdgeStyles.FLAT);

    updateHighlightAction();

    m_viewer.setTransparencyType(TransparencyTypes.OPAQUE_FIRST);
    m_viewer.setAntialiasingMode(AntialiasingModes.AUTO);
    m_viewer.setAntialiasingQuality(1);

    final Component component = m_viewer.getComponent();
    component.setPreferredSize(new Dimension(1280, 720));
    add(component, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    m_viewer.dispose();
  }

  private void setHighlightColor(SbColor color)
  {
    if ( m_renderAction instanceof SoLineHighlightRenderAction )
      ((SoLineHighlightRenderAction) m_renderAction).setColor(color);
    else if ( m_renderAction instanceof SoBoxHighlightRenderAction )
      ((SoBoxHighlightRenderAction) m_renderAction).setColor(color);
    else if ( m_renderAction instanceof SoColorHighlightRenderAction )
      ((SoColorHighlightRenderAction) m_renderAction).setColor(color);
    else if ( m_renderAction instanceof SoHaloHighlightRenderAction )
      ((SoHaloHighlightRenderAction) m_renderAction).setHaloColor(color);
  }

  private void setHighlightWidth(int width)
  {
    if ( m_renderAction instanceof SoLineHighlightRenderAction )
      ((SoLineHighlightRenderAction) m_renderAction).setLineWidth(width);
    else if ( m_renderAction instanceof SoBoxHighlightRenderAction )
      ((SoBoxHighlightRenderAction) m_renderAction).setLineWidth(width);
    else if ( m_renderAction instanceof SoHaloHighlightRenderAction )
      ((SoHaloHighlightRenderAction) m_renderAction).setHaloWidth(width);
  }

  private void setHighlightSmooth(float smooth)
  {
    if ( m_renderAction instanceof SoHaloHighlightRenderAction )
      ((SoHaloHighlightRenderAction) m_renderAction).setHaloSmoothFactor(smooth);
  }

  private void setHighlightAlwaysVisible(boolean alwaysVisible)
  {
    if ( m_renderAction instanceof SoHaloHighlightRenderAction )
      ((SoHaloHighlightRenderAction) m_renderAction).setHaloAlwaysVisible(alwaysVisible);
  }

  private void setupHighlight()
  {
    setHighlightWidth((int) m_widthSliderPanel.getSliderValue());
    setHighlightAlwaysVisible(m_visibilityCheckBox.isSelected());
    setHighlightSmooth(m_smoothSliderPanel.getSliderValue());
    setHighlightColor(new SbColor(m_highlightColor));
  }

  private void updateHighlightAction()
  {
    SoGLRenderAction newAction = m_renderAction;

    switch ( m_highlightTypesCombo.getSelectedIndex() )
    {
    case 0 : // LINE
      if ( !(newAction instanceof SoLineHighlightRenderAction) )
      {
        newAction = new SoLineHighlightRenderAction();
        m_widthSliderPanel.setEnabled(true);
        m_visibilityCheckBox.setEnabled(false);
        m_smoothSliderPanel.setEnabled(false);
      }
      break;
    case 1 : // BOX
      if ( !(newAction instanceof SoBoxHighlightRenderAction) )
      {
        newAction = new SoBoxHighlightRenderAction();
        m_widthSliderPanel.setEnabled(true);
        m_visibilityCheckBox.setEnabled(false);
        m_smoothSliderPanel.setEnabled(false);
      }
      break;
    case 2 : // COLOR
      if ( !(newAction instanceof SoColorHighlightRenderAction) )
      {
        newAction = new SoColorHighlightRenderAction();
        m_widthSliderPanel.setEnabled(false);
        m_visibilityCheckBox.setEnabled(false);
        m_smoothSliderPanel.setEnabled(false);
      }
      break;
    case 3 : // HALO
      if ( !(newAction instanceof SoHaloHighlightRenderAction) )
      {
        newAction = new SoHaloHighlightRenderAction();
        m_widthSliderPanel.setEnabled(true);
        m_visibilityCheckBox.setEnabled(true);
        m_smoothSliderPanel.setEnabled(true);
      }
      break;
    }

    if ( newAction != m_renderAction )
    {
      m_renderAction = newAction;
      setupHighlight();
      m_viewer.setGLRenderAction(m_renderAction);
      m_viewer.scheduleRedraw();
    }
  }

  private JMenuBar buildMenu()
  {
    JPopupMenu.setDefaultLightWeightPopupEnabled(false);

    // File menu
    JMenu fileMenu = new JMenu("File");

    final JFileChooser ivFileChooser = new JFileChooser(
        new File(SoPreferences.getValue("OIVJHOME") + File.separator + "data" + File.separator + "models"));
    ivFileChooser.setDialogTitle("Choose model to load:");
    ivFileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
    ivFileChooser.setMultiSelectionEnabled(false);
    ivFileChooser.setAcceptAllFileFilterUsed(false);
    ivFileChooser.addChoosableFileFilter(new FileNameExtensionFilter("Inventor files", "iv"));

    JMenuItem openIvFileItem = new JMenuItem("Open");

    openIvFileItem.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        if ( ivFileChooser.showOpenDialog(null) != JFileChooser.APPROVE_OPTION )
          return;
        m_modelFile.name.setValue(ivFileChooser.getSelectedFile().getAbsolutePath());
        m_viewer.viewAll(new SbViewportRegion());
      }
    });

    fileMenu.add(openIvFileItem);

    JMenuItem closeDemoItem = new JMenuItem("Exit");
    closeDemoItem.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        System.exit(0);
      }
    });
    fileMenu.add(closeDemoItem);

    JMenuBar menubar = new JMenuBar();
    menubar.add(fileMenu);
    return menubar;
  }

  private SoSeparator buildSceneGraph()
  {
    m_modelFile = new SoFile();
    m_modelFile.name.set("$OIVJHOME/data/examples/Models/harley_y_vertical.iv");

    m_rootSelection = new SoSelection();
    SoGradientBackground gradientBackground = new SoGradientBackground();

    gradientBackground.color0.setValue(BKG_COLOR);
    gradientBackground.color1.setValue(BKG_COLOR);
    m_rootSelection.addChild(gradientBackground);
    m_rootSelection.addChild(m_modelFile);

    // Add a callback on selection to redraw the scene and show
    // the highlight directly after picking a shape.
    m_rootSelection.addChangeCallback(new SoSelectionClassCB()
    {
      @Override
      public void invoke(SoSelection s)
      {
        m_viewer.scheduleRedraw();
      }
    });

    SoSeparator root = new SoSeparator();
    root.addChild(m_rootSelection);

    return root;
  }

  private JPanel buildOptionsPanel()
  {
    // Build Highlight parameters panel
    m_widthSliderPanel = new SliderPanel(1, 10, 3, false);
    ((FlowLayout) m_widthSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    m_widthSliderPanel.addInfoText("Highlight Width");
    m_widthSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        setHighlightWidth((int) value);
      }
    });

    m_smoothSliderPanel = new SliderPanel(0.f, 1.f, 0.f, 1);
    ((FlowLayout) m_smoothSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    m_smoothSliderPanel.addInfoText("Halo Smooth Factor");
    m_smoothSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        setHighlightSmooth(value);
      }
    });

    m_visibilityCheckBox = new JCheckBox("Halo Always Visible");
    m_visibilityCheckBox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        setHighlightAlwaysVisible(m_visibilityCheckBox.isSelected());
      }
    });

    JButton colorChoose = new JButton();
    colorChoose.setPreferredSize(new Dimension(250, 30));
    colorChoose.setText("Highlight Color");
    colorChoose.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        JColorChooser colorChooser = new JColorChooser();
        colorChooser.setPreviewPanel(new JPanel());
        colorChooser.setColor(m_highlightColor);
        JDialog colorDial =
            JColorChooser.createDialog(colorChoose, "Choose highlight color", true, colorChooser, new ActionListener()
            {
              @Override
              public void actionPerformed(ActionEvent e)
              {
                m_highlightColor = colorChooser.getColor();
                setHighlightColor(new SbColor(m_highlightColor));
              }
            }, null);
        colorDial.setVisible(true);
      }
    });

    String[] highlightTypes = { "LINE", "BOX", "COLOR", "HALO" };
    m_highlightTypesCombo = new JComboBox<String>(highlightTypes);
    m_highlightTypesCombo.setSelectedIndex(3);
    m_highlightTypesCombo.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        updateHighlightAction();
      }
    });

    JPanel highlightPanel = new JPanel();
    highlightPanel.setLayout(new FlowLayout(FlowLayout.LEFT, 5, 5));
    highlightPanel.add(new JLabel("Highlight Style:"));
    highlightPanel.add(m_highlightTypesCombo);
    highlightPanel.add(colorChoose);
    highlightPanel.add(m_widthSliderPanel);
    highlightPanel.add(m_smoothSliderPanel);
    highlightPanel.add(m_visibilityCheckBox);

    return highlightPanel;
  }

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("Highlighting Styles");
  }
}
