package inventor.advanced.pointcloud;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JCheckBox;
import javax.swing.JPanel;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec2s;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoPickedPoint;
import com.openinventor.inventor.SoSceneManager;
import com.openinventor.inventor.SoSceneManager.AntialiasingModes;
import com.openinventor.inventor.actions.SoHandleEventAction;
import com.openinventor.inventor.actions.SoRayPickAction;
import com.openinventor.inventor.details.SoDetail;
import com.openinventor.inventor.details.SoPointDetail;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoMouseButtonEvent;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.SoAnnotation;
import com.openinventor.inventor.nodes.SoCamera;
import com.openinventor.inventor.nodes.SoDrawStyle;
import com.openinventor.inventor.nodes.SoEventCallback;
import com.openinventor.inventor.nodes.SoFont;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoOrthographicCamera;
import com.openinventor.inventor.nodes.SoPickStyle;
import com.openinventor.inventor.nodes.SoPointSet;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoShaderObject;
import com.openinventor.inventor.nodes.SoShaderParameter1f;
import com.openinventor.inventor.nodes.SoShaderParameter1i;
import com.openinventor.inventor.nodes.SoShaderProgram;
import com.openinventor.inventor.nodes.SoText2;
import com.openinventor.inventor.nodes.SoTextProperty;
import com.openinventor.inventor.nodes.SoVertexProperty;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;

import util.Example;
import util.ViewerComponentsFactory;

/**
 * Procedurally generated point cloud using a custom shader in order to vary the
 * size of the points. In this context, the point cloud is used to materialize a
 * surface.
 */
public class Main extends Example
{

  public class Scene
  {
    final int NUM_POINTS_PER_AXIS = 1000;
    final float POINT_SPACE = 10.0f;
    SbVec3f m_origin;

    SoDrawStyle m_drawStyle;
    SoShaderParameter1i m_spSizeVarying;
    SoShaderParameter1i m_spColorVarying;
    SoSeparator m_root;
    SoText2 m_text;

    public Scene()
    {
      m_origin = new SbVec3f(NUM_POINTS_PER_AXIS * POINT_SPACE, 0.0f, NUM_POINTS_PER_AXIS * POINT_SPACE).over(2.0f);

      final int numPoints = NUM_POINTS_PER_AXIS * NUM_POINTS_PER_AXIS;

      SbVec3f[] points = new SbVec3f[numPoints];
      SbVec3f[] normals = new SbVec3f[numPoints];

      int pIndex = 0;
      for ( int j = 0; j < NUM_POINTS_PER_AXIS; ++j )
      {
        for ( int i = 0; i < NUM_POINTS_PER_AXIS; ++i )
        {
          points[pIndex] = getPoint(i, j);
          SbVec3f normal = computeNormal(points[pIndex], getPoint(i - 1, j - 1), getPoint(i + 1, j - 1));
          normal.normalize();
          normals[pIndex] = normal;
          pIndex++;
        }
      }

      m_root = new SoSeparator();
      m_drawStyle = new SoDrawStyle();
      m_drawStyle.pointSize.setValue(1.0f);
      m_root.addChild(m_drawStyle);

      SoMaterial material = new SoMaterial();
      material.diffuseColor.setValue(1.0f, 1.0f, 1.0f);
      m_root.addChild(material);

      SoShaderProgram shader = new SoShaderProgram();
      SoShaderObject shaderObject =
          shader.setVertexShader(0, "$OIVJHOME/examples/inventor/advanced/pointcloud/shaders/Sprite_vert.glsl");
      shader.setFragmentShader(1, "$OIVJHOME/examples/inventor/advanced/pointcloud/shaders/Sprite_frag.glsl");
      m_root.addChild(shader);

      SoShaderParameter1f pointSizeParam = new SoShaderParameter1f();
      pointSizeParam.name.setValue("basePointSize");
      pointSizeParam.value.setValue(1.0f);
      pointSizeParam.value.connectFrom(m_drawStyle.pointSize);
      shaderObject.parameter.set1Value(1, pointSizeParam);

      m_spSizeVarying = new SoShaderParameter1i();
      m_spSizeVarying.name.setValue("sizeVarying");
      m_spSizeVarying.value.setValue(1);
      shaderObject.parameter.set1Value(2, m_spSizeVarying);

      m_spColorVarying = new SoShaderParameter1i();
      m_spColorVarying.name.setValue("colorVarying");
      m_spColorVarying.value.setValue(1);
      shaderObject.parameter.set1Value(3, m_spColorVarying);

      SoPointSet ps = new SoPointSet();
      SoVertexProperty vp = new SoVertexProperty();
      ps.vertexProperty.setValue(vp);

      vp.vertex.setValues(0, points);
      vp.normal.setValues(0, normals);

      vp.materialBinding.setValue(SoVertexProperty.Bindings.PER_VERTEX);

      m_root.addChild(ps);

      SoSeparator annotationSep = new SoSeparator();
      SoAnnotation annotation = new SoAnnotation();
      SoOrthographicCamera camera = new SoOrthographicCamera();
      camera.nearDistance.setValue(-1.0f);
      camera.farDistance.setValue(1.0f);
      camera.position.setValue(0.99f, -0.95f, 0.0f);
      camera.viewportMapping.setValue(SoCamera.ViewportMappings.LEAVE_ALONE);
      annotation.addChild(camera);

      SoMaterial textMaterial = new SoMaterial();
      textMaterial.diffuseColor.setValue(1.0f, 0.0f, 0.0f);
      annotation.addChild(textMaterial);

      SoPickStyle pickStyle = new SoPickStyle();
      pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
      annotation.addChild(pickStyle);

      SoTextProperty textProperty = new SoTextProperty();
      textProperty.style.setValue(SoTextProperty.Styles.BACK_FRAME);
      textProperty.styleColors.set1Value(SoTextProperty.StyleColorType.BACK_FRAME_COLOR.ordinal(), 1.0f, 1.0f, 1.0f, 1.0f);
      annotation.addChild(textProperty);

      SoFont fontNode = new SoFont();
      fontNode.size.setValue(15);
      fontNode.renderStyle.setValue(SoFont.RenderStyles.TEXTURE);
      annotation.addChild(fontNode);

      m_text = new SoText2();
      annotation.addChild(m_text);
      m_text.string.setValue("Picked point: none");

      annotationSep.addChild(annotation);

      m_root.addChild(annotationSep);
    }

    public void setText(String text)
    {
      m_text.string.setValue(text);
    }

    public SoSeparator getRoot()
    {
      return m_root;
    }

    public SoDrawStyle getDrawStyle()
    {
      return m_drawStyle;
    }

    public SoShaderParameter1i getSpSizeVarying()
    {
      return m_spSizeVarying;
    }

    public SoShaderParameter1i getSpColorVarying()
    {
      return m_spColorVarying;
    }

    SbVec3f computeNormal(SbVec3f p0, SbVec3f p1, SbVec3f p2)
    {
      SbVec3f vec0 = p1.minus(p0);
      SbVec3f vec1 = p2.minus(p0);
      return vec0.cross(vec1);
    }

    SbVec3f getPoint(int i, int j)
    {
      SbVec3f p = (new SbVec3f(i * 10.0f, 0.0f, j * 10.0f)).minus(m_origin);
      p.setY(NUM_POINTS_PER_AXIS / 1.0f
          * (float) (Math.cos((p.minus(m_origin)).length() / (NUM_POINTS_PER_AXIS * 10) * Math.PI)));
      return p;
    }
  }

  private IRenderAreaInteractive m_viewer;

  private SliderPanel m_pointSizeSlider;
  private JCheckBox m_pointSizeVaryingCheckbox;
  private JCheckBox m_colorVaryingCheckbox;
  private Scene m_scene;

  @Override
  public void start()
  {
    setLayout(new BorderLayout());

    add(buildOptionsPanel(), BorderLayout.CENTER);

    m_viewer = ViewerComponentsFactory.createRenderAreaOrbiter();

    SoSeparator root = new SoSeparator();

    final float grayVal = 0.99f;
    final SbColor BKG_COLOR = new SbColor(grayVal, grayVal, grayVal);
    SoGradientBackground gradientBackground = new SoGradientBackground();
    gradientBackground.color0.setValue(BKG_COLOR);
    gradientBackground.color1.setValue(BKG_COLOR);
    root.addChild(gradientBackground);

    m_scene = new Scene();
    root.addChild(m_scene.getRoot());

    SoEventCallback rayPickCB = new SoEventCallback();
    rayPickCB.addEventCallback(SoMouseButtonEvent.class, new RayPickCallback(), m_scene);
    root.addChild(rayPickCB);

    m_viewer.setSceneGraph(root);
    m_viewer.viewAll(new SbViewportRegion());

    m_viewer.setAntialiasingMode(AntialiasingModes.AUTO);
    m_viewer.setAntialiasingQuality(1);

    final Component component = m_viewer.getComponent();
    component.setPreferredSize(new Dimension(1280, 720));
    add(component, BorderLayout.SOUTH);

    SoCamera cam = m_viewer.getSceneInteractor().getCamera();
    cam.position.setValue(new SbVec3f(5004.26f, -895.473f, -3690.13f));
    cam.orientation.setValue(new SbRotation(new SbVec3f(0.0571383f, -0.994446f, -0.0883862f), 4.28429f));
    cam.focalDistance.setValue(5000.0f);

    m_pointSizeSlider.setSliderValue(2.0f);
    m_pointSizeVaryingCheckbox.setSelected(true);
    m_colorVaryingCheckbox.setSelected(true);
  }

  @Override
  public void stop()
  {
    m_viewer.dispose();
  }

  class RayPickCallback extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback cb)
    {
      SoEvent event = cb.getEvent();
      if ( !SoMouseButtonEvent.isButtonPressEvent(event, SoMouseButtonEvent.Buttons.BUTTON1) )
        return;

      SbVec2s pos = event.getPosition();
      SoHandleEventAction action = cb.getAction();
      SbViewportRegion viewport = action.getViewportRegion();

      SoRayPickAction rpa = new SoRayPickAction(viewport);
      SoSceneManager sceneManager = action.getSceneManager();
      rpa.setSceneManager(sceneManager);
      rpa.setPoint(pos);

      Scene scene = (Scene) userData;
      rpa.apply(sceneManager.getSceneGraph());
      SoPickedPoint pickedPoint = rpa.getPickedPoint();
      SoDetail detail = (pickedPoint != null) ? pickedPoint.getDetail() : null;

      String baseStr = "Picked point: ";
      SoPointDetail pointDetail = (detail instanceof SoPointDetail) ? (SoPointDetail) detail : null;
      String pickedStr = "none";
      if ( pointDetail != null )
        pickedStr = "#" + String.valueOf(pointDetail.getCoordinateIndex());

      scene.setText(baseStr + pickedStr);
    }
  }

  private JPanel buildOptionsPanel()
  {
    // Build Highlight parameters panel
    m_pointSizeSlider = new SliderPanel(0.1f, 20.0f, 2.0f, 1);
    ((FlowLayout) m_pointSizeSlider.getLayout()).setAlignment(FlowLayout.LEFT);
    m_pointSizeSlider.addInfoText("Reference point Size");
    m_pointSizeSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_scene.getDrawStyle().pointSize.setValue(value);
      }
    });

    m_pointSizeVaryingCheckbox = new JCheckBox("Varying point size");
    m_pointSizeVaryingCheckbox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        m_scene.getSpSizeVarying().value.setValue(m_pointSizeVaryingCheckbox.isSelected() ? 1 : 0);
      }
    });

    m_colorVaryingCheckbox = new JCheckBox("Varying color");
    m_colorVaryingCheckbox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        m_scene.getSpColorVarying().value.setValue(m_colorVaryingCheckbox.isSelected() ? 1 : 0);
      }
    });

    JPanel highlightPanel = new JPanel();
    highlightPanel.setLayout(new FlowLayout(FlowLayout.LEFT, 5, 5));
    highlightPanel.add(m_pointSizeSlider);
    highlightPanel.add(m_pointSizeVaryingCheckbox);
    highlightPanel.add(m_colorVaryingCheckbox);

    return highlightPanel;
  }

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("Point Cloud");
  }
}
