/*------------------------------------------------------------
 *  This is an example from The Inventor Mentor
 *  chapter 10, example 3.
 *
 *  The scene graph has 4 objects which may be
 *  selected by picking with the left mouse button
 *  (use shift key to extend the selection to more
 *  than one object).
 *
 *  Hitting the up arrow key will increase the size of
 *  each selected object; hitting down arrow will decrease
 *  the size of each selected object.
 *
 *  This also demonstrates selecting objects from a Motif
 *  list, and calling select/deselect functions on the
 *  SoSelection node to change the selection. Use the Shift
 *  key to extend the selection (i.e. pick more than one
 *  item in the motif list.)
 *-----------------------------------------------------------*/

package inventor.mentor.awtList;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dialog;
import java.awt.Frame;
import java.awt.List;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.JTextArea;
import javax.swing.border.EtchedBorder;

import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.actions.SoBoxHighlightRenderAction;
import com.openinventor.inventor.actions.SoSearchAction;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.misc.callbacks.SoSelectionClassCB;
import com.openinventor.inventor.misc.callbacks.SoSelectionPathCB;
import com.openinventor.inventor.nodes.* ;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;

/**
  * mentor.AWTList example program. Using a list of the AWT and using the picking (mouse),
  * highlight of nodes in a render area.
  *
  * @author Loic Vigneras
  * @author Patrick Vigneras
  * @author Jean Kinda
  */

public class Main extends Example {

  private IViewerExaminer myViewer;

  SbViewportRegion viewportRegion = new SbViewportRegion() ;
  Dialog objectList;
  List list;
  SoTransform cubeTransform, sphereTransform, coneTransform, cylTransform;

  ///////////////////////////////////////////////////////////////
  // CODE FOR The Inventor Mentor STARTS HERE
  final int CUBE = 0 ;
  final int SPHERE = 1 ;
  final int CONE = 2 ;
  final int CYL = 3 ;
  final int NUM_OBJECTS = 4 ;

  String objectNames[] = {"Cube", "Sphere", "Cone", "Cylinder"} ;
  MySelectionCB mySelectionCB = new MySelectionCB() ;
  MySelectionCB myDeselectionCB = new MySelectionCB() ;

  public static void main (String [] argv)
  {
    Main example = new Main();
    example.demoMain("AWT List");
  }

  @Override
  public void start() {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    // Usage message
    JTextArea message_area = new JTextArea();
    message_area.setBorder(new EtchedBorder());
    message_area.append("Left mouse button \t - selects object\n");
    message_area.append("<shift>Left mouse button \t - selects multiple objects\n");
    message_area.append("Up and Down arrows \t - scale selected objects");

    // Create and set up the selection node
    SoSelection selectionRoot = new SoSelection() ;
    selectionRoot.policy.setValue(SoSelection.Policies.SHIFT);
    selectionRoot.addSelectionCallback(mySelectionCB, Boolean.valueOf(true));
    selectionRoot.addDeselectionCallback(myDeselectionCB, Boolean.valueOf(false));
    selectionRoot.addChangeCallback(new SoSelectionClassCB()
    {
      @Override
      public void invoke(SoSelection s)
      {
        myViewer.getRenderArea().scheduleRedraw();
      }
    });

    // Add an event callback so we can receive key press events
    SoEventCallback myEventCB = new SoEventCallback();
    myEventCB.addEventCallback("com.openinventor.inventor.events.SoKeyboardEvent",
			       new MyKeyPressCB(), selectionRoot);
    selectionRoot.addChild(myEventCB);

    // Add some geometry to the scene
    // a red cube
    SoSeparator cubeRoot = new SoSeparator();
    SoMaterial cubeMaterial = new SoMaterial();
    cubeTransform = new SoTransform();
    SoCube cube = new SoCube();
    cubeRoot.addChild(cubeTransform);
    cubeRoot.addChild(cubeMaterial);
    cubeRoot.addChild(cube);
    cubeTransform.translation.setValue(-2, 2, 0);
    cubeMaterial.diffuseColor.setValue(0.8F, 0, 0);
    selectionRoot.addChild(cubeRoot);

    // a blue sphere
    SoSeparator sphereRoot = new SoSeparator();
    SoMaterial sphereMaterial = new SoMaterial();
    sphereTransform = new SoTransform();
    SoSphere sphere = new SoSphere();
    sphereRoot.addChild(sphereTransform);
    sphereRoot.addChild(sphereMaterial);
    sphereRoot.addChild(sphere);
    sphereTransform.translation.setValue(2, 2, 0);
    sphereMaterial.diffuseColor.setValue(0, 0, 0.8F);
    selectionRoot.addChild(sphereRoot);

    // a green cone
    SoSeparator coneRoot = new SoSeparator();
    SoMaterial coneMaterial = new SoMaterial();
    coneTransform = new SoTransform();
    SoCone cone = new SoCone();
    coneRoot.addChild(coneTransform);
    coneRoot.addChild(coneMaterial);
    coneRoot.addChild(cone);
    coneTransform.translation.setValue(2, -2, 0);
    coneMaterial.diffuseColor.setValue(0, 0.8F, 0);
    selectionRoot.addChild(coneRoot);

    // a magenta cylinder
    SoSeparator cylRoot = new SoSeparator();
    SoMaterial cylMaterial = new SoMaterial();
    cylTransform = new SoTransform();
    SoCylinder cyl = new SoCylinder();
    cylRoot.addChild(cylTransform);
    cylRoot.addChild(cylMaterial);
    cylRoot.addChild(cyl);
    cylTransform.translation.setValue(-2, -2, 0);
    cylMaterial.diffuseColor.setValue(0.8F, 0, 0.8F);
    selectionRoot.addChild(cylRoot);

    ///////////////////////////////////////////////////////////////
    // CODE FOR The Inventor Mentor STARTS HERE  (part 2)

    cube.setName(objectNames[CUBE]);
    sphere.setName(objectNames[SPHERE]);
    cone.setName(objectNames[CONE]);
    cyl.setName(objectNames[CYL]);

    // CODE FOR The Inventor Mentor ENDS HERE
    ///////////////////////////////////////////////////////////////

    myViewer.getRenderArea().setGLRenderAction(new SoBoxHighlightRenderAction());
    myViewer.setSceneGraph(selectionRoot);

    // Make the camera see the whole scene
    myViewer.viewAll();

    // Create a list for selecting objects without picking

    List objectList = new List();

    ///////////////////////////////////////////////////////////////
    // CODE FOR The Inventor Mentor STARTS HERE  (part 3)

    // Create the list
    list = new List(NUM_OBJECTS, true) ;
    for (int i=0; i<NUM_OBJECTS ; i++) objectList.add(objectNames[i]) ;

    objectList.addItemListener(new ListPick(selectionRoot)) ;

    // CODE FOR The Inventor Mentor ENDS HERE
    ///////////////////////////////////////////////////////////////

    final Component component = myViewer.getComponent();
    component.setName("Selection List");
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(objectList, BorderLayout.WEST);
    add(message_area, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  // Create the object list widget
  Dialog createList(Frame f, SoSelection selection) {
    Dialog objectList = new Dialog(f, "List of Objects", false) ;

    ///////////////////////////////////////////////////////////////
    // CODE FOR The Inventor Mentor STARTS HERE  (part 3)

    // Create the list
    list = new List(NUM_OBJECTS, true) ;
    for (int i=0; i<NUM_OBJECTS ; i++) list.add(objectNames[i]) ;

    list.addItemListener(new ListPick(selection)) ;

    // CODE FOR The Inventor Mentor ENDS HERE
    ///////////////////////////////////////////////////////////////

    objectList.add(list) ;
    objectList.pack() ;
    return objectList;
  }


  /////////////////////////////////////////////////////////////////
  // inner classes : listeners and callbacks
  /////////////////////////////////////////////////////////////////


  class ListPick implements ItemListener {
    SoSelection selection ;
    SoSearchAction mySearchAction = new SoSearchAction() ;

    public ListPick(SoSelection s) {
      selection = s ;
    }

    @Override
    public void itemStateChanged(ItemEvent e) {
      List list = (List) e.getSource() ;

      // Remove the selection callbacks so that we don't get
      // called back while we are updating the selection list
      selection.removeSelectionCallback(mySelectionCB, null);
      selection.removeDeselectionCallback(myDeselectionCB, null);

      //////////////////////////////////////////////////////////////
      // CODE FOR The Inventor Mentor STARTS HERE  (part 4)

      // Clear the selection node, then loop through the list and reselect
      selection.deselectAll();

      // Update the SoSelection based on what is selected in the list.
      String selected [] = list.getSelectedItems() ;
      for (int i=0 ; i<selected.length ; i++) {
	mySearchAction.setName(selected[i]) ;
	mySearchAction.apply(selection);
	selection.select(mySearchAction.getPath());
      }

      // CODE FOR The Inventor Mentor ENDS HERE
      //////////////////////////////////////////////////////////////

      // Add the selection callbacks again
      selection.addSelectionCallback(mySelectionCB, Boolean.valueOf(true));
      selection.addDeselectionCallback(myDeselectionCB, Boolean.valueOf(false));
    }
  }


  // This is called whenever an object is selected or deselected.
  // if userData is TRUE, then it's a selection; else deselection.
  // (we set this convention up when we registered this callback).
  // The function updates the Motif list to reflect the current
  // selection.
  class MySelectionCB extends SoSelectionPathCB {
    @Override
    public void invoke (SoPath selectionPath) {
      boolean isSelection = ((Boolean) userData).booleanValue() ;

      SoNode node = selectionPath.regular.getTail();

      for (int i=0 ; i<NUM_OBJECTS ; i++) {
        if (node.getName().equals(objectNames[i])) {
            //	if (SbName.opeq(node.getName(), objectNames[i])) {
	  if ((isSelection) && (!list.isIndexSelected(i)))
	    list.select(i) ;
	  else if ((!isSelection) && (list.isIndexSelected(i)))
	    list.deselect(i);
	  break;
	}
      }
    }
  }

  // If the event in the graphics window is down arrow, then
  // scale every object in the selection list by 1/2; if the
  // event is up arrow, scale by 2.
  // userData is the selectionRoot from main().

  class MyKeyPressCB extends SoEventCallbackCB {
    @Override
    public void invoke (SoEventCallback eventCB) {
      SoSelection selection = (SoSelection) userData;
      SoEvent event = eventCB.getEvent();

      // check for the Up and Down arrow keys being pressed
      if (SoKeyboardEvent.isKeyPressEvent(event, SoKeyboardEvent.Keys.UP_ARROW)) {
	myScaleSelection(selection, 1.1F);
	eventCB.setHandled();
      } else if (SoKeyboardEvent.isKeyPressEvent(event, SoKeyboardEvent.Keys.DOWN_ARROW)) {
	myScaleSelection(selection, (float)(1.0/1.1));
	eventCB.setHandled();
      }
    }

    //////////////////////////////////////////////////////////////
    // CODE FOR The Inventor Mentor STARTS HERE  (Example 10-4)

    // Scale each object in the selection list
    void myScaleSelection(SoSelection selection, float sf) {
      SoPath selectedPath;
      SoTransform xform;
      SbVec3f scaleFactor = new SbVec3f() ;
      int i,j;

      // Scale each object in the selection list
      for (i=0 ; i<selection.getNumSelected() ; i++) {
	selectedPath = selection.getPath(i);
	xform = null;
	// Look for the shape node, starting from the tail of the
	// path.  Once we know the type of shape, we know which
	// transform to modify
	for (j=0 ; (j<selectedPath.regular.getLength()) && (xform == null) ; j++) {
	  SoNode n = selectedPath.regular.getNodeFromTail(j);
	  if (n instanceof SoCube)
	    xform = cubeTransform;
	  else if (n instanceof SoCone)
            xform = coneTransform;
	  else if (n instanceof SoSphere)
            xform = sphereTransform;
	  else if (n instanceof SoCylinder)
            xform = cylTransform;
	}

	// Apply the scale
	scaleFactor = xform.scaleFactor.getValue();
	scaleFactor.multiply(sf) ;
	xform.scaleFactor.setValue(scaleFactor);
      }
    }
    // CODE FOR The Inventor Mentor ENDS HERE
    //////////////////////////////////////////////////////////////
  }
}

