/*----------------------------------------------------------------
 *  This is an example from the Inventor Mentor,
 *  chapter 14, example 3.
 *
 *  This example illustrates the creation of motion hierarchies
 *  using nodekits by creating a model of a balance-style scale.
 *
 *  It adds an SoEventCallback to the "callback" list in the
 *      nodekit called 'support.'
 *  The callback will have the following response to events:
 *  Pressing right arrow key == lower the right pan
 *  Pressing left arrow key  == lower the left pan
 *  The pans are lowered by animating three rotations in the
 *      motion hierarchy.
 *  Use an SoText2Kit to print instructions to the user as part
 *      of the scene.
 *----------------------------------------------------------------*/


package inventor.mentor.balance;

import java.awt.BorderLayout;
import java.awt.Component;

import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodekits.SoSeparatorKit;
import com.openinventor.inventor.nodekits.SoShapeKit;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example {

  private IRenderAreaInteractive myRenderArea;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("Balance Scale Made of Nodekits");
  }

  @Override
  public void start() {
    myRenderArea = ViewerComponentsFactory.createRenderAreaInteractive();

    SoSeparatorKit myScene = new SoSeparatorKit();

    // Create the Balance Scale -- put each part in the
    // childList of its parent, to build up this hierarchy:
    //
    //                    myScene
    //                       |
    //                     support
    //                       |
    //                     beam
    //                       |
    //                   --------
    //                   |       |
    //                string1  string2
    //                   |       |
    //                tray1     tray2

    SoShapeKit support = new SoShapeKit();
    support.setPart("shape", new SoCone());
    support.set("shape { height 3 bottomRadius .3 }");
    myScene.setPart("childList[0]", support);

    SoShapeKit beam = new SoShapeKit();
    beam.setPart("shape", new SoCube());
    beam.set("shape { width 3 height .2 depth .2 }");
    beam.set("transform { translation 0 1.5 0 } ");
    support.setPart("childList[0]", beam);

    SoShapeKit string1 = new SoShapeKit();
    string1.setPart("shape", new SoCylinder());
    string1.set("shape { radius .05 height 2}");
    string1.set("transform { translation -1.5 -1 0 }");
    string1.set("transform { center 0 1 0 }");
    beam.setPart("childList[0]", string1);

    SoShapeKit string2 = new SoShapeKit();
    string2.setPart("shape", new SoCylinder());
    string2.set("shape { radius .05 height 2}");
    string2.set("transform { translation 1.5 -1 0 } ");
    string2.set("transform { center 0 1 0 } ");
    beam.setPart("childList[1]", string2);

    SoShapeKit tray1 = new SoShapeKit();
    tray1.setPart("shape", new SoCylinder());
    tray1.set("shape { radius .75 height .1 }");
    tray1.set("transform { translation 0 -1 0 } ");
    string1.setPart("childList[0]", tray1);

    SoShapeKit tray2 = new SoShapeKit();
    tray2.setPart("shape", new SoCylinder());
    tray2.set("shape { radius .75 height .1 }");
    tray2.set("transform { translation 0 -1 0 } ");
    string2.setPart("childList[0]", tray2);

    // Add EventCallback so Balance Responds to Events
    SoEventCallback myCallbackNode = new SoEventCallback();
    myCallbackNode.addEventCallback(SoKeyboardEvent.class, new TipTheBalance(support));
    support.setPart("callbackList[0]", myCallbackNode);

    // Add Instructions as Text in the Scene...
    SoShapeKit myText = new SoShapeKit();
    myText.setPart("shape", new SoText2());
    myText.set("shape { string \"Press Left or Right Arrow Key\" }");
    myText.set("shape { justification CENTER }");
    myText.set("font { name \"Helvetica-Bold\" }");
    myText.set("font { size 16.0 }");
    myText.set("transform { translation 0 -2 0 }");
    myScene.setPart("childList[1]", myText);

    SoSeparator root = new SoSeparator();
    { // Assemble scene graph
      root.addChild(myScene);
    }

    final Component canvas = myRenderArea.getComponent();

    myRenderArea.setSceneGraph(root);
    myRenderArea.viewAll(new SbViewportRegion(myRenderArea.getComponent().getSize()));

    canvas.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(canvas);
  }

  @Override
  public void stop()
  {
    myRenderArea.dispose();
  }

  class TipTheBalance extends SoEventCallbackCB {

    SoShapeKit m_shape;

    public TipTheBalance(SoShapeKit sh) {
      m_shape = sh;
    }

    @Override
    public void invoke(SoEventCallback eventCB) {
      SoEvent ev = eventCB.getEvent();

      // Which Key was pressed?
      // If Right or Left Arrow key, then continue...
      if (SoKeyboardEvent.isKeyPressEvent(ev, SoKeyboardEvent.Keys.RIGHT_ARROW) ||
          SoKeyboardEvent.isKeyPressEvent(ev, SoKeyboardEvent.Keys.LEFT_ARROW)) {
        SoShapeKit  beam1, string1, string2;
        SbRotation  startRot;
        SbRotation beamIncrement = new SbRotation();
        SbRotation stringIncrement = new SbRotation();

        // Get the different nodekits from the userData.
        // These three parts are extracted based on knowledge of the
        // motion hierarchy (see the diagram in the main routine.
        beam1 = (SoShapeKit)m_shape.getPart("childList[0]",true);
        string1 = (SoShapeKit)beam1.getPart("childList[0]",true);
        string2 = (SoShapeKit)beam1.getPart("childList[1]",true);

        // Set angular increments to be .1 Radians about the Z-Axis
        // The strings rotate opposite the beam, and the two types
        // of key press produce opposite effects.
        if (SoKeyboardEvent.isKeyPressEvent(ev, SoKeyboardEvent.Keys.RIGHT_ARROW)) {
          beamIncrement.setValue(new SbVec3f(0.0f, 0.0f, 1.0f), -0.1f);
          stringIncrement.setValue(new SbVec3f(0.0f, 0.0f, 1.0f), 0.1f);
        }
        else {
          beamIncrement.setValue(new SbVec3f(0.0f, 0.0f, 1.0f), 0.1f);
          stringIncrement.setValue(new SbVec3f(0.0f, 0.0f, 1.0f), -0.1f);
        }

        // Use getPart to find the transform for each of the
        // rotating parts and modify their rotations.

        SoTransform xf = (SoTransform)beam1.getPart("transform");
        startRot = xf.rotation.getValue();
        xf.rotation.setValue(startRot.times(beamIncrement));

        xf = (SoTransform)string1.getPart("transform");
        startRot = xf.rotation.getValue();
        xf.rotation.setValue(startRot.times(stringIncrement));

        xf = (SoTransform)string2.getPart("transform");
        startRot = xf.rotation.getValue();
        xf.rotation.setValue(startRot.times(stringIncrement));

        eventCB.setHandled();
      }
    }
  }
}
