/*--------------------------------------------------------------
 *  This is an example from the Inventor Mentor
 *  chapter 13, example 6.
 *
 *  Boolean engine.  Derived from example 13.5.
 *  The smaller duck stays still while the bigger duck moves,
 *  and starts moving as soon as the bigger duck stops.
 *------------------------------------------------------------*/

package inventor.mentor.booleanEngine;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Font;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;

import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.engines.SoBoolOperation;
import com.openinventor.inventor.engines.SoElapsedTime;
import com.openinventor.inventor.engines.SoGate;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoMouseButtonEvent;
import com.openinventor.inventor.fields.SoMFFloat;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example {

  private IRenderAreaInteractive myRenderArea;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("Duck and Duckling");
  }

  @Override
  public void start() {
    myRenderArea = ViewerComponentsFactory.createRenderAreaInteractive();

    // set up camera
    SoPerspectiveCamera myCamera = (SoPerspectiveCamera) myRenderArea.getSceneInteractor().getCamera();
    myCamera.position.setValue(0.0f, -4.0f, 8.0f);
    myCamera.heightAngle.setValue( (float) (Math.PI / 2.5f));
    myCamera.nearDistance.setValue(1.0f);
    myCamera.farDistance.setValue(15.0f);

    // Rotate scene slightly to get better view
    SoRotationXYZ globalRotXYZ = new SoRotationXYZ();
    globalRotXYZ.axis.setValue(SoRotationXYZ.AxisType.X);
    globalRotXYZ.angle.setValue( (float) (Math.PI / 9.0f));

    // Pond group
    SoSeparator pond = new SoSeparator();
    SoTranslation pondTranslation = new SoTranslation();
    pondTranslation.translation.setValue(0.0f, -6.725f, 0.0f);
    // water
    SoMaterial waterMaterial = new SoMaterial();
    waterMaterial.diffuseColor.setValue(0.0f, 0.3f, 0.8f);
    SoCylinder waterCylinder = new SoCylinder();
    waterCylinder.radius.setValue(4.0f);
    waterCylinder.height.setValue(0.5f);
    // rock
    SoMaterial rockMaterial = new SoMaterial();
    rockMaterial.diffuseColor.setValue(0.8f, 0.23f, 0.03f);
    SoSphere rockSphere = new SoSphere();
    rockSphere.radius.setValue(0.9f);
    {
      pond.addChild(pondTranslation);
      pond.addChild(waterMaterial);
      pond.addChild(waterCylinder);
      pond.addChild(rockMaterial);
      pond.addChild(rockSphere);
    }

    // Read the duck object from a file and add to the group
    SoInput myInput = new SoInput();
    String filename = "$OIVJHOME/data/models/duck.iv";
    if (!myInput.openFile(filename)) {
      System.err.println("Cannot open file " + filename);
      System.exit(1);
    }

    SoSeparator duckObject = SoDB.readAll(myInput);
    if (duckObject == null) {
      System.err.println("Cannot read file " + filename);
      System.exit(1);
    }

    /////////////////////////////////////////////////////////////
    // CODE FOR The Inventor Mentor STARTS HERE

    // Bigger duck group
    SoSeparator bigDuck = new SoSeparator();
    SoRotationXYZ bigDuckRotXYZ = new SoRotationXYZ();
    SoTransform bigInitialTransform = new SoTransform();
    bigInitialTransform.translation.setValue(0.0f, 0.0f, 3.5f);
    bigInitialTransform.scaleFactor.setValue(6.0f, 6.0f, 6.0f);
    {
      bigDuck.addChild(bigDuckRotXYZ);
      bigDuck.addChild(bigInitialTransform);
      bigDuck.addChild(duckObject);
    }

    // Smaller duck group
    SoSeparator smallDuck = new SoSeparator();
    SoRotationXYZ smallDuckRotXYZ = new SoRotationXYZ();
    SoTransform smallInitialTransform = new SoTransform();
    smallInitialTransform.translation.setValue(0.0f, -2.24f, 1.5f);
    smallInitialTransform.scaleFactor.setValue(4.0f, 4.0f, 4.0f);
    {
      smallDuck.addChild(smallDuckRotXYZ);
      smallDuck.addChild(smallInitialTransform);
      smallDuck.addChild(duckObject);
    }

    // Use a gate engine to start/stop the rotation of
    // the bigger duck.
    SoEventCallback myEventCB = new SoEventCallback();
    SoBoolOperation myBoolean = new SoBoolOperation();
    try {
      SoGate bigDuckGate = new SoGate(SoMFFloat.class);
      SoElapsedTime bigDuckTime = new SoElapsedTime();
      bigDuckGate.input.connectFrom(bigDuckTime.timeOut);
      bigDuckRotXYZ.axis.setValue(SoRotationXYZ.AxisType.Y); // Y axis
      bigDuckRotXYZ.angle.connectFrom(bigDuckGate.output);

      // Each mouse button press will enable/disable the gate
      // controlling the bigger duck.
      myEventCB.addEventCallback(SoMouseButtonEvent.class,
                                 new MousePressCB(bigDuckGate));

      // Use a Boolean engine to make the rotation of the smaller
      // duck depend on the bigger duck.  The smaller duck moves
      // only when the bigger duck is still.
      myBoolean.a.connectFrom(bigDuckGate.enable);
      myBoolean.operation.setValue(SoBoolOperation.Operations.NOT_A.getValue());
    }
    catch (Exception e) {
      e.printStackTrace();
    }

    try {
      SoGate smallDuckGate = new SoGate(SoMFFloat.class);
      SoElapsedTime smallDuckTime = new SoElapsedTime();
      smallDuckGate.input.connectFrom(smallDuckTime.timeOut);
      smallDuckGate.enable.connectFrom(myBoolean.output);
      smallDuckRotXYZ.axis.setValue(SoRotationXYZ.AxisType.Y); // Y axis
      smallDuckRotXYZ.angle.connectFrom(smallDuckGate.output);
    }
    catch (Exception e) {
      e.printStackTrace();
    }

    // CODE FOR The Inventor Mentor ENDS HERE
    /////////////////////////////////////////////////////////////

    SoSeparator root = new SoSeparator();
    { // Assemble scene graph
      root.addChild(globalRotXYZ);
      root.addChild(pond);
      root.addChild(bigDuck);
      root.addChild(smallDuck);
      root.addChild(myEventCB);
    }
    myRenderArea.setSceneGraph(root);

    Font f = new Font(null, 1, 15);
    JLabel label1 = new JLabel("Only one duck can move at a time.", SwingConstants.CENTER);
    label1.setFont(f);
    JLabel label2 = new JLabel("Click the left mouse button to toggle between the two ducks.", SwingConstants.CENTER);
    label2.setFont(f);
    JPanel label_panel = new JPanel(new BorderLayout());
    label_panel.add(label1, BorderLayout.NORTH);
    label_panel.add(label2, BorderLayout.SOUTH);

    final Component canvas = myRenderArea.getComponent();
    canvas.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(canvas);
    add(label_panel, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    myRenderArea.dispose();
  }

  // This routine is called for every mouse button event.
  class MousePressCB extends SoEventCallbackCB {

    private SoGate m_gate;

    public MousePressCB(SoGate gate) {
      m_gate = gate;
    }

    @Override
    public void invoke(SoEventCallback eventCB) {
      SoEvent event = eventCB.getEvent();

      // Check for mouse button being pressed
      if (SoMouseButtonEvent.isButtonPressEvent(event, SoMouseButtonEvent.Buttons.ANY)) {
        // Toggle the gate that controls the duck motion
        if (m_gate.enable.getValue())
          m_gate.enable.setValue(false);
        else
          m_gate.enable.setValue(true);

        eventCB.setHandled();
      }
    }
  }
}
