/*----------------------------------------------------------------
 *  This is an example from the Inventor Mentor.
 *  chapter 15, example 4.
 *
 *  Same as chapter 15, example 2, with one difference:
 *  The draggers are customized to use different geometry.
 *  We create our own scene graphs for the parts "translator"
 *  and "translatorActive."
 *  Then we call setPart() to replace these two parts with our
 *  new scene graphs. (Remember, draggers are derived from
 *  nodekits, so it's easy to change the parts).
 *----------------------------------------------------------------*/

package inventor.mentor.customize;

import java.awt.BorderLayout;
import java.awt.Component;

import com.openinventor.inventor.draggers.SoTranslate1Dragger;
import com.openinventor.inventor.engines.SoCalculator;
import com.openinventor.inventor.nodekits.SoShapeKit;
import com.openinventor.inventor.nodes.SoCube;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoText3;
import com.openinventor.inventor.nodes.SoTransform;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example {

  private IViewerExaminer myViewer;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("Customized Sliders");
  }

  @Override
  public void start() {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    // Create 3 translate1Draggers
    SoSeparator xDragSep = new SoSeparator();
    SoSeparator yDragSep = new SoSeparator();
    SoSeparator zDragSep = new SoSeparator();

    // Separators will each hold a different transform
    SoTransform xDragXf = new SoTransform();
    SoTransform yDragXf = new SoTransform();
    SoTransform zDragXf = new SoTransform();
    xDragXf.set("translation  0 -4 8");
    yDragXf.set("translation -8  0 8 rotation 0 0 1  1.57");
    zDragXf.set("translation -8 -4 0 rotation 0 1 0 -1.57");
    xDragSep.addChild(xDragXf);
    yDragSep.addChild(yDragXf);
    zDragSep.addChild(zDragXf);

    // Add the draggers under the separators, after transforms
    SoTranslate1Dragger xDragger = new SoTranslate1Dragger();
    SoTranslate1Dragger yDragger = new SoTranslate1Dragger();
    SoTranslate1Dragger zDragger = new SoTranslate1Dragger();
    xDragSep.addChild(xDragger);
    yDragSep.addChild(yDragger);
    zDragSep.addChild(zDragger);

    /////////////////////////////////////////////////////////////
    // CODE FOR The Inventor Mentor STARTS HERE

    // Create myTranslator and myTranslatorActive.
    // These are custom geometry for the draggers.
    SoSeparator myTranslator = new SoSeparator();
    SoSeparator myTranslatorActive = new SoSeparator();

    // Materials for the dragger in regular and active states
    SoMaterial myMtl = new SoMaterial();
    SoMaterial myActiveMtl = new SoMaterial();
    myMtl.diffuseColor.setValue(1,1,1);
    myActiveMtl.diffuseColor.setValue(1,1,0);
    myTranslator.addChild(myMtl);
    myTranslatorActive.addChild(myActiveMtl);
    // Same shape for both versions.
    SoCube  myCube = new SoCube();
    myCube.set("width 3 height .4 depth .4");
    myTranslator.addChild(myCube);
    myTranslatorActive.addChild(myCube);

    // Now, customize the draggers with the pieces we created.
    xDragger.setPart("translator",myTranslator);
    xDragger.setPart("translatorActive",myTranslatorActive);
    yDragger.setPart("translator",myTranslator);
    yDragger.setPart("translatorActive",myTranslatorActive);
    zDragger.setPart("translator",myTranslator);
    zDragger.setPart("translatorActive",myTranslatorActive);

    // CODE FOR The Inventor Mentor ENDS HERE
    /////////////////////////////////////////////////////////////

    // Create shape kit for the 3D text
    // The text says 'Slide Cubes To Move Me'
    SoShapeKit textKit = new SoShapeKit();
    SoText3 myText3 = new SoText3();
    textKit.setPart("shape", myText3);
    myText3.justification.setValue(SoText3.Justifications.CENTER);
    myText3.string.set1Value(0,"Slide Cubes");
    myText3.string.set1Value(1,"To");
    myText3.string.set1Value(2,"Move Me");
    textKit.set("font { size 2}");
    textKit.set("material { diffuseColor 1 1 0}");

    // Create shape kit for surrounding box.
    // It's an unpickable cube, sized as (16,8,16)
    SoShapeKit boxKit = new SoShapeKit();
    boxKit.setPart("shape", new SoCube());
    boxKit.set("drawStyle { style LINES }");
    boxKit.set("lightModel { model BASE_COLOR }");
    boxKit.set("pickStyle { style UNPICKABLE }");
    boxKit.set("material { diffuseColor 1 0 1 }");
    boxKit.set("shape { width 16 height 8 depth 16 }");

    // Create the calculator to make a translation
    // for the text.  The x component of a translate1Dragger's
    // translation field shows how far it moved in that
    // direction. So our text's translation is:
    // (xDragTranslate[0],yDragTranslate[0],zDragTranslate[0])
    SoCalculator myCalc = new SoCalculator();
    myCalc.A.connectFrom(xDragger.translation);
    myCalc.B.connectFrom(yDragger.translation);
    myCalc.C.connectFrom(zDragger.translation);
    myCalc.expression.setValue("oA = vec3f(A[0],B[0],C[0])");

    // Connect the the translation in textKit from myCalc
    SoTransform textXf = (SoTransform)textKit.getPart("transform", true);
    textXf.translation.connectFrom(myCalc.oA);

    SoSeparator root = new SoSeparator();
    { // Assemble scene graph
      root.addChild(xDragSep);
      root.addChild(yDragSep);
      root.addChild(zDragSep);
      root.addChild(textKit);
      root.addChild(boxKit);
    }

    myViewer.setSceneGraph(root);
    myViewer.viewAll();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }
}
