/*----------------------------------------------------------------
 *  This is an example from the Inventor Mentor.
 *  chapter 14, example 1.
 *
 *  Use SoShapeKits to create two 3-D words, "NICE" and "HAPPY"
 *  Use nodekit methods to access the fields of the "material"
 *  and "transform" parts.
 *  Use a calculator engine and an elapsed time engine to make
 *  the words change color and fly about the screen.
 *----------------------------------------------------------------*/

package inventor.mentor.frolickingWords;
import java.awt.BorderLayout;
import java.awt.Component;

import com.openinventor.inventor.engines.SoCalculator;
import com.openinventor.inventor.engines.SoElapsedTime;
import com.openinventor.inventor.nodekits.SoShapeKit;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoText3;
import com.openinventor.inventor.nodes.SoTransform;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example {

  private IViewerExaminer myViewer;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("Frolicking Words");
  }

  @Override
  public void start() {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    // Create shape kits with the words "HAPPY" and "NICE"
    SoShapeKit happyKit = new SoShapeKit();
    happyKit.setPart("shape", new SoText3());
    happyKit.set("shape { parts ALL string \"HAPPY\"}");
    happyKit.set("font {size 2}");

    SoShapeKit niceKit = new SoShapeKit();
    niceKit.setPart("shape", new SoText3());
    niceKit.set("shape { parts ALL string \"NICE\"}");
    niceKit.set("font {size 2}");

    // Create the Elapsed Time engine
    SoElapsedTime myTimer = new SoElapsedTime();

    // Create two calculator - one for HAPPY, one for NICE.
    SoCalculator happyCalc = new SoCalculator();
    happyCalc.a.connectFrom(myTimer.timeOut);
    happyCalc.expression.setValue("ta = cos(2*a);" +
                                  "tb = sin(2*a);" +
                                  "oA = vec3f(3*pow(ta,3),3*pow(tb,3),1);" +
                                  "oB = vec3f(fabs(ta)+.1,fabs(0.5*fabs(tb))+0.1,1);" +
                                  "oC = vec3f(fabs(ta),fabs(tb),0.5)");

    // The second calculator uses different arguments to
    // sin() and cos(), so it moves out of phase.
    SoCalculator niceCalc = new SoCalculator();
    niceCalc.a.connectFrom(myTimer.timeOut);
    niceCalc.expression.setValue("ta = cos(2*a+2);" +
                                 "tb = sin(2*a+2);" +
                                 "oA = vec3f(3*pow(ta,3),3*pow(tb,3),1);" +
                                 "oB = vec3f(fabs(ta)+.1,fabs(.5*fabs(tb))+.1,1);" +
                                 "oC = vec3f(fabs(ta),fabs(tb),.5)");

    // Connect the transforms from the calculators...
    SoTransform happyXf = (SoTransform)happyKit.getPart("transform",true);
    happyXf.translation.connectFrom(happyCalc.oA);
    happyXf.scaleFactor.connectFrom(happyCalc.oB);
    SoTransform niceXf = (SoTransform)niceKit.getPart("transform",true);
    niceXf.translation.connectFrom(niceCalc.oA);
    niceXf.scaleFactor.connectFrom(niceCalc.oB);

    // Connect the materials from the calculators...
    SoMaterial happyMtl = (SoMaterial)happyKit.getPart("material",true);
    happyMtl.diffuseColor.connectFrom(happyCalc.oC);
    SoMaterial niceMtl = (SoMaterial)niceKit.getPart("material",true);
    niceMtl.diffuseColor.connectFrom(niceCalc.oC);

    SoSeparator root = new SoSeparator();
    { // Assemble scene graph
      root.addChild(happyKit);
      root.addChild(niceKit);
    }

    myViewer.setSceneGraph(root);
    myViewer.viewAll();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }
}
