/*-------------------------------------------------------------
 *  This is an example from The Inventor Mentor,
 *  chapter 10, example 8.
 *
 *  This example demonstrates the use of the pick filter
 *  callback to always select nodekits. This makes it especially
 *  easy to edit attributes of objects because the nodekit takes
 *  care of the part creation details.
 *------------------------------------------------------------*/

package inventor.mentor.pickFilterNodeKit;

import java.awt.BorderLayout;
import java.awt.Component;

import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.SoPickedPoint;
import com.openinventor.inventor.actions.SoBoxHighlightRenderAction;
import com.openinventor.inventor.misc.callbacks.SoSelectionClassCB;
import com.openinventor.inventor.misc.callbacks.SoSelectionPathCB;
import com.openinventor.inventor.misc.callbacks.SoSelectionPickCB;
import com.openinventor.inventor.nodekits.SoShapeKit;
import com.openinventor.inventor.nodes.SoCube;
import com.openinventor.inventor.nodes.SoGroup;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSelection;
import com.openinventor.inventor.nodes.SoTransform;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;
import util.editors.MaterialEditor;

public class Main extends Example {

  private IViewerExaminer myViewer;

  SoSelection m_sel;
  MaterialEditor m_editor;
  boolean m_ignore;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("Select Node Kits");
  }

  @Override
  public void start() {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    // Create our scene graph.
    m_sel = new SoSelection();
    m_sel.addChild(buildScene());
    m_sel.addChangeCallback(new SoSelectionClassCB()
    {
      @Override
      public void invoke(SoSelection s)
      {
        myViewer.getRenderArea().scheduleRedraw();
      }
    });

    myViewer.setSceneGraph(m_sel);
    myViewer.viewAll();
    // Set a render action that displays highlights
    myViewer.getRenderArea().setGLRenderAction(new SoBoxHighlightRenderAction());

    // Create a material editor
    m_editor = new MaterialEditor();
    m_editor.setVisible(true);

    m_ignore = false;

    // Selection and material change callbacks
    m_editor.addListener(new MaterialEditorListener());
    m_sel.setPickFilterCallback(new PickFilterCB());
    m_sel.addSelectionCallback(new SelectCB());

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    m_editor.dispose();
    myViewer.dispose();
  }

  // Create a sample scene graph
  SoNode buildScene() {
    SoGroup g = new SoGroup();
    SoShapeKit k;
    SoTransform xf;

    // Place a dozen shapes in circular formation
    for (int i = 0; i < 12; i++) {
      k = new SoShapeKit();
      k.setPart("shape", new SoCube());
      xf = (SoTransform) k.getPart("transform", true);
      xf.translation.setValue( (float) (8 * Math.sin(i * Math.PI / 6)),
                              (float) (8 * Math.cos(i * Math.PI / 6)),
                              0.0F);
      g.addChild(k);
    }

    return g;
  }

  // This is called when the user chooses a new material
  // in the material editor. This updates the material
  // part of each selected node kit.
  class MaterialEditorListener extends MaterialEditor.Listener {
    @Override
    public void valueChanged(SoMaterial material)
    {
      // Our material change callback is invoked when the
      // user changes the material, and when we change it
      // through a call to MaterialEditor.setMaterial.
      // In this latter case, we ignore the callback invocation.

      if (m_ignore) {
        return;
      }

      // Our pick filter guarantees the path tail will
      // be a shape kit.
      for (int i = 0; i < m_sel.getNumSelected(); i++) {
        SoPath p = m_sel.getPath(i);
        SoShapeKit kit = (SoShapeKit)p.regular.getTail();
        SoMaterial kitMtl = (SoMaterial)kit.getPart("material", true);
        kitMtl.copyFieldValues(material);
      }
    }
  }

  //////////////////////////////////////////////////////////////
  // CODE FOR The Inventor Mentor STARTS HERE  (part 1)

  // Truncate the pick path so a nodekit is selected
  class PickFilterCB extends SoSelectionPickCB {
    @Override
    public SoPath invoke(SoPickedPoint pick) {
      // See which child of selection got picked
      SoPath p = pick.getPath();
      int i;
      for (i = p.regular.getLength() - 1; i >= 0; i--) {
        SoNode n = p.regular.getNode(i);
        if (n instanceof SoShapeKit)
          break;
      }

      // Copy the path down to the nodekit
      return p.regular.copy(0, i+1);
    }
  }

  // CODE FOR The Inventor Mentor ENDS HERE
  ///////////////////////////////////////////////////////////////

  // Update the material editor to reflect the selected object
  class SelectCB extends SoSelectionPathCB {
    @Override
    public void invoke(SoPath path) {
      SoShapeKit kit = (SoShapeKit)path.regular.getTail();
      SoMaterial kitMtl = (SoMaterial)kit.getPart("material", true);

      m_ignore = true;
      m_editor.setMaterial(kitMtl);
      m_ignore = false;
    }
  }
}
