/*-------------------------------------------------------------
 *  This is an example from The Inventor Mentor,
 *  chapter 10, example 6.
 *
 *  This example demonstrates the use of the pick filter
 *  callback to implement a top level selection policy.
 *  That is, always select the top most group beneath the
 *  selection node,  rather than selecting the actual
 *  shape that was picked.
 *------------------------------------------------------------*/

package inventor.mentor.pickFilterTopLevel;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Font;
import java.awt.GridLayout;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;

import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.SoPickedPoint;
import com.openinventor.inventor.actions.SoBoxHighlightRenderAction;
import com.openinventor.inventor.misc.callbacks.SoSelectionClassCB;
import com.openinventor.inventor.misc.callbacks.SoSelectionPickCB;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSelection;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.InteractionMode;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example {

  private IViewerExaminer viewer1;
  private IViewerExaminer viewer2;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("Pick Filter Callback");
  }

  @Override
  public void start() {
    viewer1 = ViewerComponentsFactory.createViewerExaminer();
    viewer2 = ViewerComponentsFactory.createViewerExaminer();

    // Open the data file
    SoInput in = new SoInput();
    String data_file = "$OIVJHOME/data/models/parkbench.iv";
    if (!in.openFile(data_file)) {
      System.err.println("Cannot open " + data_file + " for reading.");
      System.exit( -1);
    }

    // Read the input file
    SoNode n;
    SoSeparator sep = new SoSeparator();
    while ( (n = SoDB.readNode(in)) != null)
      sep.addChild(n);

      // Create two selection roots - one will use the pick filter.
    SoSelection topLevelSel = new SoSelection();
    topLevelSel.addChild(sep);
    topLevelSel.setPickFilterCallback(new PickFilterCB());
    topLevelSel.addChangeCallback(new SoSelectionClassCB()
    {
      @Override
      public void invoke(SoSelection s)
      {
        viewer1.getRenderArea().scheduleRedraw();
      }
    });

    SoSelection defaultSel = new SoSelection();
    defaultSel.addChild(sep);
    defaultSel.addChangeCallback(new SoSelectionClassCB()
    {
      @Override
      public void invoke(SoSelection s)
      {
        viewer2.getRenderArea().scheduleRedraw();
      }
    });

    // Create two viewers, one to show the pick filter for top level
    // selection, the other to show default selection.
    viewer1.setSceneGraph(topLevelSel);
    viewer1.viewAll();
    viewer1.getRenderArea().setGLRenderAction(new SoBoxHighlightRenderAction());
    viewer1.getRenderArea().getSceneInteractor().setInteractionMode(InteractionMode.SELECTION);

    viewer2.setSceneGraph(defaultSel);
    viewer2.viewAll();
    viewer2.getRenderArea().setGLRenderAction(new SoBoxHighlightRenderAction());
    viewer2.getRenderArea().getSceneInteractor().setInteractionMode(InteractionMode.SELECTION);

    final Component component1 = viewer1.getComponent();
    final Component component2 = viewer2.getComponent();
    Font f = new Font(null, 1 ,15);
    JPanel panel1 = new JPanel(new BorderLayout());
    JLabel label1 = new JLabel("Top Level Selection", SwingConstants.CENTER);
    label1.setFont(f);
    panel1.add(label1, BorderLayout.SOUTH);
    panel1.add(component1, BorderLayout.CENTER);

    JPanel panel2 = new JPanel(new BorderLayout());
    JLabel label2 = new JLabel("Default Selection", SwingConstants.CENTER);
    label2.setFont(f);
    panel2.add(label2, BorderLayout.SOUTH);
    panel2.add(component2, BorderLayout.CENTER);

    setLayout(new GridLayout(1, 2));
    component1.setPreferredSize(new java.awt.Dimension(400, 400));
    component2.setPreferredSize(new java.awt.Dimension(400, 400));
    add(panel1);
    add(panel2);
  }

  @Override
  public void stop()
  {
    viewer1.dispose();
    viewer2.dispose();
  }

  class PickFilterCB extends SoSelectionPickCB {
    @Override
    public SoPath invoke(SoPickedPoint pick) {
      // See which child of selection got picked
      SoPath p = pick.getPath();
      int i;
      for (i = 0; i < p.regular.getLength() - 1; i++) {
        SoNode n = p.regular.getNode(i);
        if (n instanceof SoSelection)
          break;
      }

      // Copy 2 nodes from the path:
      // selection and the picked child
      return p.regular.copy(i, 2);
    }
  }
}
