package inventor.sample.backgroundNode;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;

import javax.swing.*;

import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoImageBackground;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;
import util.editors.ColorEditor;

public class Main extends Example
{
  SoSwitch pGradientSwitch = null;
  SoSwitch pImgBackgSwitch = null;
  SoGradientBackground pGradient = null;
  SoImageBackground pImgBackg = null;
  ColorEditor pColEdit = null;

  JButton butColor0, butColor1;
  JCheckBox chkGVisible, chkIVisible, chkSwap;
  JComboBox<String> comStyle;
  JButton butBrowse;
  JTextField txtPath;

  // Constants
  // This image demonstrates that partially transparent images are
  // correctly rendered (other backgrounds show through)
  String imgFilename = "$OIVJHOME/data/textures/png/cypress.png";
  private IViewerExaminer myViewer;

  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    SoSeparator root = new SoSeparator();

    // read scene graph from iv file
    root.addChild(readSceneGraph("$OIVJHOME/data/models/spongetri4.iv"));

    // Create background nodes
    //
    // Image background node will be traversed after gradient background
    // node to show how we can put an image on top of a gradient...
    // Initially the image will be displayed in the lower left corner.
    pGradient = new SoGradientBackground();
    pImgBackg = new SoImageBackground();
    pImgBackg.filename.setValue(imgFilename);
    pImgBackg.style.setValue(SoImageBackground.Styles.LOWER_LEFT);

    // Create some switch nodes so we can turn the backgrounds on and off
    pGradientSwitch = new SoSwitch();
    pImgBackgSwitch = new SoSwitch();
    pGradientSwitch.whichChild.setValue(0);
    pImgBackgSwitch.whichChild.setValue(0);
    pGradientSwitch.addChild(pGradient);
    pImgBackgSwitch.addChild(pImgBackg);
    SoSeparator pBkgSep = new SoSeparator();
    pBkgSep.addChild(pGradientSwitch);
    pBkgSep.addChild(pImgBackgSwitch);

    // Insert the background nodes at the top of the scene graph
    root.insertChild(pBkgSep, 0);

    // attach and show viewer
    myViewer.setSceneGraph(root);
    myViewer.viewAll();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(new BackgroundFrame(), BorderLayout.WEST);
  }

  @Override
  public void stop()
  {
    if ( pColEdit != null )
      pColEdit.dispose();
    myViewer.dispose();
  }

  private static SoSeparator readSceneGraph(String fileName)
  {
    SoInput input = new SoInput();

    if ( !input.openFile(fileName) )
    {
      System.err.println("Cannot open file " + fileName);
      return null;
    }

    SoSeparator node = SoDB.readAll(input);
    if ( node == null )
    {
      System.err.println("Problem reading file");
      input.closeFile();
      return null;
    }

    input.closeFile();
    return node;
  }

  public static void main(String argv[])
  {
    Main example = new Main();
    example.demoMain("Background Test");
  }

  // ///////////////////////////////////////////////////////////
  //
  // Load simple interface to experiment with background options

  class BackgroundFrame extends JPanel
  {

    public BackgroundFrame()
    {
      Box verBox = Box.createVerticalBox();
      add(verBox);

      verBox.add(new JLabel("Gradient Background"));
      chkGVisible = new JCheckBox("Visible", null, true);
      chkGVisible.addActionListener(new ActListener());
      verBox.add(chkGVisible);
      butColor0 = new JButton("Color 0");
      butColor0.addActionListener(new ActListener());
      verBox.add(butColor0);
      butColor1 = new JButton("Color 1");
      butColor1.addActionListener(new ActListener());
      verBox.add(butColor1);
      chkSwap = new JCheckBox("Swap colors", null, false);
      chkSwap.addActionListener(new ActListener());
      verBox.add(chkSwap);
      verBox.add(Box.createVerticalStrut(15));
      verBox.add(new JLabel("Image Background"));
      chkIVisible = new JCheckBox("Visible", null, true);
      chkIVisible.addActionListener(new ActListener());
      verBox.add(chkIVisible);
      String[] styles = { "NONE", "CENTER", "LOWER_LEFT", "UPPER_LEFT", "LOWER_RIGHT", "UPPER_RIGHT", "STRETCH", "TILE" };
      comStyle = new JComboBox<String>(styles);
      comStyle.addActionListener(new ActListener());
      verBox.add(comStyle);
      txtPath = new JTextField(imgFilename, 20);
      txtPath.addActionListener(new ActListener());
      verBox.add(txtPath);
      butBrowse = new JButton("Browse");
      butBrowse.addActionListener(new ActListener());
      verBox.add(butBrowse);
    }
  }

  // /////////////////////////////////////////////////////////////////////

  public class ActListener implements ActionListener
  {

    @Override
    public void actionPerformed(ActionEvent ev)
    {
      /*----------------------- Gradient background visible -----------------------*/
      if ( ev.getSource() == chkGVisible )
      {
        pGradientSwitch.whichChild.setValue(chkGVisible.isSelected() ? 0 : -1);
      }

      /*----------------------- Gradient background color 0 -----------------------*/
      if ( ev.getSource() == butColor0 )
      {
        if ( pColEdit == null )
        {
          pColEdit = new ColorEditor();
          pColEdit.setCurrentSliders(ColorEditor.RGB_V);
        }
        pColEdit.attach(pGradient.color0, pGradient);
        pColEdit.setVisible(true);
      }

      /*----------------------- Gradient background color 1 -----------------------*/
      if ( ev.getSource() == butColor1 )
      {
        if ( pColEdit == null )
        {
          pColEdit = new ColorEditor();
          pColEdit.setCurrentSliders(ColorEditor.RGB_V);
        }
        pColEdit.attach(pGradient.color1, pGradient);
        pColEdit.setVisible(true);
      }

      /*----------------------- Gradient background swap -----------------------*/
      if ( ev.getSource() == chkSwap )
      {
        pGradient.swapColors.setValue(chkSwap.isSelected() ? true : false);
      }

      /*----------------------- Image background visible -----------------------*/
      if ( ev.getSource() == chkIVisible )
      {
        pImgBackgSwitch.whichChild.setValue(chkIVisible.isSelected() ? 0 : -1);
      }

      /*----------------------- Image background style -----------------------*/
      if ( ev.getSource() == comStyle )
      {
        int choice = comStyle.getSelectedIndex();
        switch ( choice )
        {
        case 0 :
          pImgBackg.style.setValue(SoImageBackground.Styles.NONE);
          break;
        case 1 :
          pImgBackg.style.setValue(SoImageBackground.Styles.CENTER);
          break;
        case 2 :
          pImgBackg.style.setValue(SoImageBackground.Styles.LOWER_LEFT);
          break;
        case 3 :
          pImgBackg.style.setValue(SoImageBackground.Styles.UPPER_LEFT);
          break;
        case 4 :
          pImgBackg.style.setValue(SoImageBackground.Styles.LOWER_RIGHT);
          break;
        case 5 :
          pImgBackg.style.setValue(SoImageBackground.Styles.UPPER_RIGHT);
          break;
        case 6 :
          pImgBackg.style.setValue(SoImageBackground.Styles.STRETCH);
          break;
        case 7 :
          pImgBackg.style.setValue(SoImageBackground.Styles.TILE);
          break;
        }
      }

      /*----------------------- Image background image file -----------------------*/
      if ( ev.getSource() == txtPath )
      {
        pImgBackg.filename.setValue(txtPath.getText());
      }

      /*----------------------- Image background browser -----------------------*/
      if ( ev.getSource() == butBrowse )
      {
        JFileChooser fileChooser = new JFileChooser(SoPreferences.getValue("OIVJHOME") + File.separator + "data");
        if ( fileChooser.showOpenDialog(null) == JFileChooser.APPROVE_OPTION )
        {
          txtPath.setText(fileChooser.getSelectedFile().getPath());
          pImgBackg.filename.setValue(txtPath.getText());
        }
      }
    }
  }
}
