package inventor.sample.extSelection;

import java.awt.*;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.actions.SoHaloHighlightRenderAction;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.misc.callbacks.SoSelectionClassCB;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;

/**
 * This example is similar to the demo/mentor/inventor/AWTList.java. However, it
 * uses an SoExtSelection nodes instead of SoSelection.
 *
 * The scene graph has 4 objects which may be selected by picking with the left
 * mouse button, a lasso or a rectangle. Use shift key to extend the selection
 * to more than one object).
 *
 * Hitting the up arrow key will increase the size of each selected object;
 * hitting down arrow will decrease the size of each selected object.
 *
 * @author Jean-Michel Godinaud
 */
public class Main extends Example
{
  private IViewerExaminer m_viewer;
  SoExtSelection selectionRoot;

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("Extension Selection");
  }

  public Main()
  {}

  @Override
  public void start()
  {
    m_viewer = ViewerComponentsFactory.createViewerExaminer();

    // Selection
    // -----------------------------------------------------------------
    selectionRoot = new SoExtSelection();
    selectionRoot.useFastEditing(true);
    selectionRoot.lassoWidth.setValue(1);
    selectionRoot.lassoColor.setValue(new SbColor(0, 1, 0));
    selectionRoot.lassoPattern.setValue((short)0xE0E0);
    selectionRoot.animateLasso.setValue(true);
    selectionRoot.policy.setValue(SoSelection.Policies.SINGLE);
    selectionRoot.lassoType.setValue(SoExtSelection.LassoTypes.NOLASSO);
    selectionRoot.lassoPolicy.setValue(SoExtSelection.LassoPolicies.PART);
    selectionRoot.addChangeCallback(new SoSelectionClassCB()
    {
      @Override
      public void invoke(SoSelection s)
      {
        m_viewer.getRenderArea().scheduleRedraw();
      }
    });

    // Add an event callback so we can receive key press events
    SoEventCallback myEventCB = new SoEventCallback();
    myEventCB.addEventCallback("com.openinventor.inventor.events.SoKeyboardEvent", new MyKeyPressCB(), selectionRoot);

    // Scene graph
    // ---------------------------------------------------------------
    SoTransform t;
    SoMaterial m;

    // Sphere
    t = new SoTransform();
    t.translation.setValue(2, 2, 0);
    m = new SoMaterial();
    m.diffuseColor.setValue(1, 0, 0);
    SoSeparator sphereRoot = new SoSeparator();
    {
      sphereRoot.addChild(t);
      sphereRoot.addChild(m);
      sphereRoot.addChild(new SoSphere());
    }

    // Cone
    SoSeparator coneRoot = new SoSeparator();
    t = new SoTransform();
    t.translation.setValue(2, -2, 0);
    m = new SoMaterial();
    m.diffuseColor.setValue(0, 1, 0);
    {
      coneRoot.addChild(t);
      coneRoot.addChild(m);
      coneRoot.addChild(new SoCone());
    }

    // Cylinder
    SoSeparator cylinderRoot = new SoSeparator();
    t = new SoTransform();
    t.translation.setValue(-2, -2, 0);
    m = new SoMaterial();
    m.diffuseColor.setValue(0, 0, 1);
    {
      cylinderRoot.addChild(t);
      cylinderRoot.addChild(m);
      cylinderRoot.addChild(new SoCylinder());
    }

    // Cube
    SoSeparator cubeRoot = new SoSeparator();
    t = new SoTransform();
    t.translation.setValue(-2, 2, 0);
    m = new SoMaterial();
    m.diffuseColor.setValue(1, 1, 0);
    {
      cubeRoot.addChild(t);
      cubeRoot.addChild(m);
      cubeRoot.addChild(new SoCube());
    }

    { // assemble scene graph
      selectionRoot.addChild(myEventCB);
      selectionRoot.addChild(sphereRoot);
      selectionRoot.addChild(coneRoot);
      selectionRoot.addChild(cylinderRoot);
      selectionRoot.addChild(cubeRoot);
    }

    // Create the halohighlight render action
    SoHaloHighlightRenderAction haloRenderAction = new SoHaloHighlightRenderAction();
    haloRenderAction.setHaloWidth(7);
    haloRenderAction.setHaloSmoothFactor(0.75f);
    haloRenderAction.setHaloColor(new SbColor(1.f, 1.f, 0.8f));

    // Create a viewer for viewing the scene
    m_viewer.getRenderArea().setGLRenderAction(haloRenderAction);
    m_viewer.setSceneGraph(selectionRoot);
    m_viewer.viewAll();

    // Menu Selection policy
    Label labelSelectionPolicy = new Label("Selection policy");
    labelSelectionPolicy.setForeground(Color.red);
    CheckboxGroup cbgSelectionPolicy = new CheckboxGroup();
    Checkbox singleSelectionPolicy = new Checkbox("single", cbgSelectionPolicy, true);
    Checkbox toggleSelectionPolicy = new Checkbox("toggle", cbgSelectionPolicy, false);
    Checkbox shiftSelectionPolicy = new Checkbox("shift", cbgSelectionPolicy, false);
    // listener
    singleSelectionPolicy.addItemListener(new SingleSelectionPolicy());
    toggleSelectionPolicy.addItemListener(new ToggleSelectionPolicy());
    shiftSelectionPolicy.addItemListener(new ShiftSelectionPolicy());

    // Menu Selection Type
    Label labelSelectionType = new Label("Selection type");
    labelSelectionType.setForeground(Color.red);
    CheckboxGroup cbgSelectionType = new CheckboxGroup();
    Checkbox nolassoSelectionType = new Checkbox("nolasso", cbgSelectionType, true);
    Checkbox lassoSelectionType = new Checkbox("lasso", cbgSelectionType, false);
    Checkbox rectangleSelectionType = new Checkbox("rectangle", cbgSelectionType, false);
    // listener
    nolassoSelectionType.addItemListener(new NolassoSelectionType());
    lassoSelectionType.addItemListener(new LassoSelectionType());
    rectangleSelectionType.addItemListener(new RectangleSelectionType());

    // Menu Lasso policy
    Label labelLassoPolicy = new Label("Lasso policy");
    labelLassoPolicy.setForeground(Color.red);
    CheckboxGroup cbgLassoPolicy = new CheckboxGroup();
    Checkbox fullBboxLassoPolicy = new Checkbox("Full Bounding Box", cbgLassoPolicy, false);
    Checkbox partBboxLassoPolicy = new Checkbox("Part Bounding Box", cbgLassoPolicy, false);
    Checkbox fullLassoPolicy = new Checkbox("Full", cbgLassoPolicy, false);
    Checkbox partLassoPolicy = new Checkbox("Part", cbgLassoPolicy, true);
    // listener
    fullBboxLassoPolicy.addItemListener(new FullBboxLassoPolicy());
    partBboxLassoPolicy.addItemListener(new PartBboxLassoPolicy());
    fullLassoPolicy.addItemListener(new FullLassoPolicy());
    partLassoPolicy.addItemListener(new PartLassoPolicy());

    Panel leftPanel = new Panel(new GridLayout(0, 1));
    leftPanel.add(labelSelectionPolicy);
    leftPanel.add(singleSelectionPolicy);
    leftPanel.add(toggleSelectionPolicy);
    leftPanel.add(shiftSelectionPolicy);
    leftPanel.add(new Label(""));
    leftPanel.add(labelSelectionType);
    leftPanel.add(nolassoSelectionType);
    leftPanel.add(lassoSelectionType);
    leftPanel.add(rectangleSelectionType);
    leftPanel.add(new Label(""));
    leftPanel.add(labelLassoPolicy);
    leftPanel.add(fullBboxLassoPolicy);
    leftPanel.add(partBboxLassoPolicy);
    leftPanel.add(fullLassoPolicy);
    leftPanel.add(partLassoPolicy);

    final Component component = m_viewer.getComponent();
    component.setName("Selection List");
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    setBackground(Color.white);
    setForeground(Color.black);
    add(component);
    add(leftPanel, BorderLayout.WEST);
  }

  @Override
  public void stop()
  {
    selectionRoot = null;
    m_viewer.dispose();
  }

  // ////////////////////////////////////////////////////////////////////
  // inner classes : listeners
  // ////////////////////////////////////////////////////////////////////
  class MyKeyPressCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback eventCB)
    {
      SoSelection selection = (SoSelection) userData;
      SoEvent event = eventCB.getEvent();

      // check for the Up and Down arrow keys being pressed
      if ( SoKeyboardEvent.isKeyPressEvent(event, SoKeyboardEvent.Keys.UP_ARROW) )
      {
        myScaleSelection(selection, 1.1F);
        eventCB.setHandled();
      }
      else if ( SoKeyboardEvent.isKeyPressEvent(event, SoKeyboardEvent.Keys.DOWN_ARROW) )
      {
        myScaleSelection(selection, (float) (1.0 / 1.1));
        eventCB.setHandled();
      }
    }

    // Scale each object in the selection list
    void myScaleSelection(SoSelection selection, float sf)
    {
      SoPath selectedPath;
      SoTransform xform = null;
      SbVec3f scaleFactor = new SbVec3f();
      int i, j;

      // Scale each object in the selection list
      for ( i = 0; i < selection.getNumSelected(); i++ )
      {
        selectedPath = selection.getPath(i);
        // Look for the SoSeparator node, starting from the tail of the
        // path. We assume that its first child is the SoTransform to modify
        for ( j = 0; (j < selectedPath.regular.getLength()); j++ )
        {
          SoNode n = selectedPath.regular.getNodeFromTail(j);
          if ( n instanceof SoSeparator )
          {
            SoSeparator s = (SoSeparator) n;
            xform = (SoTransform) (s.getChild(0)); // assumes 1st child is a
                                                   // SoTransform
            break;
          }
        }

        // Apply the scale
        scaleFactor = xform.scaleFactor.getValue();
        scaleFactor.multiply(sf);
        xform.scaleFactor.setValue(scaleFactor);
      }
    }
  }

  class SingleSelectionPolicy implements ItemListener
  {
    @Override
    public void itemStateChanged(ItemEvent e)
    {
      selectionRoot.policy.setValue(SoSelection.Policies.SINGLE);
    }
  }

  class ToggleSelectionPolicy implements ItemListener
  {
    @Override
    public void itemStateChanged(ItemEvent e)
    {
      selectionRoot.policy.setValue(SoSelection.Policies.TOGGLE);
    }
  }

  class ShiftSelectionPolicy implements ItemListener
  {
    @Override
    public void itemStateChanged(ItemEvent e)
    {
      selectionRoot.policy.setValue(SoSelection.Policies.SHIFT);
    }
  }

  class NolassoSelectionType implements ItemListener
  {
    @Override
    public void itemStateChanged(ItemEvent e)
    {
      selectionRoot.lassoType.setValue(SoExtSelection.LassoTypes.NOLASSO);
    }
  }

  class LassoSelectionType implements ItemListener
  {
    @Override
    public void itemStateChanged(ItemEvent e)
    {
      selectionRoot.lassoType.setValue(SoExtSelection.LassoTypes.LASSO);
    }
  }

  class RectangleSelectionType implements ItemListener
  {
    @Override
    public void itemStateChanged(ItemEvent e)
    {
      selectionRoot.lassoType.setValue(SoExtSelection.LassoTypes.RECTANGLE);
    }
  }

  class FullBboxLassoPolicy implements ItemListener
  {
    @Override
    public void itemStateChanged(ItemEvent e)
    {
      selectionRoot.lassoPolicy.setValue(SoExtSelection.LassoPolicies.FULL_BBOX);
    }
  }

  class PartBboxLassoPolicy implements ItemListener
  {
    @Override
    public void itemStateChanged(ItemEvent e)
    {
      selectionRoot.lassoPolicy.setValue(SoExtSelection.LassoPolicies.PART_BBOX);
    }
  }

  class FullLassoPolicy implements ItemListener
  {
    @Override
    public void itemStateChanged(ItemEvent e)
    {
      selectionRoot.lassoPolicy.setValue(SoExtSelection.LassoPolicies.FULL);
    }
  }

  class PartLassoPolicy implements ItemListener
  {
    @Override
    public void itemStateChanged(ItemEvent e)
    {
      selectionRoot.lassoPolicy.setValue(SoExtSelection.LassoPolicies.PART);
    }
  }
}
