package inventor.sample.wellBore;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec2f;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{

  private IViewerExaminer myViewer;

  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    // Lower epsilon value to avoid "twisting" artifacts
    SoPreferences.setValue("OIV_EXTRUSION_EPSILON", "0.99");

    // Try to read the files
    String pkgName = this.getClass().getPackage().getName();
    pkgName = pkgName.replace('.', File.separatorChar);
    String prefix =
        SoPreferences.getValue("OIVJHOME") + File.separator + "examples" + File.separator + pkgName + File.separator;

    WellBore well0 = WellBore.createFromFile(prefix + "data/data0.txt");
    WellBore well1 = WellBore.createFromFile(prefix + "data/data1.txt");
    ColorMap colorMap = ColorMap.createFromFile(prefix + "data/colormap.txt");

    if ( well0 == null || well1 == null || colorMap == null )
    {
      System.err.println("Exiting -- unable to load data or colormap");
      System.exit(0);
    }

    // Simulate having a second data set on the second bore
    well1.setData1(new float[well1.getNumPoints()]);
    for ( int i = 0; i < well1.getNumPoints(); i++ )
      well1.setData1(i, well1.getData0(i));

    // Create scene graph
    SoSeparator root = new SoSeparator();

    SoGradientBackground bg = new SoGradientBackground();
    bg.color0.setValue(193f / 256f, 218f / 256f, 255f / 256f);
    bg.color1.setValue(0f / 256f, 53f / 256f, 128f / 256f);
    root.addChild(bg);

    // We happen to know the bounds of the data volume these
    // well bores are associated with...
    float xmax = 300;
    float ymax = 455;
    float zmax = 91;

    // Simple bounding box
    SoTranslation trans = new SoTranslation();
    trans.translation.setValue(xmax / 2, ymax / 2, zmax / 2);

    SoDrawStyle style = new SoDrawStyle();
    style.style.setValue(SoDrawStyle.Styles.LINES);

    SoLightModel model = new SoLightModel();
    model.model.setValue(SoLightModel.Models.BASE_COLOR);

    SoCube cube = new SoCube();
    cube.width.setValue(xmax);
    cube.height.setValue(ymax);
    cube.depth.setValue(zmax);

    SoSeparator boxSep = new SoSeparator();
    boxSep.addChild(trans);
    boxSep.addChild(style);
    boxSep.addChild(model);
    boxSep.addChild(cube);

    root.addChild(boxSep);

    // Number of sides (12-16 gives a reasonable looking cylinder)
    int numSides = 12;

    // Range of data values that color map actually applies to
    float dataMin = 125;
    float dataMax = 200;

    // First bore only has one data set, so default bore will have
    // diameter of 1. Scale it by 8 to be more visible.
    root.addChild(makeExtrusion(well0, colorMap, numSides, dataMin, dataMax, 8));

    // Second bore has two data sets (actually both contain same values),
    // so bore diameter will be taken from second data set. Scale it
    // down to be approximately in scale with the first bore.
    root.addChild(makeExtrusion(well1, colorMap, numSides, dataMin, dataMax, 0.08f));

    myViewer.setSceneGraph(root);
    myViewer.viewAll();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  SoSeparator makeExtrusion(WellBore well, ColorMap map, int numSides, float minValForColor, float maxValForColor,
      float scaleFactor)
  {
    // Group the attribute nodes and extrusion
    SoSeparator sep = new SoSeparator();

    // Extrusion will be considered "solid" to enable back-face culling.
    // Also set crease angle to "smooth" surface for more than 4 sides.
    SoShapeHints hints = new SoShapeHints();
    hints.vertexOrdering.setValue(SoShapeHints.VertexOrderings.COUNTERCLOCKWISE);
    hints.shapeType.setValue(SoShapeHints.ShapeTypes.SOLID);
    hints.creaseAngle.setValue((float) (Math.PI / 2.1));
    sep.addChild(hints);

    SoExtrusion ext = new SoExtrusion();

    // Cross section (prescaled to diameter=1 to allow meaningful scaling)
    // 16 sides makes a reasonable cylinder (with smoothing).
    // 4 sides is useful for checking twist or to improve performance.

    // Compute x and z coordinates around circle
    if ( numSides < 2 )
      numSides = 2;
    int side;
    float theta = 0.0f;
    float dTheta = (float) (2.0 * Math.PI / numSides);
    float eps = 1e-6f;

    ext.crossSection.setNum(numSides + 1);

    for ( side = 0; side < numSides; side++ )
    {
      float x = (float) (0.5f * Math.sin(theta));
      float z = (float) (0.5f * Math.cos(theta));
      if ( Math.abs(x) < eps )
        x = 0;
      if ( Math.abs(z) < eps )
        z = 0;
      ext.crossSection.set1Value(side, new SbVec2f(x, z));
      theta += dTheta;
    }
    ext.crossSection.set1Value(numSides, new SbVec2f(0, 0.5f)); // close loop

    // Coordinates of well bore define the spine
    ext.spine.setValues(0, well.getPoints());

    // Get data range
    int nPoints = well.getNumPoints();

    // First data set maps to color
    // (requires new SoExtrusion node)
    if ( (well.getData0().length != 0) && (map != null) )
    {
      // Assign color value to each spine point
      SoMaterialBinding bind = new SoMaterialBinding();
      bind.value.setValue(SoMaterialBinding.Bindings.PER_VERTEX);

      // Define colors (one per spine vertex)
      SoMaterial mat = new SoMaterial();
      mat.diffuseColor.setNum(nPoints);

      // Number of colors in map
      int nColors = map.getNColors();

      for ( int i = 0; i < nPoints; ++i )
      {
        float data = well.getData0(i);
        SbColor color;
        if ( data <= minValForColor )
          color = map.getColors(0);
        else if ( data >= maxValForColor )
          color = map.getColors(map.getNColors() - 1);
        else
        {
          float t = (data - minValForColor) / (maxValForColor - minValForColor);
          t *= nColors;
          int index = (int) (t + 0.5);
          color = map.getColors(index);
        }
        mat.diffuseColor.set1Value(i, color);
      }

      sep.addChild(bind);
      sep.addChild(mat);
    }

    // Second data set maps to radius (defines scale factor)
    if ( well.getData1() != null )
    {
      ext.scale.setNum(well.getNumPoints());

      for ( int i = 0; i < well.getNumPoints(); ++i )
      {
        // Apply global scale factor
        float scale = well.getData1(i) * scaleFactor;
        ext.scale.set1Value(i, new SbVec2f(scale, scale));
      }
    }
    // Else just use global scale factor for all points
    else
      ext.scale.setValue(new SbVec2f(scaleFactor, scaleFactor));

    sep.addChild(ext);

    return sep;
  }

  public static void main(String argv[])
  {
    Main example = new Main();
    example.demoMain("Well bore");
  }
}
