package medical.input.medicalDicomImageViewer;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.openinventor.inventor.SbVec2f;
import com.openinventor.inventor.SbVec2s;
import com.openinventor.inventor.SbVec3i32;
import com.openinventor.inventor.SoPickedPoint;
import com.openinventor.inventor.details.SoDetail;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.events.SoLocation2Event;
import com.openinventor.inventor.events.SoMouseButtonEvent;
import com.openinventor.inventor.events.SoMouseWheelEvent;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.SoEventCallback;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoGroup;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoMatrixTransform;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.InteractionMode;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.NavigationMode;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor.CameraMode;
import com.openinventor.ldm.nodes.SoDataRange;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.ldm.nodes.SoTransferFunction.PredefColorMaps;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.medical.nodes.DicomInfo;
import com.openinventor.medical.nodes.SliceOrientationMarkers;
import com.openinventor.medical.nodes.TextBox;
import com.openinventor.volumeviz.details.SoOrthoSliceDetail;
import com.openinventor.volumeviz.nodes.SoOrthoSlice;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeShape;
import com.openinventor.volumeviz.readers.SoVRDicomFileReader;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  private static final int FILE_LAST = 557;
  private static final int FILE_FIRST = 1;

  public static final String EXAMPLE_NAME = "Dicom Image Viewer";
  public static final String DATA_PATH = "/medical/data/dicomSample";
  public static final String FILE_MASK = "CVH%03d.dcm";
  public static final String ANNOTATION = "/medical/data/dicomSample/CVH001.dcm";

  private final static Logger LOGGER = Logger.getLogger(Main.class.getName());
  private SoSeparator _root;
  private static String _dataPath;
  private static String _annotation;
  private IRenderAreaExaminer _renderArea;

  private SbVec2f _resetWinCtrWidth;
  // TextBox to display dynamic info and corresponding line numbers.
  private TextBox _infoDisplay = null;
  private int _line_sliceNum = 0;
  private int _line_ctrWidth = 1;
  private int _line_voxValue = 2;

  private SoVolumeData _volData = null;
  private SoDataRange _dataRange = null;
  private SoOrthoSlice _sliceNode = null;
  private SliceOrientationMarkers _sliceMarkers = null;

  // Slice properties needed for the user interface
  private SoOrthoSlice.AxisType _sliceAxis;
  private int _sliceNum = 0;
  private int _numSlices = 0;

  // Mouse button state and enable display of voxel values
  enum MouseMode
  {
    MOUSE_NOOP, // Moving the mouse does nothing
    MOUSE_SHOW_VALUE, // Show voxel pos/value as mouse moves (default when no
                      // button)
    MOUSE_SCROLL_IMAGE, // Change image number as mouse moves up/down (bound to
                        // mouse 1)
    MOUSE_CHANGE_WINCW // Change window center/level as mouse moves (bound to
                       // mouse 2)
  };

  private boolean _mouse1Down = false;
  private boolean _mouse2Down = false;
  private SbVec2s _mousePosition = new SbVec2s((short) 0, (short) 0);
  private MouseMode _mouseMode = MouseMode.MOUSE_SHOW_VALUE;

  public static void printHelp()
  {
    System.out.println("Launching example: " + EXAMPLE_NAME);
    System.out.println("Mouse buttons");
    System.out.println("  - In Selection mode (the default):");
    System.out.println("      - No buttons: Voxel position and value are displayed as mouse moves.");
    System.out.println("      - Button 1: Image number changes as mouse moves up and down.");
    System.out.println("      - Button 2: Window level and width change as mouse moves.");
    System.out.println("      - Wheel   : Image number.");
    System.out.println("  - In Viewing mode (press ESC to toggle):");
    System.out.println("      - Button 1: Zoom in and out as mouse moves.");
    System.out.println("      - Button 2: Pan image as mouse moves.");
    System.out.println("      - Wheel   : Zoom.");
    System.out.println("Hot keys:");
    System.out.println("  - A : Switch to Axial (Transverse) view.");
    System.out.println("  - C : Switch to Coronal view.");
    System.out.println("  - S : Switch to Sagittal view.");
    System.out.println("  - R : Reset window level and width.");
    System.out.println("  - H : Reset pan and zoom (default view).");
    System.out.println("  - I : Reset image numbers to initial state.");
  }

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain(EXAMPLE_NAME);
  }

  @Override
  public void start()
  {
    printHelp();

    // Load example resources
    try
    {
      _dataPath = (new File(Main.class.getResource(DATA_PATH).toURI())).toString();
      _annotation = (new File(Main.class.getResource(ANNOTATION).toURI()).toString());
    }
    catch (Exception e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load resources", e);
      return;
    }

    _renderArea = ViewerComponentsFactory.createRenderAreaExaminer();
    _renderArea.setCameraType(CameraMode.ORTHOGRAPHIC);
    _renderArea.setNavigationMode(NavigationMode.PLANE);
    _renderArea.setInteractionMode(InteractionMode.SELECTION);

    buildSceneGraph();

    _renderArea.setSceneGraph(_root);

    // This is an image viewer and we will always begin viewing along the Z axis
    // because this is the axis orthogonal to the actual images. In many cases
    // this will be the medical Axial (HF) axis, but not always.
    // Get the medical axis corresponding to the volume Z axis.
    MedicalHelper.Axis medicalAxis = MedicalHelper.MedicalAxisFromViewAxis(com.openinventor.inventor.Axis.Z, _volData);
    System.out.println("  MedicalAxis = " + medicalAxis);
    MedicalHelper.orientView(medicalAxis, _renderArea.getSceneInteractor().getCamera(), _volData);
    _renderArea.saveCamera();

    final Component canvas = _renderArea.getComponent();
    canvas.setPreferredSize(new java.awt.Dimension(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    setLayout(new BorderLayout());
    add(canvas);
  }

  @Override
  public void stop()
  {
    _renderArea.dispose();
  }

  /**
   * Build scene graph for Dicom text annotations.
   */
  private void buildAnnotations(SoGroup root)
  {
    SoMaterial dicomMatl = new SoMaterial();
    dicomMatl.diffuseColor.setValue(0.8f, 0.8f, 0.5f);
    root.addChild(dicomMatl);

    DicomInfo upperLeft = new DicomInfo();
    upperLeft.fileName.setValue(_annotation);
    upperLeft.position.setValue(-0.99f, 0.99f, 0);
    upperLeft.alignmentV.setValue(TextBox.AlignmentV.TOP);
    // Patient Name
    upperLeft.displayDicomInfo("", (short) 0x0010, (short) 0x0010);
    // Patient Birth Date
    upperLeft.displayDicomInfo("", (short) 0x0010, (short) 0x0030);
    // Study Description
    upperLeft.displayDicomInfo("", (short) 0x0008, (short) 0x1030);
    // Series Description
    upperLeft.displayDicomInfo("", (short) 0x0008, (short) 0x103E);
    root.addChild(upperLeft);

    DicomInfo upperRight = new DicomInfo();
    upperRight.fileName.setValue(_annotation);
    upperRight.position.setValue(0.99f, 0.99f, 0);
    upperRight.alignmentH.setValue(TextBox.AlignmentH.RIGHT);
    upperRight.alignmentV.setValue(TextBox.AlignmentV.TOP);
    upperRight.textAlignH.setValue(TextBox.AlignmentH.RIGHT);
    // Institution
    upperRight.displayDicomInfo("", (short) 0x0008, (short) 0x0080);
    // Physician
    upperRight.displayDicomInfo("", (short) 0x0008, (short) 0x0090);
    // Model name
    upperRight.displayDicomInfo("", (short) 0x0008, (short) 0x1090);
    root.addChild(upperRight);

    DicomInfo lowerRight = new DicomInfo();
    lowerRight.fileName.setValue(_annotation);
    lowerRight.position.setValue(0.99f, -0.99f, 0);
    lowerRight.alignmentH.setValue(TextBox.AlignmentH.RIGHT);
    lowerRight.alignmentV.setValue(TextBox.AlignmentV.BOTTOM);
    lowerRight.textAlignH.setValue(TextBox.AlignmentH.RIGHT);
    // Modality
    lowerRight.displayDicomInfo("", (short) 0x0008, (short) 0x0060);
    // X-Ray Tube Current
    lowerRight.displayDicomInfo("mA: ", (short) 0x0018, (short) 0x1151);
    // KVP (Kilo Voltage Peak)
    lowerRight.displayDicomInfo("kV: ", (short) 0x0018, (short) 0x0060);
    // Acquisition date
    lowerRight.displayDicomInfo("", (short) 0x0008, (short) 0x0022);
    root.addChild(lowerRight);

    // This next TextBox is where we'll display dynamic info like image number,
    // voxel value, etc. So give it a different color to be more visible.
    SoMaterial dynInfoMatl = new SoMaterial();
    dynInfoMatl.diffuseColor.setValue(0.8f, 0.5f, 0.5f);
    root.addChild(dynInfoMatl);

    TextBox lowerLeft = new TextBox();
    lowerLeft.fontSize.setValue(17); // Slightly larger than default
    lowerLeft.position.setValue(-0.99f, -0.94f, 0); // Leave room for OIV logo
    lowerLeft.alignmentV.setValue(TextBox.AlignmentV.BOTTOM);
    lowerLeft.addLine(""); // Placeholder for image number
    lowerLeft.addLine(""); // Placeholder for window center/width
    lowerLeft.addLine(""); // Placeholder for voxel pos/value
    root.addChild(lowerLeft);

    // For user interface updates
    _infoDisplay = lowerLeft;
  }

  /**
   * Build the scene graph to be displayed.
   */
  private void buildSceneGraph()
  {
    _root = new SoSeparator();

    // Event handling
    SoEventCallback eventNode = new SoEventCallback();
    eventNode.addEventCallback(SoMouseButtonEvent.class, new OnMouseButton());
    eventNode.addEventCallback(SoLocation2Event.class, new OnMouseMove());
    eventNode.addEventCallback(SoMouseWheelEvent.class, new OnMouseWheel());
    eventNode.addEventCallback(SoKeyboardEvent.class, new OnKeyPress());
    _root.addChild(eventNode);

    // Data reader.
    // Generate a list of files from a filename mask and integer sequence.
    SoVRDicomFileReader volreader = new SoVRDicomFileReader();
    String filemask = _dataPath + File.separator + FILE_MASK;
    volreader.setFilenameList(filemask, FILE_FIRST, FILE_LAST);

    // Transform node for DICOM ImageOrientation.
    // We do not actually _need_ it for a 2D image viewer, but passing it to
    // dicomAdjustVolume() avoids a warning message.
    SoMatrixTransform volMatrix = new SoMatrixTransform();

    // Data node
    SoVolumeData volData = new SoVolumeData();
    volData.setReader(volreader);
    MedicalHelper.dicomAdjustVolume(volData, volMatrix);
    _root.addChild(volData);

    // Data range (from data file by default)
    SoDataRange dataRange = new SoDataRange();
    MedicalHelper.dicomAdjustDataRange(dataRange, volData);
    _root.addChild(dataRange);

    // Base material.
    // By default Open Inventor uses gray 0.8 to leave room for lighting to
    // brighten the image. For slice viewing it's better to use full intensity.
    SoMaterial volumeMaterial = new SoMaterial();
    volumeMaterial.diffuseColor.setValue(1.f, 1.f, 1.f);
    _root.addChild(volumeMaterial);

    // Transfer function (gray scale)
    SoTransferFunction transFunc = new SoTransferFunction();
    transFunc.predefColorMap.setValue(PredefColorMaps.INTENSITY);
    MedicalHelper.dicomCheckMonochrome1(transFunc, volData, false);
    _root.addChild(transFunc);

    // Ortho slice rendering
    // If we loaded a volume (more than 1 slice) then pick an "interesting"
    // slice in the middle for demoing. Else we may have loaded a single image.
    // We always start viewing the Z axis because the images are stacked along
    // the volume Z axis.
    int numSlices = volData.getDimension().getValue()[2];
    SoOrthoSlice orthoSlice = new SoOrthoSlice();
    orthoSlice.axis.setValue(SoOrthoSlice.AxisType.Z);
    orthoSlice.sliceNumber.setValue(numSlices > 1 ? (numSlices / 2) : 0);
    orthoSlice.interpolation.setValue(SoVolumeShape.Interpolations.MULTISAMPLE_12);
    _root.addChild(orthoSlice);

    // Background color of render area
    SoGradientBackground bgColor = new SoGradientBackground();
    bgColor.color0.setValue(0, 0.1f, 0.1f);
    bgColor.color1.setValue(0, 0.1f, 0.1f);
    _root.addChild(bgColor);

    // OIV Logo
    // Add Open Inventor logo at the left-bottom corner
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }
    _root.addChild(logoBackground);

    // Slice orientation markers
    // Note that the markers node needs to know the _medical_ axis,
    // so we cannot conect the markers axis directly from the slice axis. :-(
    _sliceMarkers = new SliceOrientationMarkers();
    _sliceMarkers.axis.setValue(MedicalHelper.MedicalAxisFromViewAxis(com.openinventor.inventor.Axis.Z, volData));
    _root.addChild(_sliceMarkers);

    // Dynamic scale bars
    SoSeparator scaleSep = MedicalHelper.buildSliceScaleBars(_renderArea.getSceneInteractor().getCamera());
    _root.addChild(scaleSep);

    // DICOM annotation
    buildAnnotations(_root);

    // Set up variables for UI
    _volData = volData;
    _dataRange = dataRange;
    _sliceNode = orthoSlice;
    _sliceNum = _sliceNode.sliceNumber.getValue();
    // Always display image stack (Z) by default
    _sliceAxis = SoOrthoSlice.AxisType.Z;
    _numSlices = volData.getDimension().getValue()[_sliceAxis.getValue()];
    _resetWinCtrWidth = MedicalHelper.dicomGetWindowCenterWidth(_dataRange);
    ui_updateSliceNum(_sliceNum, _numSlices);
    ui_updateWinCtrWidth(_resetWinCtrWidth.getX(), _resetWinCtrWidth.getY());
  }

  /**
   * Handle key press events. <br>
   * Note: When slice axis changes the SliceOrientation object will be updated
   * automatically because its field is connected from the slice's field.
   *
   */
  private class OnKeyPress extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoKeyboardEvent theEvent = (SoKeyboardEvent) node.getEvent();

      // 'H'ome: reset view
      if ( SoKeyboardEvent.isKeyPressEvent(theEvent, SoKeyboardEvent.Keys.H) )
      {
        _renderArea.restoreCamera();
        node.setHandled();
      }
      // 'I'mage reset
      // Reset image numbers to initial state.
      else if ( SoKeyboardEvent.isKeyPressEvent(theEvent, SoKeyboardEvent.Keys.I) )
      {
        _sliceNum = _numSlices > 1 ? (_numSlices / 2) : 0;
        _sliceNode.sliceNumber.setValue(_sliceNum);
        ui_updateSliceNum(_sliceNum, _numSlices);
        node.setHandled();
      }
      // 'R'eset window center/width to original values
      else if ( SoKeyboardEvent.isKeyPressEvent(theEvent, SoKeyboardEvent.Keys.R) )
      {
        MedicalHelper.dicomSetWindowCenterWidth(_dataRange, _resetWinCtrWidth);
        ui_updateWinCtrWidth(_resetWinCtrWidth.getX(), _resetWinCtrWidth.getY());
        node.setHandled();
      }
      // A : Axial view
      else if ( SoKeyboardEvent.isKeyPressEvent(theEvent, SoKeyboardEvent.Keys.A) )
      {
        goToNewAxis(MedicalHelper.Axis.AXIAL);
        node.setHandled();
      }
      // C : Coronal view
      else if ( SoKeyboardEvent.isKeyPressEvent(theEvent, SoKeyboardEvent.Keys.C) )
      {
        goToNewAxis(MedicalHelper.Axis.CORONAL);
        node.setHandled();
      }
      // S : Sagittal view
      else if ( SoKeyboardEvent.isKeyPressEvent(theEvent, SoKeyboardEvent.Keys.S) )
      {
        goToNewAxis(MedicalHelper.Axis.SAGITTAL);
        node.setHandled();
      }
    }
  }

  /**
   * Handle mouse button events --> Affects result of moving the mouse cursor
   *
   */
  private class OnMouseButton extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {

      SoMouseButtonEvent theEvent = (SoMouseButtonEvent) node.getEvent();

      if ( SoMouseButtonEvent.isButtonPressEvent(theEvent, SoMouseButtonEvent.Buttons.BUTTON1) )
      {
        _mouse1Down = true;
        // Don't change if we're already in a non-default mode.
        if ( !_mouse2Down )
          _mouseMode = MouseMode.MOUSE_SCROLL_IMAGE;
        node.setHandled();
      }
      else if ( SoMouseButtonEvent.isButtonReleaseEvent(theEvent, SoMouseButtonEvent.Buttons.BUTTON1) )
      {
        _mouse1Down = false;
        if ( !_mouse2Down )
          // Default when no button pressed
          _mouseMode = MouseMode.MOUSE_SHOW_VALUE;
        node.setHandled();
      }
      else if ( SoMouseButtonEvent.isButtonPressEvent(theEvent, SoMouseButtonEvent.Buttons.BUTTON2) )
      {
        _mouse2Down = true;
        if ( !_mouse1Down )
          _mouseMode = MouseMode.MOUSE_CHANGE_WINCW;
        node.setHandled();
      }
      else if ( SoMouseButtonEvent.isButtonReleaseEvent(theEvent, SoMouseButtonEvent.Buttons.BUTTON2) )
      {
        _mouse2Down = false;
        if ( !_mouse1Down )
          // Default when no button pressed
          _mouseMode = MouseMode.MOUSE_SHOW_VALUE;
        node.setHandled();
      }
      else if ( SoMouseButtonEvent.isButtonPressEvent(theEvent, SoMouseButtonEvent.Buttons.BUTTON3) )
      {
        node.setHandled();
      }
      else if ( SoMouseButtonEvent.isButtonReleaseEvent(theEvent, SoMouseButtonEvent.Buttons.BUTTON3) )
      {
        node.setHandled();
      }
      // Remember position
      _mousePosition = theEvent.getPosition();
    }
  }

  /**
   * Handle mouse move events for slices. Behavior depends on which mouse
   * buttons are pressed (if any).
   *
   */
  private class OnMouseMove extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {

      SoLocation2Event theEvent = (SoLocation2Event) node.getEvent();

      // Check what mode we are in.
      if ( _mouseMode == MouseMode.MOUSE_SHOW_VALUE )
      {
        // Check if the cursor is over any data objects.
        SoPickedPoint pickedPt = node.getPickedPoint();
        SbVec3i32 ijkPos = null;
        float value = 0;

        if ( pickedPt != null )
        {
          // The cursor is over something interesting (probably the ortho slice
          // because we set all the annotation to be unpickable).
          SoDetail detail = pickedPt.getDetail();
          if ( detail instanceof SoOrthoSliceDetail )
          {
            SoOrthoSliceDetail orthoDetail = (SoOrthoSliceDetail) detail;
            // Get picked voxel
            ijkPos = orthoDetail.getValueDataPos();
            // Get value of voxel
            value = (float) orthoDetail.getValueD();
          }
        }
        // Update the user interface
        ui_updateVoxelPosVal(ijkPos, value);
      }
      else if ( _mouseMode == MouseMode.MOUSE_SCROLL_IMAGE )
      {
        // ---------------------------------------------------------------
        // This change only applies to the current slice window.
        SbVec2s newPos = theEvent.getPosition();
        int delta = newPos.getY() - _mousePosition.getY();
        // Sometimes we get a stream of mouse move events with the same event
        // position.
        // This is probably a glitch in the mouse and/or Windows. Ignore this
        // case.
        if ( delta != 0 )
        {
          // Increment slice number and update display
          goToNewImage(delta);
        }
      }
      else if ( _mouseMode == MouseMode.MOUSE_CHANGE_WINCW )
      {
        // ---------------------------------------------------------------
        // This change applies to all slice windows.
        SbVec2s newPos = theEvent.getPosition();
        int deltaX = newPos.getX() - _mousePosition.getX();
        int deltaY = newPos.getY() - _mousePosition.getY();
        SbVec2f winCW = MedicalHelper.dicomGetWindowCenterWidth(_dataRange);
        winCW.setX(winCW.getX() + deltaY);
        winCW.setY(winCW.getY() + deltaX);
        if ( winCW.getY() < 1 )
          winCW.setY(1);
        MedicalHelper.dicomSetWindowCenterWidth(_dataRange, winCW);
        ui_updateWinCtrWidth(winCW.getX(), winCW.getY());
      }

      // Remember new position
      _mousePosition = theEvent.getPosition();
    }

  }

  /**
   * Handle mouse wheel events for slices --> Change the image number.<br>
   * For consistency with other applications:
   * <li>Mouse wheel forward decreases image number.
   * <li>Mouse wheel backward increases image number.
   */
  private class OnMouseWheel extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback cb)
    {
      SoMouseWheelEvent theEvent = (SoMouseWheelEvent) cb.getEvent();
      int delta = theEvent.getDelta();
      goToNewImage(delta);
    }
  }

  /**
   * Change to the next image in stack. If delta is negative, increment the
   * image number. Else decrement the image number.
   */
  private void goToNewImage(int delta)
  {
    if ( delta < 0 )
    {
      // Increment
      if ( _sliceNum < _numSlices - 1 )
      {
        _sliceNum++;
        _sliceNode.sliceNumber.setValue(_sliceNum);
      }
    }
    else if ( delta != 0 )
    {
      // Decrement
      if ( _sliceNum > 0 )
      {
        _sliceNum--;
        _sliceNode.sliceNumber.setValue(_sliceNum);
      }
    }

    // Update image number display
    ui_updateSliceNum(_sliceNum, _numSlices);
    // This (temporarily) invalidates the voxel value display
    ui_updateVoxelPosVal(null, 0);
  }

  /**
   * Change to a different axis
   */
  private void goToNewAxis(MedicalHelper.Axis medicalAxis)
  {
    // If we only have 1 slice there is no way to view from other axes. :-)
    if ( _numSlices == 1 )
    {
      LOGGER.log(Level.WARNING, "Only 1 slice loaded. Not possible to view other axes.");
      return;
    }
    System.out.println("  MedicalAxis = " + medicalAxis);
    // In each case:
    // - Remember the new axis.
    // - Flip the slice to the new axis.
    // - Orient the camera to the new axis and save new home position.
    // - Force slice number to be valid for the new axis.

    // Update slice node using view axis.
    // Remember that view axis and medical axis are not (necessarily) the same!
    // However view axis and slice axis enums are aligned.
    com.openinventor.inventor.Axis viewAxis = MedicalHelper.ViewAxisFromMedicalAxis(medicalAxis, _volData);
    _sliceAxis = SoOrthoSlice.AxisType.values()[viewAxis.getValue()];
    _numSlices = _volData.getDimension().getValue()[_sliceAxis.getValue()];
    if ( _sliceNum > (_numSlices - 1) )
    {
      _sliceNum = _numSlices - 1;
    }
    _sliceNode.axis.setValue(_sliceAxis);
    _sliceNode.sliceNumber.setValue(_sliceNum);

    // Update view
    MedicalHelper.orientView(medicalAxis, _renderArea.getSceneInteractor().getCamera(), _volData);
    _renderArea.saveCamera();

    // Update slice markers, slice number display and invalidate voxel value
    // display.
    _sliceMarkers.axis.setValue(medicalAxis); // Slice markers always use the
                                              // current medical axis
    ui_updateSliceNum(_sliceNum, _numSlices);
    ui_updateVoxelPosVal(null, 0);
  }

  /**
   * Update wherever the UI displays the slice number.
   *
   * NOTE! OIV numbers slices starting at 0, but medical applications typically
   * display the image number starting at 1.
   *
   */
  private void ui_updateSliceNum(int sliceNum, int numSlices)
  {
    String str = "Image  " + (sliceNum + 1) + "  /  " + numSlices;
    _infoDisplay.setLine(str, _line_sliceNum);
  }

  /**
   * Update wherever the UI displays the window center/width.
   *
   */
  private void ui_updateWinCtrWidth(float center, float width)
  {
    String str = "WL: " + (int) center + "  WW: " + (int) width;
    _infoDisplay.setLine(str, _line_ctrWidth);
  }

  /**
   * Update wherever the UI displays the voxel position and value. If values are
   * not valid, clear the display.
   */
  private void ui_updateVoxelPosVal(SbVec3i32 ijkPos, float value)
  {
    String str = "";

    if ( ijkPos != null )
    {
      // We have valid values, but which voxel coords are relevant depends on
      // the current slice orientation (Axial, Coronal, Sagittal).
      int i, j;
      switch ( _sliceAxis )
      {
      case Y :
        i = 0; // X
        j = 2; // Z
        break;
      case X :
        i = 1; // Y
        j = 2; // Z
        break;
      case Z :
      default:
        i = 0;
        j = 1;
        break;
      }
      str = "Pos: " + ijkPos.array[i] + " , " + ijkPos.array[j] + "  Val: " + value;
    }
    _infoDisplay.setLine(str, _line_voxValue);
  }

}