package medical.rendering.techniques.simpleshader;

import java.awt.BorderLayout;
import java.awt.Panel;
import java.io.FileNotFoundException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbDataType;
import com.openinventor.inventor.SbVec3i32;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.nodes.SoFragmentShader;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoShaderObject;
import com.openinventor.inventor.nodes.SoShaderParameter1i;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaExaminer;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;
import com.openinventor.volumeviz.nodes.SoVolumeShader;

import util.ViewerComponentsFactory;

public class SimpleShaderPanel extends Panel
{
  private static final Logger LOGGER = Logger.getLogger(SimpleShaderPanel.class.getName());
  private final IRenderAreaExaminer m_renderArea;

  // Simple shader program which inverts colors
  //@formatter:off
  public final String m_shaderProgram =
      "//!oiv_include <VolumeViz/vvizfnc_frag.h>\n"
    + "//!oiv_include <VolumeViz/vvizGetData_frag.h>\n"
    + "//!oiv_include <VolumeViz/vvizTransferFunction_frag.h>\n"
    + "uniform VVizDataSetId voldata;\n"
    + "uniform int transfer;\n"
    + "\n"
    + "vec4 VVizComputeFragmentColor(VVizDataSetId dataset, vec3 rayDir, inout VVizVoxelInfo voxelInfoFront, in VVizVoxelInfo voxelInfoBack, int mask)\n"
    + "{\n"
    + "  VVIZ_DATATYPE sf = VVizGetData(voldata,voxelInfoFront.texCoord);\n"
    + "  vec4 res = VVizTransferFunction(sf, transfer);\n"
    + "  res.xyz = vec3(1.) - res.xyz;\n"
    + "  return res;\n"
    + "}";
  //@formatter:on

  /**
   * Constructor
   *
   * @param shaderProgramPath
   */
  public SimpleShaderPanel(String shaderProgramPath)
  {
    initializeUI();
    SoSeparator rootSep = buildSceneGraph(shaderProgramPath);
    m_renderArea = ViewerComponentsFactory.createRenderAreaExaminer();
    m_renderArea.setSceneGraph(rootSep);
    m_renderArea.viewAll(new SbViewportRegion(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    add(m_renderArea.getComponent());
  }

  /**
   * Initialize the panel
   */
  private void initializeUI()
  {
    setLayout(new BorderLayout());
  }

  /**
   * @param shaderProgramPath
   * @return Scene graph
   */
  public SoSeparator buildSceneGraph(String shaderProgramPath)
  {
    // Add the background logo node
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }

    // Create volume data
    SbVec3i32 dataSize = new SbVec3i32(10, 10, 10);
    int dataVolume = dataSize.getX() * dataSize.getY() * dataSize.getZ();
    ByteBuffer buffer = ByteBuffer.allocateDirect(dataVolume);
    buffer.order(ByteOrder.nativeOrder());
    for ( int i = 0; i < dataVolume; i++ )
      buffer.put(i, (byte) (i % 256));
    SbDataType dataType = new SbDataType(SbDataType.DataTypes.UNSIGNED_BYTE);
    SbBox3f box = new SbBox3f(-10, -10, -10, 10, 10, 10);

    // Node holding the volume data
    SoVolumeData volumeData = new SoVolumeData();
    volumeData.data.setValue(dataSize, dataType, buffer);
    volumeData.extent.setValue(box);

    // Use a predefined colorMap (associate data values to colors)
    SoTransferFunction transfertFuncion = new SoTransferFunction();
    transfertFuncion.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.STANDARD);

    // Node in charge of drawing the volume
    SoVolumeRender volumeRender = new SoVolumeRender();
    volumeRender.samplingAlignment.setValue(SoVolumeRender.SamplingAlignments.DATA_ALIGNED);

    // Set up shader
    SoFragmentShader fragmentShader = getFragmentShader(shaderProgramPath);
    SoVolumeShader volumeShader = new SoVolumeRenderingQuality();
    volumeShader.shaderObject.set1Value(SoVolumeShader.ShaderPositions.FRAGMENT_COMPUTE_COLOR.getValue(),
        fragmentShader);

    // Assemble nodes
    SoSeparator rootSep = new SoSeparator();
    rootSep.setName("example_root");
    rootSep.addChild(logoBackground);
    rootSep.addChild(volumeShader);
    rootSep.addChild(volumeData);
    rootSep.addChild(transfertFuncion);
    rootSep.addChild(volumeRender);
    return rootSep;
  }

  /**
   * @param shaderProgramPath
   * @return SoFragmentShader node. Uses default shader program is param is null
   */
  public SoFragmentShader getFragmentShader(String shaderProgramPath)
  {
    SoFragmentShader fragmentShader = new SoFragmentShader();
    if ( shaderProgramPath != null )
    {
      fragmentShader.sourceType.setValue(SoShaderObject.SourceTypes.FILENAME);
      fragmentShader.sourceProgram.setValue(shaderProgramPath);
    }
    else
    {
      fragmentShader.sourceType.setValue(SoShaderObject.SourceTypes.GLSL_PROGRAM);
      fragmentShader.sourceProgram.setValue(m_shaderProgram);
    }

    // Create uniform parameters allowing shader to access textures

    // Transfer function is in texture unit 0 (default)
    SoShaderParameter1i paramTex0 = new SoShaderParameter1i();
    paramTex0.name.setValue("transfer");
    paramTex0.value.setValue(0);
    // Volume data is in texture unit 1 (default) SoShaderParameter1i
    SoShaderParameter1i paramTex1 = new SoShaderParameter1i();
    paramTex1.name.setValue("voldata");
    paramTex1.value.setValue(1);
    fragmentShader.parameter.set1Value(0, paramTex0);
    fragmentShader.parameter.set1Value(1, paramTex1);
    return fragmentShader;
  }

  public void destroy()
  {
    // Destroy NEWT resources
    m_renderArea.dispose();
  }
}
