package medical.rendering.visualization.medicalDTI;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.nodes.SoCamera;
import com.openinventor.inventor.nodes.SoEnvironment;
import com.openinventor.inventor.nodes.SoLightModel;
import com.openinventor.inventor.nodes.SoLineSet;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoTransform;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoVertexProperty;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaExaminer;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.medical.nodes.Gnomon;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  public static final String EXAMPLE_NAME = "Medical DTI";
  public static final String DTIFILENAME = "/medical/data/dtiLineSet/dti_10000_lines.iv";

  private final static Logger LOGGER = Logger.getLogger(Main.class.getName());
  private static String _dataFile;

  private IRenderAreaExaminer _renderArea;

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain(EXAMPLE_NAME);
  }

  @Override
  public void start()
  {

    try
    {
      _dataFile = (new File(Main.class.getResource(DTIFILENAME).toURI())).toString();
    }
    catch (Exception e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load resources", e);
      return;
    }

    _renderArea = ViewerComponentsFactory.createRenderAreaExaminer();

    _renderArea.setSceneGraph(buildSceneGraph());

    // Configure Camera
    SoCamera camera = _renderArea.getSceneInteractor().getCameraInteractor().getCamera();
    camera.position.setValue(180.f, 0.f, 0.f);
    camera.orientation.setValue(new SbVec3f(0.5f, 0.5f, 0.5f), (float) (2 * Math.PI / 3));
    _renderArea.viewAll(new SbViewportRegion(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    _renderArea.setTransparencyType(SoGLRenderAction.TransparencyTypes.NO_SORT);

    final Component canvas = _renderArea.getComponent();
    canvas.setPreferredSize(new java.awt.Dimension(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    setLayout(new BorderLayout());
    add(canvas);
  }

  @Override
  public void stop()
  {
    _renderArea.dispose();
  }

  /**
   * Create scene graph
   */
  private SoSeparator buildSceneGraph()
  {
    // Create the scene root
    SoSeparator root = new SoSeparator();

    // Load LineSet Dataset.
    SoSeparator lineSetRoot;
    SoLineSet fileLineSet;
    {
      SoInput input = new SoInput();
      if ( !input.openFile(_dataFile) )
      {
        System.out.println("Can't open");
      }
      lineSetRoot = SoDB.readAll(input);
      fileLineSet = (SoLineSet) lineSetRoot.getChild(0);
    }

    // Compute and add in vertexProperty the tangents and colors
    computeLineSetColorsAndTangents(fileLineSet);

    // Add an Environment node to activate AmbientOcclusion
    SoEnvironment env = new SoEnvironment();
    env.ambientOcclusion.setValue(true);
    root.addChild(env);

    // Add a lightModel with PER_PIXEL_PHONG lighting to activate tangent
    // lighting
    SoLightModel lightModel = new SoLightModel();
    lightModel.model.setValue(SoLightModel.Models.PER_PIXEL_PHONG);
    root.addChild(lightModel);

    // Add a Material to modulate the lighting
    SoMaterial mat = new SoMaterial();
    mat.ambientColor.setValue(0.8f, 0.8f, 0.8f);
    mat.diffuseColor.setValue(0.7f, 0.7f, 0.7f);
    mat.specularColor.setValue(0.2f, 0.2f, 0.2f);
    mat.shininess.setValue(1.0f);
    root.addChild(mat);

    // Align the LineSet to fit medical conventions with the Gnomon
    SoTransform transform = new SoTransform();
    transform.translation.setValue(-72.f, -76.f, -26.f);
    transform.rotation.setValue(new SbVec3f(-1.f, 0.f, 0.f), 0.5f);
    root.addChild(transform);

    root.addChild(fileLineSet);

    // Open Inventor logo
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }
    root.addChild(logoBackground);

    // Medical Gnomon.
    root.addChild(new Gnomon());

    return root;
  }

  // Compute DTI lines colors based on tangents
  private void computeLineSetColorsAndTangents(SoLineSet lineSet)
  {
    SoVertexProperty vp = (SoVertexProperty) lineSet.vertexProperty.getValue();

    SbVec3f[] vertices = vp.vertex.getValues(0);
    int numLines = lineSet.numVertices.getNum();
    int[] numVertByLine = lineSet.numVertices.getValues(0);

    vp.tangentBinding.setValue(SoVertexProperty.Bindings.PER_VERTEX);
    vp.materialBinding.setValue(SoVertexProperty.Bindings.PER_VERTEX);
    SbVec3f[] tangents = new SbVec3f[vp.vertex.getNum()];
    int[] colors = new int[vp.vertex.getNum()];

    int vtxIt = 0;
    for ( int i = 0; i < numLines; i++ )
    {
      int numVerts = numVertByLine[i];

      if ( numVerts == 0 )
        continue;

      {
        SbVec3f tangent = vertices[vtxIt + 1].plus(vertices[vtxIt]);
        tangent.normalize();
        tangents[vtxIt] = tangent;
        SbColor color = new SbColor(Math.abs(tangent.getX()), Math.abs(tangent.getY()), Math.abs(tangent.getZ()));
        colors[vtxIt++] = color.getPackedValue();
      }

      if ( numVerts == 1 )
        continue;

      for ( int j = 1; j < numVerts - 1; j++ )
      {
        SbVec3f prevVec = vertices[vtxIt].minus(vertices[vtxIt - 1]);
        SbVec3f nextVec = vertices[vtxIt + 1].minus(vertices[vtxIt]);

        SbVec3f tangent = nextVec.plus(prevVec);
        tangent.normalize();
        tangents[vtxIt] = tangent;
        SbColor color = new SbColor(Math.abs(tangent.getX()), Math.abs(tangent.getY()), Math.abs(tangent.getZ()));
        colors[vtxIt++] = color.getPackedValue();
      }

      {
        SbVec3f tangent = vertices[vtxIt].minus(vertices[vtxIt - 1]);
        tangent.normalize();
        tangents[vtxIt] = tangent;
        SbColor color = new SbColor(Math.abs(tangent.getX()), Math.abs(tangent.getY()), Math.abs(tangent.getZ()));
        colors[vtxIt++] = color.getPackedValue();
      }
    }

    vp.tangent.setValues(0, tangents);
    vp.orderedRGBA.setValues(0, colors);
  }

}
