package medical.rendering.visualization.medicalDTIViewer;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.nodes.SoCamera;
import com.openinventor.inventor.nodes.SoComplexity;
import com.openinventor.inventor.nodes.SoInteractiveComplexity;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoMatrixTransform;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor;
import com.openinventor.ldm.nodes.SoDataRange;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.medical.nodes.Gnomon;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  public static final String EXAMPLE_NAME = "Medical DTI Viewer";
  public static final String DTIFILENAME = "/medical/data/dicomSample/DTI/DTI.iv";
  public static final String COLORMAPFILE = "/medical/data/resources/volrenGlow.am";
  public static final String DCMFILENAME = "/medical/data/dicomSample/DTI/list.dcm";
  public static final String ANNOTATION = "/medical/data/dicomSample/DTI/T1-BrainOnly-0001.dcm";

  private final static Logger LOGGER = Logger.getLogger(Main.class.getName());
  private static String _dataFile;
  private static String _colorMapFile;
  private static String _dcmFile;
  private static String _annotation;
  private SoVolumeData _volData;
  private IRenderAreaExaminer _renderArea;

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain(EXAMPLE_NAME);
  }

  @Override
  public void start()
  {
    // Load example resources
    try
    {
      _dataFile = (new File(Main.class.getResource(DTIFILENAME).toURI())).toString();
      _colorMapFile = (new File(Main.class.getResource(COLORMAPFILE).toURI())).toString();
      _dcmFile = (new File(Main.class.getResource(DCMFILENAME).toURI())).toString();
      _annotation = (new File(Main.class.getResource(ANNOTATION).toURI()).toString());
    }
    catch (Exception e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load resources", e);
      return;
    }

    _renderArea = ViewerComponentsFactory.createRenderAreaExaminer();
    _renderArea.getSceneInteractor().setCameraMode(SceneInteractor.CameraMode.ORTHOGRAPHIC);

    _renderArea.setSceneGraph(buildSceneGraph());
    _renderArea.setTransparencyType(SoGLRenderAction.TransparencyTypes.OPAQUE_FIRST);
    SoCamera camera = _renderArea.getSceneInteractor().getCameraInteractor().getCamera();
    MedicalHelper.orientView(MedicalHelper.Axis.AXIAL, camera, _volData, 0.8f);

    final Component canvas = _renderArea.getComponent();
    canvas.setPreferredSize(new java.awt.Dimension(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    setLayout(new BorderLayout());
    add(canvas);
  }

  @Override
  public void stop()
  {
    _renderArea.dispose();
  }

  /**
   * Create scene graph
   */
  private SoSeparator buildSceneGraph()
  {
    // Create the scene root
    SoSeparator root = new SoSeparator();

    SoSeparator lineSep = MedicalHelper.readFile(_dataFile);
    if ( lineSep != null )
      root.addChild(lineSep);

    // Create root node for volume rendering
    SoSeparator sepVR = new SoSeparator();
    root.addChild(sepVR);
    sepVR.setName("Root_VR");

    // Node to hold the volume data
    _volData = new SoVolumeData();
    _volData.fileName.setValue(_dcmFile);
    SoMatrixTransform transform = new SoMatrixTransform();
    MedicalHelper.dicomAdjustVolume(_volData, transform);
    sepVR.addChild(_volData);

    // Load the ColorMap
    SoTransferFunction transFunc = new SoTransferFunction();
    transFunc.loadColormap(_colorMapFile);
    transFunc.maxValue.setValue(87);
    sepVR.addChild(transFunc);

    // Re-map data range to full range of data
    SoDataRange volRange = new SoDataRange();
    MedicalHelper.dicomAdjustDataRange(volRange, _volData);
    sepVR.addChild(volRange);

    // Change complexity when the dataset move
    SoInteractiveComplexity interactiveComplexity = new SoInteractiveComplexity();
    interactiveComplexity.fieldSettings.set1Value(0, "SoComplexity value 0.3 0.9");
    interactiveComplexity.refinementDelay.setValue(0);
    sepVR.addChild(interactiveComplexity);
    sepVR.addChild(new SoComplexity());

    // Property node which allows SoVolumeRender to draw high quality volumes
    SoVolumeRenderingQuality volQuality = new SoVolumeRenderingQuality();
    volQuality.interpolateOnMove.setValue(true);
    volQuality.preIntegrated.setValue(true);
    volQuality.deferredLighting.setValue(false);
    volQuality.ambientOcclusion.setValue(false);
    sepVR.addChild(volQuality);

    // To play with VolumeRendering transparency
    SoMaterial volMat = new SoMaterial();
    volMat.transparency.setValue(0.9f);
    sepVR.addChild(volMat);

    // Node in charge of drawing the volume
    SoVolumeRender volRender = new SoVolumeRender();
    volRender.numSlicesControl.setValue(SoVolumeRender.NumSlicesControls.AUTOMATIC);
    volRender.subdivideTile.setValue(true);
    volRender.interpolation.setValue(SoVolumeRender.Interpolations.CUBIC);
    volRender.samplingAlignment.setValue(SoVolumeRender.SamplingAlignments.BOUNDARY_ALIGNED);
    sepVR.addChild(volRender);

    // Open Inventor logo
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }
    root.addChild(logoBackground);

    // Medical Gnomon.
    root.addChild(new Gnomon());

    // DICOM annotation
    root.addChild(MedicalHelper.exampleDicomAnnotation(_annotation));

    return root;
  }

}
