package medical.rendering.visualization.medicalmip;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.manips.SoTransformerManip;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.InteractionMode;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor.CameraMode;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.volumeviz.nodes.SoVolumeClippingGroup;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  private static final String EXAMPLE_NAME = "Medical MIP";
  private static final String DATA = "/medical/data/files/3DHEAD.ldm";
  private final static SbVec3f m_slabDefaultScale = new SbVec3f(0.15f, 1, 1);

  private final static Logger LOGGER = Logger.getLogger(Main.class.getName());

  private IRenderAreaExaminer viewer3D = null;
  private int m_renderMode;
  private SoVolumeRender m_volumeRender;
  private SoTransferFunction m_transfertFunction;
  private SoTransformerManip m_manip;
  private SoText2 m_helpText;


  public static void printHelp()
  {
    System.out.println("Launching example: " + EXAMPLE_NAME);
    System.out.println("LeftClick a face of the bounding box to drag along two axis");
    System.out.println("LeftClick a 'white cube' to scale the dragger along all axis");
    System.out.println("LeftClick a 'green ball' to rotate the dragger around one axis");
    System.out.println("Shift+LeftClick a 'white cube' to scale the dragger along one axis");
    System.out.println("Shift+LeftClick a 'green ball' to rotate the dragger around all axis");
    System.out.println("Press R on the 3D viewer to reset the dragger fields");
  }

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain(EXAMPLE_NAME);
  }

  /**
   * Keyboard callback.
   */
  private class CustomKeyboardEventCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoEvent evt = node.getEvent();
      if ( evt != null )
      {
        if ( SoKeyboardEvent.isKeyPressEvent(evt, SoKeyboardEvent.Keys.PAD_ADD) )
        {
          m_renderMode = (m_renderMode + 1) % 4;
          switch ( m_renderMode )
          {
          case 0 : // Min
            m_volumeRender.renderMode.setValue(SoVolumeRender.RenderModes.MIN_INTENSITY_PROJECTION);
            m_transfertFunction.maxValue.setValue(93);
            break;
          case 1 : // Max
            m_volumeRender.renderMode.setValue(SoVolumeRender.RenderModes.MAX_INTENSITY_PROJECTION);
            m_transfertFunction.maxValue.setValue(255);
            break;
          case 2 : // Average
            m_volumeRender.renderMode.setValue(SoVolumeRender.RenderModes.AVERAGE_INTENSITY_PROJECTION);
            m_transfertFunction.maxValue.setValue(155);
            break;
          case 3 : // MIDA
            m_volumeRender.renderMode.setValue(SoVolumeRender.RenderModes.MAX_INTENSITY_DIFFERENCE_ACCUMULATION);
            m_transfertFunction.maxValue.setValue(255);
            break;
          }
          m_helpText.string.set1Value(0, m_volumeRender.renderMode.get());
        }
        else if ( SoKeyboardEvent.isKeyPressEvent(evt, SoKeyboardEvent.Keys.R) )
        {
          // Reset manip fields
          m_manip.translation.setValue(0, 0, 0);
          m_manip.rotation.setValue(SbRotation.identity());
          m_manip.scaleFactor.setValue(m_slabDefaultScale);
          m_manip.rotation.setValue(new SbRotation(new SbVec3f(0,0,1), (float)Math.PI/-2.f));
        }
      }
    }
  }


  @Override
  public void start()
  {
    printHelp();

    viewer3D = ViewerComponentsFactory.createRenderAreaExaminer();
    viewer3D.setCameraType(CameraMode.ORTHOGRAPHIC);
    viewer3D.setInteractionMode(InteractionMode.SELECTION);
      
    viewer3D.setSceneGraph(buildSceneGraph());
    
    // Adjust camera pose to have a nicer view of the scene
    var camRot1 = new SbRotation(new SbVec3f(1, 0, 0), (float)Math.PI);
    var camRot2 = new SbRotation( new SbVec3f(0,1,0), (float)Math.PI/3 );
    camRot1.multiply(camRot2);
    var camRot3 = new SbRotation( new SbVec3f(0,0,1), (float)Math.PI/30 );
    camRot1.multiply(camRot3);

    viewer3D.getSceneInteractor().getCamera().orientation.setValue(camRot1);
    viewer3D.viewAll(new SbViewportRegion((short)800, (short)600));

    final Component component = viewer3D.getComponent();
    component.setPreferredSize(new java.awt.Dimension(800, 600));
    setLayout(new BorderLayout());
    add(component);
  }
  
  private SoNode buildSceneGraph()
  {
    // Load example resources
    File dataFile;
    try
    {
      dataFile = new File(Main.class.getResource(DATA).toURI());
    }
    catch (Exception e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load resources", e);
      return null;
    }
    String dataPath = dataFile.toString();
    
    SoSeparator viewer3DRoot = new SoSeparator();

    // Define keyboard callback on the 3D viewer
    SoEventCallback eventCB = new SoEventCallback();
    eventCB.addEventCallback(SoKeyboardEvent.class, new CustomKeyboardEventCB());
    viewer3DRoot.addChild(eventCB);

    // Load background logo node
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }
    viewer3DRoot.addChild(logoBackground);

    // SoTransformerManip allows user to drag/scale/rotate clipping box
    SoTransformSeparator protectManip = new SoTransformSeparator();
    m_manip = new SoTransformerManip();
    m_manip.scaleFactor.setValue(m_slabDefaultScale);
    protectManip.addChild(m_manip);
    viewer3DRoot.addChild(protectManip);

    // Create the volume clipping group
    SoVolumeClippingGroup clippingGroup = new SoVolumeClippingGroup();
    viewer3DRoot.addChild(clippingGroup);

    // SoTransformSeparator prevents local transform from "leaking" out of
    // clipping group
    SoTransformSeparator localProtection = new SoTransformSeparator();
    clippingGroup.addChild(localProtection);

    // This transform controls the position/size/rotation of the clipping
    // geometry. Its fields are connected from the manip's fields so it
    // automatically tracks the changes that the user makes to the manip.
    SoTransform localTransform = new SoTransform();
    localTransform.setName("Local_Transfo");
    localTransform.translation.connectFrom(m_manip.translation);
    localTransform.rotation.connectFrom(m_manip.rotation);
    localTransform.scaleFactor.connectFrom(m_manip.scaleFactor);
    localTransform.scaleOrientation.connectFrom(m_manip.scaleOrientation);
    localTransform.center.connectFrom(m_manip.center);
    localProtection.addChild(localTransform);

    // Add the actual clipping geometry
    localProtection.addChild(new SoCube());

    SoVolumeData volumeData;
    // Sub-graph containing the data
    SoSeparator dataSep = new SoSeparator();
    viewer3DRoot.addChild(dataSep);
    {
      // Flip the data (because by default it is upside down)
      SoRotationXYZ dataRot = new SoRotationXYZ();
      dataRot.axis.setValue(0);
      dataRot.angle.setValue((float) Math.PI);
      //dataSep.addChild(dataRot);

      // Node holding the volume data
      volumeData = new SoVolumeData();
      volumeData.fileName.setValue(dataPath);
      dataSep.addChild(volumeData);

      // Describes the association between data data values and colors
      m_transfertFunction = new SoTransferFunction();
      m_transfertFunction.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.INTENSITY);
      m_transfertFunction.maxValue.setValue(255);
      dataSep.addChild(m_transfertFunction);

      // Node in charge of drawing the volume
      m_volumeRender = new SoVolumeRender();
      m_volumeRender.renderMode.setValue(SoVolumeRender.RenderModes.MAX_INTENSITY_PROJECTION);
      m_volumeRender.numSlicesControl.setValue(SoVolumeRender.NumSlicesControls.AUTOMATIC);
      m_volumeRender.samplingAlignment.setValue(SoVolumeRender.SamplingAlignments.BOUNDARY_ALIGNED);
      dataSep.addChild(m_volumeRender);
      m_renderMode = 1;
    }
    
    viewer3DRoot.addChild(createBBox(volumeData.extent.getValue()));
    
    var helpSep = new SoSeparator();
    viewer3DRoot.addChild(helpSep);
    {
      var bboxNode = new SoBBox();
      bboxNode.mode.setValue(SoBBox.Modes.NO_BOUNDING_BOX);
      helpSep.addChild(bboxNode);

      var orthoCam = new SoOrthographicCamera();
      helpSep.addChild(orthoCam);
      orthoCam.viewportMapping.setValue(SoCamera.ViewportMappings.LEAVE_ALONE);

      var helpColor = new SoBaseColor();
      helpSep.addChild(helpColor);
      helpColor.rgb.set1Value(0, new SbVec3f(1, 1, 1));

      var helpTrans = new SoTranslation();
      helpSep.addChild(helpTrans);
      helpTrans.translation.setValue(new SbVec3f(0, -.88f, 0));

      var helpFont = new SoFont();
      helpFont.name.setValue("Arial:Bold");
      helpSep.addChild(helpFont);
      helpFont.size.setValue(15);

      m_helpText = new SoText2();
      m_helpText.justification.setValue(SoText2.Justifications.CENTER);
      helpSep.addChild(m_helpText);
      m_helpText.string.setNum(2);
      m_helpText.string.set1Value(0, m_volumeRender.renderMode.get());
      m_helpText.string.set1Value(1, "Press keypad '+' to change projection mode");
    }
    return viewer3DRoot;
  }
  
  // Helper for Bbox creation
  private SoSeparator createBBox(SbBox3f bbox)
  {
    var bboxSep = new SoSeparator();

    SoPickStyle pickStyle = new SoPickStyle();
    bboxSep.addChild(pickStyle);
    pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);

    var drawStyle = new SoDrawStyle();
    bboxSep.addChild(drawStyle);
    drawStyle.style.setValue(SoDrawStyle.Styles.LINES);
    drawStyle.lineWidth.setValue(2.0f);

    var bboxMat = new SoBaseColor();
    bboxMat.rgb.setValue(new SbVec3f(0.8f, 0.4f, 0));
    bboxSep.addChild(bboxMat);

    var lightModel = new SoLightModel();
    bboxSep.addChild(lightModel);
    lightModel.model.setValue(SoLightModel.Models.BASE_COLOR);

    var translation = new SoTranslation();
    bboxSep.addChild(translation);
    translation.translation.setValue(bbox.getCenter());

    var cube = new SoCube();
    SbVec3f bboxSize = bbox.getSize();
    cube.width.setValue(bboxSize.getValueAt(0));
    cube.height.setValue(bboxSize.getValueAt(1));
    cube.depth.setValue(bboxSize.getValueAt(2));
    bboxSep.addChild(cube);

    return bboxSep;
  }


  @Override
  public void stop()
  {
    viewer3D.dispose();
  }
}
