package medical.segmentation.simpleclippinggroup;

import java.awt.BorderLayout;
import java.awt.Panel;
import java.io.FileNotFoundException;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.nodes.SoComplexity;
import com.openinventor.inventor.nodes.SoDrawStyle;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoRotationXYZ;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoSphere;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaExaminer;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.volumeviz.nodes.SoVolumeClippingGroup;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;
import com.openinventor.volumeviz.nodes.SoVolumeShape;

import util.ViewerComponentsFactory;

public class SimpleClippingGroupPanel extends Panel
{
  private static final Logger LOGGER = Logger.getLogger(SimpleClippingGroupPanel.class.getName());
  private final IRenderAreaExaminer m_renderArea;

  /**
   * Constructor
   *
   * @param dataPath
   * @param colormapPath
   */
  public SimpleClippingGroupPanel(String dataPath, String colormapPath)
  {
    initializeUI();
    SoSeparator rootSep = buildSceneGraph(dataPath, colormapPath);
    m_renderArea = ViewerComponentsFactory.createRenderAreaExaminer();
    m_renderArea.setSceneGraph(rootSep);
    m_renderArea.viewAll(new SbViewportRegion(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    add(m_renderArea.getComponent());
  }

  /**
   * Initialize the panel
   */
  private void initializeUI()
  {
    setLayout(new BorderLayout());
  }

  /**
   * @param dataPath
   * @param colormapPath
   * @return Scene graph
   */
  public SoSeparator buildSceneGraph(String dataPath, String colormapPath)
  {
    // Add the background logo node
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }

    // Set the complexity value to its maximum
    // For example, this setting increases the sphere number of points
    SoComplexity complexity = new SoComplexity();
    complexity.value.setValue(1.0f);

    // Node holding the volume data
    SoVolumeData volumeData = new SoVolumeData();
    volumeData.fileName.setValue(dataPath);

    // Use a predefined colorMap (associate data values to colors)
    SoTransferFunction transfertFuncion = new SoTransferFunction();
    transfertFuncion.loadColormap(colormapPath);
    transfertFuncion.minValue.setValue(30);
    transfertFuncion.maxValue.setValue(159);

    // Node in charge of drawing the volume
    SoVolumeRender volumeRender = new SoVolumeRender();
    volumeRender.samplingAlignment.setValue(SoVolumeRender.SamplingAlignments.BOUNDARY_ALIGNED);
    volumeRender.interpolation.setValue(SoVolumeShape.Interpolations.CUBIC);

    // Improve data rendering by adding the following node
    SoVolumeRenderingQuality volumeRenderingQuality = new SoVolumeRenderingQuality();
    volumeRenderingQuality.ambientOcclusion.setValue(true);
    volumeRenderingQuality.deferredLighting.setValue(true);

    // Node to rotate the volume (to see the face in profile)
    SoRotationXYZ rotation = new SoRotationXYZ();
    rotation.angle.setValue((float) Math.PI);

    // Define the clipping geometry
    SoSphere sphere = new SoSphere();
    sphere.radius.setValue(0.6f);
    SoDrawStyle sphereDrawStyle = new SoDrawStyle();
    sphereDrawStyle.style.setValue(SoDrawStyle.Styles.POINTS);

    // Node to clip the volume data with a geometry
    SoVolumeClippingGroup volumeClippingGroup = new SoVolumeClippingGroup();
    volumeClippingGroup.addChild(sphere);

    // Assemble nodes
    SoSeparator rootSep = new SoSeparator();
    rootSep.setName("root");
    if ( logoBackground != null )
      rootSep.addChild(logoBackground);
    rootSep.addChild(complexity);
    rootSep.addChild(rotation);
    SoSeparator headSep = new SoSeparator();
    headSep.setName("head");
    headSep.addChild(volumeClippingGroup);
    headSep.addChild(volumeData);
    headSep.addChild(transfertFuncion);
    headSep.addChild(volumeRenderingQuality);
    headSep.addChild(volumeRender);
    rootSep.addChild(headSep);
    SoSeparator sphereSep = new SoSeparator();
    sphereSep.setName("sphere");
    sphereSep.addChild(sphereDrawStyle);
    sphereSep.addChild(sphere);
    rootSep.addChild(sphereSep);
    return rootSep;
  }

  public void destroy()
  {
    // Destroy NEWT resources
    m_renderArea.dispose();
  }
}
