package medical.tools.medicalMagnifier;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.nodes.SoCamera;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor;
import com.openinventor.ldm.nodes.SoDataRange;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.medical.nodes.Magnifier;
import com.openinventor.medical.nodes.TextBox;
import com.openinventor.volumeviz.nodes.SoOrthoSlice;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeShape;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  public static final String EXAMPLE_NAME = "Medical Magnifier";
  public static final String DICO_DATA = "/medical/data/dicomSample/listOfDicomFiles.dcm";
  public static final String ANNOTATION = "/medical/data/dicomSample/CVH001.dcm";

  private final static Logger LOGGER = Logger.getLogger(Main.class.getName());
  private SoSeparator _root;
  private static String _dicom;
  private static String _annotation;
  private IRenderAreaExaminer _renderArea;

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain(EXAMPLE_NAME);
  }

  @Override
  public void start()
  {
    // Load example resources
    try
    {
      _dicom = (new File(Main.class.getResource(DICO_DATA).toURI())).toString();
      _annotation = (new File(Main.class.getResource(ANNOTATION).toURI())).toString();
    }
    catch (Exception e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load resources", e);
      return;
    }

    _renderArea = ViewerComponentsFactory.createRenderAreaExaminer();
    _renderArea.getSceneInteractor().setCameraMode(SceneInteractor.CameraMode.ORTHOGRAPHIC);
    buildSceneGraph();

    _renderArea.setSceneGraph(_root);
    _renderArea.viewAll(new SbViewportRegion(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    _renderArea.getSceneInteractor().setInteractionMode(SceneExaminer.InteractionMode.SELECTION);
    MedicalHelper.dollyZoom(1.5f, _renderArea.getSceneInteractor().getCameraInteractor());
    _renderArea.saveCamera();

    final Component canvas = _renderArea.getComponent();
    canvas.setPreferredSize(new java.awt.Dimension(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    setLayout(new BorderLayout());
    add(canvas);
  }

  @Override
  public void stop()
  {
    _renderArea.dispose();
  }

  /**
   * Build the scene graph to be displayed.
   */
  private void buildSceneGraph()
  {
    _root = new SoSeparator();

    SoCamera mainCam = _renderArea.getSceneInteractor().getCameraInteractor().getCamera();
    mainCam.orientation.setValue(new SbVec3f(0, 0, 1), (float) Math.PI);
    _root.addChild(mainCam);

    SoSeparator scene = new SoSeparator();
    _root.addChild(scene);

    // Data
    SoVolumeData volData = new SoVolumeData();
    volData.fileName.setValue(_dicom);
    MedicalHelper.dicomAdjustVolume(volData, true);
    scene.addChild(volData);

    SoTransferFunction transFunc = new SoTransferFunction();
    transFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.INTENSITY);
    scene.addChild(transFunc);

    SoDataRange dataRange = new SoDataRange();
    MedicalHelper.dicomAdjustDataRange(dataRange, volData);
    scene.addChild(dataRange);

    SoMaterial material = new SoMaterial();
    material.diffuseColor.setValue(1, 1, 1); // Display true intensity values
    scene.addChild(material);

    // Otho slice rendering & manipulation
    SoOrthoSlice orthoSlice = new SoOrthoSlice();
    orthoSlice.sliceNumber.setValue(volData.data.getSize().getZ() / 2);
    orthoSlice.interpolation.setValue(SoVolumeShape.Interpolations.MULTISAMPLE_12);
    scene.addChild(orthoSlice);

    // Magnifier
    Magnifier myMagnifier = new Magnifier();
    myMagnifier.sceneToMagnify.setValue(scene);
    myMagnifier.magnifierColor.setValue(new SbColor(0, 0.8f, 0));
    // Magnifier initial position is based on volume extent.
    // Move it to the front and offset left so it's easier to see.
    SbBox3f volExt = volData.extent.getValue();
    SbVec3f magPos = volExt.getCenter();
    magPos.setX(magPos.getX() - (volExt.getSize().getX() / 4)); // Offset left
    magPos.setZ(volExt.getMax().getZ()); // In front of the volume
    myMagnifier.translation.setValue(magPos);
    _root.addChild(myMagnifier);

    // Add some typical image viewer annotation
    _root.addChild(MedicalHelper.buildSliceAnnotation(mainCam, orthoSlice, _annotation));

    // Add Open Inventor logo at the left-bottom corner
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }
    _root.addChild(logoBackground);

    // Give the user a hint what to do...
    SoMaterial textMat = new SoMaterial();
    textMat.diffuseColor.setValue(0.5f, 1, 0.5f);
    _root.addChild(textMat);
    TextBox hints = new TextBox();
    hints.alignmentV.setValue(TextBox.AlignmentV.BOTTOM);
    hints.position.setValue(-0.99f, -0.85f, 0); // Normalized device coords
                                                // -1..1
    hints.addLine("Click and drag to move magnifier");
    _root.addChild(hints);
  }
}
