package medical.tools.roimanip;

import java.awt.*;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.FileNotFoundException;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.UIManager;
import javax.swing.border.TitledBorder;

import com.openinventor.inventor.SbVec3i32;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.actions.SoSearchAction;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor.CameraMode;
import com.openinventor.ldm.manips.SoROIManip;
import com.openinventor.ldm.nodes.SoDataRange;
import com.openinventor.ldm.nodes.SoROI;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;
import com.openinventor.volumeviz.nodes.SoVolumeSkin;

import medical.tools.rotateroi.RotateRoiPanel;
import util.ViewerComponentsFactory;

public class RoiManipPanel extends JPanel
{
  private static final Logger LOGGER = Logger.getLogger(RotateRoiPanel.class.getName());
  private IRenderAreaInteractive m_renderArea;
  private SoSeparator m_rootSep;
  private JPanel m_renderAreaPanel;
  private SoROI m_ROI;
  private SoROIManip m_ROIManip;
  private SoSwitch m_renderSwitch;

  /**
   * Constructor
   *
   * @param dataPath
   */
  public RoiManipPanel(String dataPath)
  {
    initializeUI();
    m_rootSep = buildSceneGraph(dataPath);
    m_renderArea = ViewerComponentsFactory.createRenderAreaInteractive();
    m_renderArea.setCameraType(CameraMode.ORTHOGRAPHIC);
    m_renderArea.setSceneGraph(m_rootSep);
    m_renderArea.viewAll(new SbViewportRegion(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    m_renderAreaPanel.add(m_renderArea.getComponent());
  }

  /**
   * Initialize the panel
   */
  private void initializeUI()
  {
    setLayout(new BoxLayout(this, BoxLayout.X_AXIS));

    m_renderAreaPanel = new JPanel();
    m_renderAreaPanel.setLayout(new BorderLayout());
    m_renderAreaPanel.setPreferredSize(new Dimension(0, 0));
    this.add(m_renderAreaPanel);

    JPanel settingsPanel = new JPanel();
    settingsPanel.setPreferredSize(new Dimension(200, 200));
    this.add(settingsPanel);
    settingsPanel.setLayout(new BoxLayout(settingsPanel, BoxLayout.Y_AXIS));

    JPanel constantFieldsPanel = new JPanel();
    constantFieldsPanel.setMaximumSize(new Dimension(10000, 0));
    constantFieldsPanel.setSize(new Dimension(10, 10));
    constantFieldsPanel.setBorder(new TitledBorder(UIManager.getBorder("TitledBorder.border"), "Nodes",
        TitledBorder.LEADING, TitledBorder.TOP, null, new Color(0, 0, 0)));
    settingsPanel.add(constantFieldsPanel);
    GridBagLayout gbl_constantFieldsPanel = new GridBagLayout();
    gbl_constantFieldsPanel.columnWidths = new int[] { 77, 0, 0 };
    gbl_constantFieldsPanel.rowHeights = new int[] { 0, 0, 0 };
    gbl_constantFieldsPanel.columnWeights = new double[] { 0.0, 1.0, Double.MIN_VALUE };
    gbl_constantFieldsPanel.rowWeights = new double[] { 0.0, 0.0, Double.MIN_VALUE };
    constantFieldsPanel.setLayout(gbl_constantFieldsPanel);

    JLabel roiLabel = new JLabel("Roi node:");
    GridBagConstraints gbc_roiLabel = new GridBagConstraints();
    gbc_roiLabel.anchor = GridBagConstraints.EAST;
    gbc_roiLabel.insets = new Insets(0, 0, 5, 5);
    gbc_roiLabel.gridx = 0;
    gbc_roiLabel.gridy = 0;
    constantFieldsPanel.add(roiLabel, gbc_roiLabel);

    final JComboBox<String> roiComboBox = new JComboBox<>();
    GridBagConstraints gbc_roiComboBox = new GridBagConstraints();
    gbc_roiComboBox.insets = new Insets(0, 0, 5, 0);
    gbc_roiComboBox.fill = GridBagConstraints.HORIZONTAL;
    gbc_roiComboBox.gridx = 1;
    gbc_roiComboBox.gridy = 0;
    constantFieldsPanel.add(roiComboBox, gbc_roiComboBox);
    roiComboBox.addItem("SoRoiManip");
    roiComboBox.addItem("SoRoi");

    JLabel renderLabel = new JLabel("Render node:");
    GridBagConstraints gbc_renderLabel = new GridBagConstraints();
    gbc_renderLabel.anchor = GridBagConstraints.EAST;
    gbc_renderLabel.insets = new Insets(0, 0, 0, 5);
    gbc_renderLabel.gridx = 0;
    gbc_renderLabel.gridy = 1;
    constantFieldsPanel.add(renderLabel, gbc_renderLabel);

    final JComboBox<String> renderComboBox = new JComboBox<>();
    GridBagConstraints gbc_renderComboBox = new GridBagConstraints();
    gbc_renderComboBox.fill = GridBagConstraints.HORIZONTAL;
    gbc_renderComboBox.gridx = 1;
    gbc_renderComboBox.gridy = 1;
    constantFieldsPanel.add(renderComboBox, gbc_renderComboBox);
    renderComboBox.addItem("SoVolumeRender");
    renderComboBox.addItem("SoVolumeSkin");

    final JPanel fieldsPanel = new JPanel();
    fieldsPanel.setMaximumSize(new Dimension(10000, 0));
    settingsPanel.add(fieldsPanel);
    fieldsPanel.setLayout(new CardLayout(0, 0));

    final JPanel roiManipFieldsPanel = new JPanel();
    fieldsPanel.add(roiManipFieldsPanel, "RoiManipFields");
    roiManipFieldsPanel.setMaximumSize(new Dimension(10000, 0));
    roiManipFieldsPanel
        .setBorder(new TitledBorder(null, "SoRoiManip Fields", TitledBorder.LEADING, TitledBorder.TOP, null, null));
    GridBagLayout gbl_roiManipFieldsPanel = new GridBagLayout();
    gbl_roiManipFieldsPanel.columnWidths = new int[] { 77, 0, 0 };
    gbl_roiManipFieldsPanel.rowHeights = new int[] { 0, 0, 0, 0 };
    gbl_roiManipFieldsPanel.columnWeights = new double[] { 0.0, 1.0, Double.MIN_VALUE };
    gbl_roiManipFieldsPanel.rowWeights = new double[] { 0.0, 0.0, 0.0, Double.MIN_VALUE };
    roiManipFieldsPanel.setLayout(gbl_roiManipFieldsPanel);

    JLabel roimfConstrainedLabel = new JLabel("Constrained:");
    GridBagConstraints gbc_roimfConstrainedLabel = new GridBagConstraints();
    gbc_roimfConstrainedLabel.anchor = GridBagConstraints.EAST;
    gbc_roimfConstrainedLabel.insets = new Insets(0, 0, 5, 5);
    gbc_roimfConstrainedLabel.gridx = 0;
    gbc_roimfConstrainedLabel.gridy = 0;
    roiManipFieldsPanel.add(roimfConstrainedLabel, gbc_roimfConstrainedLabel);

    final JCheckBox roimfContrainedCheckBox = new JCheckBox("");
    roimfContrainedCheckBox.setSelected(true);
    GridBagConstraints gbc_roimfContrainedCheckBox = new GridBagConstraints();
    gbc_roimfContrainedCheckBox.anchor = GridBagConstraints.WEST;
    gbc_roimfContrainedCheckBox.insets = new Insets(0, 0, 5, 0);
    gbc_roimfContrainedCheckBox.gridx = 1;
    gbc_roimfContrainedCheckBox.gridy = 0;
    roiManipFieldsPanel.add(roimfContrainedCheckBox, gbc_roimfContrainedCheckBox);

    JLabel roimfColorLabel = new JLabel("Dragger color:");
    GridBagConstraints gbc_roimfColorLabel = new GridBagConstraints();
    gbc_roimfColorLabel.anchor = GridBagConstraints.EAST;
    gbc_roimfColorLabel.insets = new Insets(0, 0, 5, 5);
    gbc_roimfColorLabel.gridx = 0;
    gbc_roimfColorLabel.gridy = 1;
    roiManipFieldsPanel.add(roimfColorLabel, gbc_roimfColorLabel);

    final JComboBox<String> roimfColorComboBox = new JComboBox<>();
    GridBagConstraints gbc_roimfColorComboBox = new GridBagConstraints();
    gbc_roimfColorComboBox.insets = new Insets(0, 0, 5, 0);
    gbc_roimfColorComboBox.fill = GridBagConstraints.HORIZONTAL;
    gbc_roimfColorComboBox.gridx = 1;
    gbc_roimfColorComboBox.gridy = 1;
    roiManipFieldsPanel.add(roimfColorComboBox, gbc_roimfColorComboBox);
    roimfColorComboBox.addItem("Green");
    roimfColorComboBox.addItem("Red");

    JLabel roimfFlagsLabel = new JLabel("Flags:");
    GridBagConstraints gbc_roimfFlagsLabel = new GridBagConstraints();
    gbc_roimfFlagsLabel.anchor = GridBagConstraints.EAST;
    gbc_roimfFlagsLabel.insets = new Insets(0, 0, 0, 5);
    gbc_roimfFlagsLabel.gridx = 0;
    gbc_roimfFlagsLabel.gridy = 2;
    roiManipFieldsPanel.add(roimfFlagsLabel, gbc_roimfFlagsLabel);

    final JComboBox<String> roimfFlagsComboBox = new JComboBox<>();
    GridBagConstraints gbc_roimfFlagsComboBox = new GridBagConstraints();
    gbc_roimfFlagsComboBox.fill = GridBagConstraints.HORIZONTAL;
    gbc_roimfFlagsComboBox.gridx = 1;
    gbc_roimfFlagsComboBox.gridy = 2;
    roiManipFieldsPanel.add(roimfFlagsComboBox, gbc_roimfFlagsComboBox);
    roimfFlagsComboBox.addItem("SUB_VOLUME");
    roimfFlagsComboBox.addItem("EXCLUSION_BOX");
    roimfFlagsComboBox.addItem("CROSS");
    roimfFlagsComboBox.addItem("CROSS_INVERT");
    roimfFlagsComboBox.addItem("FENCE");
    roimfFlagsComboBox.addItem("FENCE_INVERT");

    final JPanel roiFieldsPanel = new JPanel();
    fieldsPanel.add(roiFieldsPanel, "RoiFields");
    roiFieldsPanel.setMaximumSize(new Dimension(10000, 0));
    roiFieldsPanel
        .setBorder(new TitledBorder(null, "SoRoi Fields", TitledBorder.LEADING, TitledBorder.TOP, null, null));
    GridBagLayout gbl_roiFieldsPanel = new GridBagLayout();
    gbl_roiFieldsPanel.columnWidths = new int[] { 77, 0, 0 };
    gbl_roiFieldsPanel.rowHeights = new int[] { 0, 0, 0 };
    gbl_roiFieldsPanel.columnWeights = new double[] { 0.0, 1.0, Double.MIN_VALUE };
    gbl_roiFieldsPanel.rowWeights = new double[] { 0.0, 0.0, Double.MIN_VALUE };
    roiFieldsPanel.setLayout(gbl_roiFieldsPanel);

    JLabel roifRelativeLabel = new JLabel("Relative:");
    GridBagConstraints gbc_roifRelativeLabel = new GridBagConstraints();
    gbc_roifRelativeLabel.anchor = GridBagConstraints.EAST;
    gbc_roifRelativeLabel.insets = new Insets(0, 0, 5, 5);
    gbc_roifRelativeLabel.gridx = 0;
    gbc_roifRelativeLabel.gridy = 0;
    roiFieldsPanel.add(roifRelativeLabel, gbc_roifRelativeLabel);

    final JCheckBox roifRelativeCheckBox = new JCheckBox("");
    GridBagConstraints gbc_roifRelativeCheckBox = new GridBagConstraints();
    gbc_roifRelativeCheckBox.anchor = GridBagConstraints.WEST;
    gbc_roifRelativeCheckBox.insets = new Insets(0, 0, 5, 0);
    gbc_roifRelativeCheckBox.gridx = 1;
    gbc_roifRelativeCheckBox.gridy = 0;
    roiFieldsPanel.add(roifRelativeCheckBox, gbc_roifRelativeCheckBox);

    JLabel roifFlagsLabel = new JLabel("Flags:");
    GridBagConstraints gbc_roifFlagsLabel = new GridBagConstraints();
    gbc_roifFlagsLabel.anchor = GridBagConstraints.EAST;
    gbc_roifFlagsLabel.insets = new Insets(0, 0, 0, 5);
    gbc_roifFlagsLabel.gridx = 0;
    gbc_roifFlagsLabel.gridy = 1;
    roiFieldsPanel.add(roifFlagsLabel, gbc_roifFlagsLabel);

    final JComboBox<String> roifFlagsComboBox = new JComboBox<>();
    GridBagConstraints gbc_roifFlagsComboBox = new GridBagConstraints();
    gbc_roifFlagsComboBox.fill = GridBagConstraints.HORIZONTAL;
    gbc_roifFlagsComboBox.gridx = 1;
    gbc_roifFlagsComboBox.gridy = 1;
    roiFieldsPanel.add(roifFlagsComboBox, gbc_roifFlagsComboBox);
    roifFlagsComboBox.addItem("SUB_VOLUME");
    roifFlagsComboBox.addItem("EXCLUSION_BOX");
    roifFlagsComboBox.addItem("CROSS");
    roifFlagsComboBox.addItem("CROSS_INVERT");
    roifFlagsComboBox.addItem("FENCE");
    roifFlagsComboBox.addItem("FENCE_INVERT");

    Component settingsStrut = Box.createHorizontalStrut(20);
    settingsPanel.add(settingsStrut);

    // Link events

    // Switch between SoROI and SoROIManip
    roiComboBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( e.getStateChange() == ItemEvent.SELECTED )
        {
          SoSearchAction searchAction = new SoSearchAction();
          CardLayout cardLayout = (CardLayout) fieldsPanel.getLayout();

          switch ( roiComboBox.getSelectedIndex() )
          {
          case 0 :
          {
            searchAction.setNode(m_ROI);
            searchAction.apply(m_rootSep);
            SoPath path = searchAction.getPath();
            m_ROIManip.replaceNode(path);
            cardLayout.show(fieldsPanel, "RoiManipFields");
          }
            break;
          case 1 :
          {
            searchAction.setNode(m_ROIManip);
            searchAction.apply(m_rootSep);
            SoPath path = searchAction.getPath();
            m_ROIManip.replaceManip(path, m_ROI);
            cardLayout.show(fieldsPanel, "RoiFields");
          }
            break;
          }
        }
      }
    });

    // Switch between SoVolumeRender and SoVolumeSkin
    renderComboBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( e.getStateChange() == ItemEvent.SELECTED )
          m_renderSwitch.whichChild.setValue(renderComboBox.getSelectedIndex());
      }
    });

    // Constrains the dragger to fit in the data volume or not
    roimfContrainedCheckBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        m_ROIManip.constrained.setValue(roimfContrainedCheckBox.isSelected());
      }
    });

    // Change the color of the SoRoiManip boundaries
    roimfColorComboBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( e.getStateChange() == ItemEvent.SELECTED )
        {
          SoMaterial material = (SoMaterial) (m_ROIManip.getDragger().getPart("tabPlane1.scaleTabMaterial", false));
          switch ( roimfColorComboBox.getSelectedIndex() )
          {
          case 0 :
            material.diffuseColor.setValue(0, 1, 0);
            material.emissiveColor.setValue(0, 1, 0);
            break;
          case 1 :
            material.diffuseColor.setValue(1, 0, 0);
            material.emissiveColor.setValue(1, 0, 0);
            break;
          }
        }
      }
    });

    // Change flag of the ROIManip
    roimfFlagsComboBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( e.getStateChange() == ItemEvent.SELECTED )
        {
          int selectedIndex = roimfFlagsComboBox.getSelectedIndex();
          switch ( selectedIndex )
          {
          case 0 :
            m_ROIManip.flags.setValue(SoROI.FlagsType.SUB_VOLUME);
            break;
          case 1 :
            m_ROIManip.flags.setValue(SoROI.FlagsType.EXCLUSION_BOX);
            break;
          case 2 :
            m_ROIManip.flags.setValue(SoROI.FlagsType.CROSS);
            break;
          case 3 :
            m_ROIManip.flags.setValue(SoROI.FlagsType.CROSS_INVERT);
            break;
          case 4 :
            m_ROIManip.flags.setValue(SoROI.FlagsType.FENCE);
            break;
          case 5 :
            m_ROIManip.flags.setValue(SoROI.FlagsType.FENCE_INVERT);
            break;
          }
          roifFlagsComboBox.setSelectedIndex(selectedIndex);
        }
      }
    });

    roifRelativeCheckBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        m_ROI.relative.setValue(roifRelativeCheckBox.isSelected());
      }
    });

    // Change flag of the ROIManip
    roifFlagsComboBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( e.getStateChange() == ItemEvent.SELECTED )
        {
          int selectedIndex = roifFlagsComboBox.getSelectedIndex();
          switch ( selectedIndex )
          {
          case 0 :
            m_ROI.flags.setValue(SoROI.FlagsType.SUB_VOLUME);
            break;
          case 1 :
            m_ROI.flags.setValue(SoROI.FlagsType.EXCLUSION_BOX);
            break;
          case 2 :
            m_ROI.flags.setValue(SoROI.FlagsType.CROSS);
            break;
          case 3 :
            m_ROI.flags.setValue(SoROI.FlagsType.CROSS_INVERT);
            break;
          case 4 :
            m_ROI.flags.setValue(SoROI.FlagsType.FENCE);
            break;
          case 5 :
            m_ROI.flags.setValue(SoROI.FlagsType.FENCE_INVERT);
            break;
          }
          roimfFlagsComboBox.setSelectedIndex(selectedIndex);
        }
      }
    });
  }

  /**
   * @param dataPath
   * @return Scene graph
   */
  public SoSeparator buildSceneGraph(String dataPath)
  {
    SoSeparator rootSep = new SoSeparator();
    rootSep.setName("root");

    // Create scene elements

    SoSeparator sceneSep = new SoSeparator();
    sceneSep.setName("scene");
    rootSep.addChild(sceneSep);

    // Add the background logo node
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }
    sceneSep.addChild(logoBackground);

    // Node holding the volume data
    SoVolumeData volData = new SoVolumeData();
    volData.fileName.setValue(dataPath);
    sceneSep.addChild(volData);

    // If necessary, specify the actual range of the data values.
    // By default VolumeViz maps the entire range of the voxel data type
    // (e.g. 0..65535 for unsigned short) into the colormap. This works
    // great for byte (8 bit) voxels, but not so well for 16 bit voxels
    // and not at all for floating point voxels. So it's not actually
    // necessary for this data set, but shown here for completeness.
    // NOTE: Min/max values are stored in the header for LDM format
    // files, but for other formats the getMinMax query can take a
    // long time because VolumeViz has to examine every voxel.
    SoDataRange volRange = new SoDataRange();
    if ( volData.getDatumSize() > 1 )
    {
      double[] minMax = volData.getDoubleMinMax();
      volRange.min.setValue(minMax[0]);
      volRange.max.setValue(minMax[1]);
    }
    sceneSep.addChild(volRange);

    // Improve data rendering by adding the following node
    SoVolumeRenderingQuality volQual = new SoVolumeRenderingQuality();
    volQual.ambientOcclusion.setValue(true);
    volQual.deferredLighting.setValue(true);
    sceneSep.addChild(volQual);

    // Use a predefined colorMap (associate data values to colors)
    SoTransferFunction volTF = new SoTransferFunction();
    volTF.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.STANDARD);
    volTF.minValue.setValue(69); // Make low intensity voxels transparent
    sceneSep.addChild(volTF);

    // Initialize ROI box
    m_ROI = new SoROI();
    m_ROIManip = new SoROIManip();
    SbVec3i32 size = volData.data.getSize();
    SbVec3i32 volMin = new SbVec3i32(0, 0, 0);
    SbVec3i32 volMax = size.minus(new SbVec3i32(1, 1, 1));
    SbVec3i32 subvolMin = volMax.minus(volMin).over(3);
    SbVec3i32 subvolMax = subvolMin.times(2);
    volMax.setValueAt(2, (volMax.getValueAt(2) - volMin.getValueAt(2)) / 2);
    m_ROIManip.subVolume.setValue(volMin, volMax);
    m_ROIManip.box.setValue(subvolMin, subvolMax);
    // Constrain the ROIManip to stay inside the volume
    m_ROIManip.constrained.setValue(true);
    sceneSep.addChild(m_ROIManip);

    // Switch to enable either SoVolumeRender or SoVolumeSkin
    // Indicates which child of switch to traverse
    m_renderSwitch = new SoSwitch();
    m_renderSwitch.whichChild.setValue(0);
    sceneSep.addChild(m_renderSwitch);

    // Child 0: SoVolumeRender
    SoVolumeRender volRend = new SoVolumeRender();
    volRend.samplingAlignment.setValue(SoVolumeRender.SamplingAlignments.BOUNDARY_ALIGNED);
    volRend.fixedNumSlicesInRoi.setValue(true);
    m_renderSwitch.addChild(volRend);

    // Child 1: SoVolumeSkin
    SoVolumeSkin volSkin = new SoVolumeSkin();
    m_renderSwitch.addChild(volSkin);

    return rootSep;
  }

  public void destroy()
  {
    // Destroy NEWT resources
    m_renderArea.dispose();
  }
}
