package meshviz.mesh.sample;

import java.net.MalformedURLException;
import java.net.URL;

import com.openinventor.inventor.*;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.draggers.SoDragger;
import com.openinventor.inventor.draggers.SoJackDragger;
import com.openinventor.inventor.misc.callbacks.SoDraggerCB;
import com.openinventor.inventor.nodes.*;
import com.openinventor.meshviz.data.*;
import com.openinventor.meshviz.graph.PoBase;
import com.openinventor.meshviz.misc.callbacks.PoRebuildCB;
import com.openinventor.meshviz.nodes.PoDomain;
import com.openinventor.meshviz.nodes.PoMeshProperty;

import meshviz.mesh.sample.dialog.PoAwtChoiceButtonData;
import meshviz.mesh.sample.dialog.PoAwtDialog;
import meshviz.mesh.sample.dialog.PoAwtElementData;
import meshviz.mesh.sample.dialog.PoAwtToggleButtonData;

// Not implemented: draggers, legends, sliders.
// Added: Toggle button to change transparency (instead of a slider).

/**
  * @author Jean Michel Godinaud
  * @author Rodolphe Albout
  * @author Loic Vigneras
  * @author Patrick Vigneras
  */
public class Mesh3DViewer extends MeshViewer {

  // Constants
  // ---------

  // ---------------------- !!! IMPORTANT - IMPORTANT !!! -------------------------
  // These constants must be consecutive, starting from 0 to the last one NUM_EVENT.
  // They correspond to element numbers, and so must also be in the same order as
  // corresponding elements are in the dialog box.
  // ---------------------- !!! IMPORTANT - IMPORTANT !!! -------------------------
  private static final int CHOICE_DATA_SET = 0;
  private static final int CHOICE_COLORING = 1;
  private static final int TOGGLE_SKIN = 2;
  private static final int TOGGLE_SKELETON = 3;
  private static final int TOGGLE_LEVEL_SURF = 4;
  private static final int TOGGLE_CROSS_SECTION = 5;
  private static final int TOGGLE_CROSS_CONTOUR = 6;
  private static final int TOGGLE_VEC_FIELD = 7;
  //  private static final int TOGGLE_DRAGGER = 8;
  //  private static final int TOGGLE_DATA_LEGEND = 9;
  private static final int CHOICE_STREAM_REP = 8;
  private static final int CHOICE_VEC_TYPE = 9;
  private static final int CHOICE_PLANE_ORIENT = 10;
  private static final int CHOICE_TRANSPARENCY = 11;
  private static final int TOGGLE_TRANSPARENCY = 12;
  //  private static final int SLIDER_TRANSPARENCY = 13;
  //  private static final int SLIDER_LEVEL_SURF = 14;
  private static final int TOGGLE_CLIP = 13;
  //  private static final int TRIGGER_IV_FILE = 16;
  private static final int NUM_EVENT = 14;

  private static final int NUM_COLOR = 7 ;
  private static final int WIDTH_FACTOR = 200 ;
  private static final int DEF_MAX_STEP_NUM = 10000 ;
  private static final float RADIUS_SRC_CIRCLE = 0.25F ;

  private PoMeshSkin v_MeshSkin ;
  private PoMeshCrossSection v_MeshCrossSection ;
  private PoMeshSkeleton v_MeshSkeleton ;
  private PoMeshLevelSurf v_MeshLevelSurf ;
  private PoMeshCrossContour v_MeshCrossContour ;
  private PoMesh3DVecGridCrossSection v_MeshVecField ;
  private PoStreamSurface v_MeshStreamSurfaces ;
  private PoStreamLine v_MeshStreamLines ;
  private PoStreamSphereMotion v_MeshStreamSphere ;
  private PoStreamTadpoleMotion v_MeshStreamTadpole ;
  private PoStreamLineMotion v_MeshStreamLineParticle ;
  private PoStreamPointMotion v_MeshStreamPoint ;
  private SoSeparator v_StreamSource ;

  private SoClipPlane v_ClipPlane ;
  private SoJackDragger v_Dragger ;

  private SoSwitch v_MeshSkinSwitch ;
  private SoSwitch v_MeshCrossSectionSwitch ;
  private SoSwitch v_MeshSkeletonSwitch ;
  private SoSwitch v_DraggerSwitch ;
  private SoSwitch v_MeshLevelSurfSwitch ;
  private SoSwitch v_MeshCrossContourSwitch ;
  private SoSwitch v_MeshVecFieldSwitch ;
  private SoSwitch v_MeshStreamSwitch ;
  private SoSwitch v_MeshStreamRepSwitch ;
  private SoSwitch v_StreamSourceSwitch ;

  private SbVec3f v_PlaneNormal ;
  private SbVec3f v_DraggerInitPos;
  private SbBox3f v_BoundingBox;
  private boolean v_DraggerAtCenter;

  private float v_TadpoleLengthFactor;

  private float v_TransparencyValue;

  private int v_NumStartPoints;

  private SoSwitch v_DataSwitch;
  // Constructor
  // -----------

  //--------------------------------------------------------------------------------------------
  public Mesh3DViewer () {
    v_TransparencyValue = 0.0f;
    v_PlaneNormal = new SbVec3f (1,0,0);
    v_DraggerInitPos = new SbVec3f(0,0,0) ;
    v_NumStartPoints = 7 ;
    v_TadpoleLengthFactor = 1 ;
    v_DraggerAtCenter = true ;
    v_WindowTitle = "3D_mesh_viewer";
  }

  // Methods
  // -------

  //--------------------------------------------------------------------------------------------
  public void setPlaneNormal (SbVec3f normal) {
    v_PlaneNormal = normal;
  }

  //--------------------------------------------------------------------------------------------
  public void setDraggerInitPos(SbVec3f pos) {
    v_DraggerInitPos = pos;
    v_DraggerAtCenter = false ;
  }

  public class TransparencyHandler extends PoRebuildCB {
    public void invoke(PoBase base) {
      if (v_TransparencyValue != 0.0f) {
	SoMaterial material = (SoMaterial)v_MeshSkin.getPart("appearance.material");
	float tranparencies[] = new float[material.diffuseColor.getNum()];
	for (int i=0; i<material.diffuseColor.getNum(); i++) tranparencies[i] = v_TransparencyValue;
	material.transparency.setValues(0,tranparencies);
      }
    }
  }


  //--------------------------------------------------------------------------------------------
  public void show (PoMeshProperty po_mesh) { show(po_mesh, null); }

  //--------------------------------------------------------------------------------------------
  public void show (PoMeshProperty po_mesh, PoDomain domain) {

    PbMesh mesh = po_mesh.getMesh();

    // Initialize the data mapping associated to each set of value
    // Initialize the list of isovalues
    initShow (po_mesh);

    v_Domain = domain;

    // get the mesh bounding box
    v_BoundingBox = mesh.getBoundingBox();
    SbVec3f bounding_box_center = v_BoundingBox.getCenter();

    float[] bbSize = v_BoundingBox.getSize().array ;
    float maxs = (bbSize[0] > bbSize[1]) ? bbSize[0] : bbSize[1] ;
    if (bbSize[2] > maxs) maxs = bbSize[2] ;

    float max_vec = mesh.getMaxVecsSet(0) ;
    float time_step = 0.05F * maxs / max_vec;

    v_TadpoleLengthFactor = 0.1F*maxs/max_vec;

    SbColor[] particle_colors = new SbColor[NUM_COLOR] ;
    particle_colors[0] = new SbColor(1,1,1);
    particle_colors[1] = new SbColor(1,0,0);
    particle_colors[2] = new SbColor(0,1,0);
    particle_colors[3] = new SbColor(0,0,1);
    particle_colors[4] = new SbColor(1,1,0);
    particle_colors[5] = new SbColor(0,1,1);
    particle_colors[6] = new SbColor(1,0,1);

    // define StreamLine
    v_MeshStreamSurfaces = new PoStreamSurface() ;
    v_MeshStreamSurfaces.startPoints.set1Value(0,bounding_box_center);
    v_MeshStreamSurfaces.rakeOrientation.set1Value(0,new SbVec3f(1,0,0));
    v_MeshStreamSurfaces.rakeLength.setValue(2*maxs/WIDTH_FACTOR);
    v_MeshStreamSurfaces.numLinesPerRake.setValue(5);
    v_MeshStreamSurfaces.vecsIndex.setValue(0);
    v_MeshStreamSurfaces.coloringType.setValue(PoMesh.ColoringTypes.COLOR_MAPPING);
    v_MeshStreamSurfaces.integrationMaxStepNumber.setValue(DEF_MAX_STEP_NUM);
    v_MeshStreamSurfaces.colors.setValues(0,particle_colors);

    v_MeshStreamLines = new PoStreamLine() ;
    v_MeshStreamLines.vecsIndex.setValue(0);
    v_MeshStreamLines.coloringType.setValue(PoMesh.ColoringTypes.COLOR_MAPPING) ;
    v_MeshStreamLines.integrationMaxStepNumber.setValue(DEF_MAX_STEP_NUM) ;
    v_MeshStreamLines.colors.setValues(0,particle_colors);

    v_MeshStreamPoint = new PoStreamPointMotion() ;
    v_MeshStreamPoint.vecsIndex.setValue(0);
    v_MeshStreamPoint.coloringType.setValue(PoMesh.ColoringTypes.COLOR_MAPPING) ;
    v_MeshStreamPoint.pulseFrequency.setValue(15) ;
    v_MeshStreamPoint.integrationMaxStepNumber.setValue(DEF_MAX_STEP_NUM) ;
    v_MeshStreamPoint.colors.setValues(0,particle_colors);
    v_MeshStreamPoint.startPoints.connectFrom(v_MeshStreamLines.startPoints);
    v_MeshStreamPoint.timeStep.setValue(time_step) ;

    v_MeshStreamSphere = new PoStreamSphereMotion() ;
    v_MeshStreamSphere.sphereRadius.setValue(maxs/150F) ;
    v_MeshStreamSphere.vecsIndex.setValue(0);
    v_MeshStreamSphere.coloringType.setValue(PoMesh.ColoringTypes.COLOR_MAPPING) ;
    v_MeshStreamSphere.pulseFrequency.setValue(15) ;
    v_MeshStreamSphere.integrationMaxStepNumber.setValue(DEF_MAX_STEP_NUM) ;
    v_MeshStreamSphere.colors.setValues(0,particle_colors);
    v_MeshStreamSphere.startPoints.connectFrom(v_MeshStreamLines.startPoints);
    v_MeshStreamSphere.timeStep.setValue(time_step) ;

    v_MeshStreamLineParticle = new PoStreamLineMotion();
    v_MeshStreamLineParticle.vecsIndex.setValue(0);
    v_MeshStreamLineParticle.coloringType.setValue(PoMesh.ColoringTypes.COLOR_MAPPING) ;
    v_MeshStreamLineParticle.pulseFrequency.setValue(15) ;
    v_MeshStreamLineParticle.integrationMaxStepNumber.setValue(DEF_MAX_STEP_NUM) ;
    v_MeshStreamLineParticle.startPoints.connectFrom(v_MeshStreamLines.startPoints);
    v_MeshStreamLineParticle.timeStep.setValue(time_step) ;

    v_MeshStreamTadpole = new PoStreamTadpoleMotion();
    v_MeshStreamTadpole.vecsIndex.setValue(0);
    v_MeshStreamTadpole.coloringType.setValue(PoMesh.ColoringTypes.COLOR_MAPPING) ;
    v_MeshStreamTadpole.pulseFrequency.setValue(15) ;
    v_MeshStreamTadpole.integrationMaxStepNumber.setValue(DEF_MAX_STEP_NUM) ;
    v_MeshStreamTadpole.startPoints.connectFrom(v_MeshStreamLines.startPoints);
    v_MeshStreamTadpole.lengthFactor.setValue(v_TadpoleLengthFactor) ;
    v_MeshStreamTadpole.timeStep.setValue(time_step) ;

    // define the vector field
    v_MeshVecField = new PoMesh3DVecGridCrossSection();
    v_MeshVecField.vecsIndex.setValue (0);
    v_MeshVecField.density.setValue (v_VecFieldDensity);
    v_MeshVecField.bodyLengthFactor.setValue (v_VecLengthFactor);
    v_MeshVecField.bodyLengthType.setValue (PoMesh3DVec.BodyLengthTypes.CONSTANT_LENGTH);
    v_MeshVecField.endArrowHeightFactor.setValue(0.25f);
    v_MeshVecField.bodyColoringType.setValue (PoMesh3DVec.ArrowColoringTypes.MODULE_MAPPING_COLOR);

    // define the mesh skin
    v_MeshSkin = new PoMeshSkin ();
    v_MeshSkin.valuesIndex.setValue (v_DataSetIndex);
    v_MeshSkin.coloringType.setValue (PoMesh.ColoringTypes.COLOR_MAPPING);
    v_MeshSkin.addPostRebuildCallback(new TransparencyHandler(),null);

    // define the mesh skeleton
    v_MeshSkeleton = new PoMeshSkeleton ();
    v_MeshSkeleton.numXContour.setValue(2);
    v_MeshSkeleton.numYContour.setValue(2);
    v_MeshSkeleton.numZContour.setValue(2);
    v_MeshSkeleton.set("appearance.material","diffuseColor 0.5 0.25 0.25");

    // define the mesh isosurface
    v_MeshLevelSurf = new PoMeshLevelSurf ();
    v_MeshLevelSurf.valuesIndex.setValue (v_DataSetIndex);
    v_MeshLevelSurf.valuesIndexForLevel.setValue (v_DataSetIndex);
    v_MeshLevelSurf.coloringType.setValue (PoMesh.ColoringTypes.COLOR_MAPPING);
    v_MeshLevelSurf.levelValue.setValue ((v_Vmin[v_DataSetIndex] +
					  v_Vmax[v_DataSetIndex]) / 2);

    // define a mesh cross section
    v_MeshCrossSection = new PoMeshCrossSection ();
    v_MeshCrossSection.valuesIndex.setValue (v_DataSetIndex);
    v_MeshCrossSection.coloringType.setValue (PoMesh.ColoringTypes.COLOR_MAPPING);
    v_MeshCrossSection.plane.setValue (new SbPlane (v_PlaneNormal,
						    bounding_box_center));

    // define a mesh cross contour
    v_MeshCrossContour = new PoMeshCrossContour ();
    v_MeshCrossContour.valuesIndex.setValue (v_DataSetIndex);
    v_MeshCrossContour.coloringType.setValue (PoMesh.ColoringTypes.COLOR_MAPPING);
    v_MeshCrossContour.plane.setValue (new SbPlane (v_PlaneNormal,
						    bounding_box_center));

    SoMaterial material = new SoMaterial();
    material.diffuseColor.set1Value(0,1,0,0);
    SoDrawStyle draw_style = new SoDrawStyle();
    draw_style.pointSize.setValue(4);
    SoCoordinate3 source_coord = new SoCoordinate3();
    source_coord.point.connectFrom(v_MeshStreamLines.startPoints);
    SoPointSet source_points = new SoPointSet();
    source_points.numPoints.setValue(-1); // use all points in source_coord
    v_StreamSource = new SoSeparator();
    v_StreamSource.addChild(material);
    v_StreamSource.addChild(draw_style);
    v_StreamSource.addChild(source_coord);
    v_StreamSource.addChild(source_points);

    v_ClipPlane = new SoClipPlane ();
    v_ClipPlane.on.setValue (false);

    float[] d = v_BoundingBox.getSize().array;
    v_ScaleFactor = Math.max (Math.max (d[0], d[1]), d[2]);

    v_ScaleFactor /= 10;
    v_VecFieldWidth = v_ScaleFactor / 4;

    v_Dragger = new SoJackDragger ();
    v_Dragger.addMotionCallback (new MotionCallback (), this);

    // dialog
    v_Dialog = new PoAwtDialog ("3D Mesh viewer");
    buildDialogBox (v_Dialog);
    v_Dialog.addListener(new DialogListener());

    if (v_DraggerAtCenter)
      setCrossSection(v_BoundingBox.getCenter()) ;
    else
      setCrossSection(v_DraggerInitPos) ;

    v_MeshSkinSwitch = new SoSwitch ();
    v_MeshSkinSwitch.addChild (v_MeshSkin) ;
    v_MeshSkinSwitch.whichChild.setValue (SoSwitch.SO_SWITCH_ALL);

    v_MeshSkeletonSwitch = new SoSwitch ();
    v_MeshSkeletonSwitch.addChild (v_MeshSkeleton) ;
    v_MeshSkeletonSwitch.whichChild.setValue (SoSwitch.SO_SWITCH_NONE);

    v_MeshLevelSurfSwitch = new SoSwitch ();
    v_MeshLevelSurfSwitch.addChild (v_MeshLevelSurf);
    v_MeshLevelSurfSwitch.whichChild.setValue (SoSwitch.SO_SWITCH_ALL);

    v_MeshCrossSectionSwitch = new SoSwitch ();
    v_MeshCrossSectionSwitch.addChild (v_MeshCrossSection);
    v_MeshCrossSectionSwitch.whichChild.setValue (SoSwitch.SO_SWITCH_ALL);

    v_MeshCrossContourSwitch = new SoSwitch ();
    v_MeshCrossContourSwitch.addChild (v_MeshCrossContour);
    v_MeshCrossContourSwitch.whichChild.setValue (SoSwitch.SO_SWITCH_NONE);

    v_MeshVecFieldSwitch = new SoSwitch ();
    v_MeshVecFieldSwitch.addChild (v_MeshVecField) ;
    v_MeshVecFieldSwitch.whichChild.setValue (SoSwitch.SO_SWITCH_NONE);

    v_DraggerSwitch = new SoSwitch ();
    v_DraggerSwitch.addChild (v_Dragger) ;
    v_DraggerSwitch.whichChild.setValue (SoSwitch.SO_SWITCH_ALL);

    v_MeshStreamRepSwitch = new SoSwitch();
    v_MeshStreamRepSwitch.addChild(v_MeshStreamLines);
    v_MeshStreamRepSwitch.addChild(v_MeshStreamLineParticle);
    v_MeshStreamRepSwitch.addChild(v_MeshStreamTadpole);
    v_MeshStreamRepSwitch.addChild(v_MeshStreamPoint);
    v_MeshStreamRepSwitch.addChild(v_MeshStreamSphere);
    v_MeshStreamRepSwitch.addChild(v_MeshStreamSurfaces);
    v_MeshStreamRepSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);

    v_StreamSourceSwitch = new SoSwitch();
    v_StreamSourceSwitch.addChild(v_StreamSource);
    v_StreamSourceSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);

    v_MeshStreamSwitch = new SoSwitch();
    v_MeshStreamSwitch.addChild(v_StreamSourceSwitch);
    v_MeshStreamSwitch.addChild(v_MeshStreamRepSwitch);
    v_MeshStreamSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);

    // compute a scaling matrix relative to the bounding box
    SoSeparator draggerSeparator = new SoSeparator();
    if (v_Domain != null)
      draggerSeparator.addChild(v_Domain.getTransform());
    draggerSeparator.addChild(v_DraggerSwitch);

    SoPickStyle pickStyle = new SoPickStyle ();
    pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);

    v_DataSwitch = new SoSwitch();
    v_DataSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
    v_DataSwitch.addChild(v_IsoValues[v_DataSetIndex]);
    v_DataSwitch.addChild(v_DataMapping[v_DataSetIndex]);

    v_Scene3D = new SoSeparator();
    v_Scene3D.setName(Scene3DName);
    v_Scene3D.addChild(po_mesh);
    v_Scene3D.addChild(v_DataSwitch);
    if (v_Domain != null) v_Scene3D.addChild (v_Domain);
    v_Scene3D.addChild(draggerSeparator);

    v_Scene3D.addChild(pickStyle);
    v_Scene3D.addChild(v_MeshCrossSectionSwitch);
    v_Scene3D.addChild(v_MeshCrossContourSwitch);

    v_Scene3D.addChild(v_ClipPlane);
    v_Scene3D.addChild(v_MeshSkinSwitch);
    v_Scene3D.addChild(v_MeshSkeletonSwitch);
    v_Scene3D.addChild(v_MeshLevelSurfSwitch);

    v_Scene3D.addChild(v_ModuleDataMapping);
    v_Scene3D.addChild(v_MeshStreamSwitch) ;
    v_Scene3D.addChild(v_MeshVecFieldSwitch);

    startShow();
  }

  //--------------------------------------------------------------------------------------------
  private void buildDialogBox (PoAwtDialog dialog) {

    PoAwtChoiceButtonData choiceCind =
      new PoAwtChoiceButtonData ("Data used for coloring :");
    choiceCind.setItems (v_DataSetNames);
    choiceCind.setSelectedItem(v_DataSetIndex + 1);

    PoAwtChoiceButtonData choiceStreamRep =
      new PoAwtChoiceButtonData ("Stream representation :");
    choiceStreamRep.setItems (stream_rep);
    choiceStreamRep.setSelectedItem(3);

    PoAwtChoiceButtonData choiceCType =
      new PoAwtChoiceButtonData ("Coloring type:");
    choiceCType.setItems (colrtList);
    choiceCType.setSelectedItem (2);

    PoAwtChoiceButtonData choiceTransp =
      new PoAwtChoiceButtonData ("Transparency type:");
    choiceTransp.setItems (transType);
    choiceTransp.setSelectedItem (0);

    PoAwtChoiceButtonData choiceVecType =
      new PoAwtChoiceButtonData ("Vector rep type:");
    choiceVecType.setItems (vecType);
    choiceVecType.setSelectedItem(0);

    PoAwtChoiceButtonData choicePlaneOrient =
      new PoAwtChoiceButtonData ("Plane orientation:");
    choicePlaneOrient.setItems (planeOrient);
    if (v_PlaneNormal.equals(new SbVec3f (1,0,0)))
      choicePlaneOrient.setSelectedItem (0);
    else if (v_PlaneNormal.equals(new SbVec3f (0,1,0)))
      choicePlaneOrient.setSelectedItem (1);
    else
      choicePlaneOrient.setSelectedItem (2);

    float dmax;
    float dx,dy,dz;
    float[] d;
    d = v_BoundingBox.getSize().array;
    dx = d[0]; dy = d[1]; dz = d[2];
    if (v_PlaneNormal.equals(new SbVec3f (1,0,0))) dmax = dx / 2;
    else if (v_PlaneNormal.equals(new SbVec3f (0,1,0))) dmax = dy / 2;
    else if (v_PlaneNormal.equals(new SbVec3f (0,0,1))) dmax = dz / 2;
    else {
      dmax = 0.75f * Math.max (Math.max (dx, dy), dz);
  }

    PoAwtToggleButtonData switchSkin =
      new PoAwtToggleButtonData ("mesh Skin", PoAwtToggleButtonData.ON,"","");
    PoAwtToggleButtonData switchSkeleton =
      new PoAwtToggleButtonData ("mesh Skeleton",PoAwtToggleButtonData.OFF,"","");
    PoAwtToggleButtonData switchLevelSurf =
      new PoAwtToggleButtonData ("mesh LevelSurf",PoAwtToggleButtonData.ON,"","");
    PoAwtToggleButtonData switchCrossSection =
      new PoAwtToggleButtonData ("mesh CrossSection",PoAwtToggleButtonData.ON,"","");
    PoAwtToggleButtonData switchCrossContour =
      new PoAwtToggleButtonData ("mesh CrossContour",PoAwtToggleButtonData.OFF,"","");
    PoAwtToggleButtonData switchVecField =
      new PoAwtToggleButtonData ("vec field",PoAwtToggleButtonData.OFF,"","");
    //    PoAwtToggleButtonData switchDataLegend =
    //      new PoAwtToggleButtonData ("data legend",PoAwtToggleButtonData.ON,"","");
    //    PoAwtRealSliderData sliderTransparency =
    //      new PoAwtRealSliderData("Skin transparency",0,1,0);
    //    PoAwtRealSliderData sliderLevelVal =
    //      new PoAwtRealSliderData("Level Surface value",v_Vmin[v_DataSetIndex],v_Vmax[v_DataSetIndex],(v_Vmin[v_DataSetIndex]+v_Vmax[v_DataSetIndex])/2);
    //    PoAwtToggleButtonData toggleDragger =
    //      new PoAwtToggleButtonData ("plane dragger",PoAwtToggleButtonData.ON,"","");
    PoAwtToggleButtonData toggleClip =
      new PoAwtToggleButtonData ("activate clipping",PoAwtToggleButtonData.OFF,"","");
    // ADDED: Toggle button for transparency
    PoAwtToggleButtonData toggleTransparency =
      new PoAwtToggleButtonData ("Skin transparency",PoAwtToggleButtonData.OFF,"","");
    // END OF ADDED PART

    //    PoAwtTriggerButtonData *buttonIvFile = new PoAwtTriggerButtonData("write",".iv file");

    PoAwtElementData[] elements = new PoAwtElementData[NUM_EVENT];
    elements[CHOICE_DATA_SET] = choiceCind;
    elements[CHOICE_COLORING] = choiceCType;
    elements[CHOICE_PLANE_ORIENT] = choicePlaneOrient;
    elements[TOGGLE_SKIN] = switchSkin;
    elements[TOGGLE_SKELETON] = switchSkeleton;
    elements[TOGGLE_LEVEL_SURF] = switchLevelSurf;
    elements[TOGGLE_CROSS_SECTION] = switchCrossSection;
    elements[TOGGLE_CROSS_CONTOUR] = switchCrossContour;
    elements[TOGGLE_VEC_FIELD] = switchVecField;
    //    elements[TOGGLE_DATA_LEGEND] = switchDataLegend;
    elements[CHOICE_STREAM_REP] = choiceStreamRep;
    elements[CHOICE_VEC_TYPE] = choiceVecType;
    elements[CHOICE_TRANSPARENCY] = choiceTransp;
    elements[TOGGLE_TRANSPARENCY] = toggleTransparency;
    //    elements[SLIDER_TRANSPARENCY] = sliderTransparency;
    //    elements[SLIDER_LEVEL_SURF] = sliderLevelVal;
    //    elements[TOGGLE_DRAGGER] = toggleDragger;
    elements[TOGGLE_CLIP] = toggleClip;
    //    elements[TRIGGER_IV_FILE] = buttonIvFile;
    dialog.setValues (elements);
  }

  //--------------------------------------------------------------------------------------------
  // Default scope to be accessible from DialogCallback
  void manageDialog (int elementIndex) {

    PoAwtElementData elt = v_Dialog.get1Value (elementIndex);

    switch (elt.getType ()) {
    case PoAwtElementData.CHOICE_BUTTON: {
      PoAwtChoiceButtonData choice = (PoAwtChoiceButtonData) elt;
      int choiceVal = choice.getSelectedItem ();

      switch (elementIndex) {
      case CHOICE_DATA_SET :
	// select another data set
	if (choiceVal != 0) {
	  v_DataSetIndex = choiceVal-1;
	  v_DataSwitch.replaceChild(0,v_IsoValues[v_DataSetIndex]);
	  v_DataSwitch.replaceChild(1,v_DataMapping[v_DataSetIndex]);

	  v_MeshSkin.valuesIndex.setValue (v_DataSetIndex);
	  v_MeshSkeleton.valuesIndex.setValue (v_DataSetIndex);
	  v_MeshLevelSurf.valuesIndex.setValue (v_DataSetIndex);
	  v_MeshLevelSurf.valuesIndexForLevel.setValue (v_DataSetIndex);
	  v_MeshCrossSection.valuesIndex.setValue (v_DataSetIndex);
	  v_MeshCrossContour.valuesIndex.setValue (v_DataSetIndex);

	  v_MeshLevelSurf.levelValue.setValue ( (v_Vmin[v_DataSetIndex] +
						 v_Vmax[v_DataSetIndex]) / 2);
	} else {
	  v_MeshSkin.valuesIndex.setValue(-1);
	  v_MeshSkeleton.valuesIndex.setValue(-1);
	  v_MeshLevelSurf.valuesIndex.setValue(-1);
	  v_MeshCrossSection.valuesIndex.setValue(-1);
	  v_MeshCrossContour.valuesIndex.setValue(-1);
	}
	/*
	if (v_TransparencyValue > 0) {
	  SoMaterial material = (SoMaterial) v_MeshSkin.getPart("appearance.material");
	  for (int i = 0; i < material.diffuseColor.getNum (); i++)
	    material.transparency.set1Value (i, v_TransparencyValue);
	}
	*/
	break;

      case CHOICE_STREAM_REP :
	if (choiceVal > 0) {
	  v_MeshStreamRepSwitch.whichChild.setValue(choiceVal - 1) ;
	  v_MeshStreamSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL) ;
	}
	else
	  v_MeshStreamSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE) ;
	break ;

      case CHOICE_COLORING :
	// change coloring type
	v_MeshCrossSection.coloringType.setValue(choiceVal) ;
	v_MeshSkin.coloringType.setValue(choiceVal) ;
	v_MeshLevelSurf.coloringType.setValue(choiceVal);
	v_MeshCrossContour.coloringType.setValue(choiceVal);
	v_MeshStreamSurfaces.coloringType.setValue(choiceVal) ;
	v_MeshStreamLines.coloringType.setValue(choiceVal) ;
	v_MeshStreamSphere.coloringType.setValue(choiceVal) ;
	v_MeshStreamPoint.coloringType.setValue(choiceVal) ;
	v_MeshStreamLineParticle.coloringType.setValue(choiceVal) ;
	v_MeshStreamTadpole.coloringType.setValue(choiceVal) ;
	/*
	if (v_TransparencyValue > 0) {
	  v_MeshSkin.rebuild();
	  SoMaterial material = (SoMaterial)v_MeshSkin.getPart("appearance.material", true);
	  for (int i=0 ; i < material.diffuseColor.getNum( ); i++)
	    material.transparency.set1Value(i,v_TransparencyValue);
	}
	*/
	break;

      case CHOICE_PLANE_ORIENT :
	switch (choiceVal) {
	case 0: v_PlaneNormal.setValue (1,0,0); setCrossSection(v_BoundingBox.getCenter()); break;
	case 1: v_PlaneNormal.setValue (0,1,0); setCrossSection(v_BoundingBox.getCenter()); break;
	case 2: v_PlaneNormal.setValue (0,0,1); setCrossSection(v_BoundingBox.getCenter()); break;
	}
	break;

      case CHOICE_VEC_TYPE :
	switch (choiceVal) {
	case 0:
	  v_MeshVecField.density.setValue (v_VecFieldDensity);
	  v_MeshVecField.bodyLengthFactor.setValue (v_VecLengthFactor);
	  v_MeshVecField.bodyShape.setValue (PoMesh3DVec.BodyShapes.LINE);
	  v_MeshVecField.endArrowShape.setValue (PoMesh3DVec.ArrowShapes.CHEVRON);
	  v_MeshVecField.endArrowHeightFactor.setValue (0.1f);
	  v_MeshVecField.bodyRadiusType.setValue (PoMesh3DVec.BodyRadiusTypes.CONSTANT_RADIUS);
	  break;
	case 1:
	  //	  v_MeshVecField.density.setValue (v_VecFieldDensity * 4);
	  //	  v_MeshVecField.bodyLengthFactor.setValue (v_VecLengthFactor * 2);
	  v_MeshVecField.bodyShape.setValue (PoMesh3DVec.BodyShapes.CYLINDER);
	  v_MeshVecField.endArrowShape.setValue (PoMesh3DVec.ArrowShapes.CONE);
	  v_MeshVecField.bodyRadiusType.setValue (PoMesh3DVec.BodyRadiusTypes.RELATIVE_RADIUS);
	  break;
	}
	break;

      case CHOICE_TRANSPARENCY :
	switch (choiceVal) {
	case 0:
	  v_Viewer.getRenderArea().setTransparencyType (SoGLRenderAction.TransparencyTypes.NO_SORT);
	  break;
	case 1:
	  v_Viewer.getRenderArea().setTransparencyType (SoGLRenderAction.TransparencyTypes.OPAQUE_FIRST);
	  break;
	case 2:
	  v_Viewer.getRenderArea().setTransparencyType (SoGLRenderAction.TransparencyTypes.SORTED_OBJECT);
	  break;
	case 3:
	  v_Viewer.getRenderArea().setTransparencyType (SoGLRenderAction.TransparencyTypes.SORTED_PIXEL);
	  break;
	}
	break;
      }
    } break;

/*    case  PoAwtElementData.TRIGGER_BUTTON: {
      SoWriteAction myAction = new SoWriteAction ();
      myAction.getOutput().openFile (v_WindowTitle + ".iv");
      myAction.getOutput().setBinary (false);
      myAction.apply (v_Root);
      myAction.getOutput().closeFile ();
    } break;*/

    case  PoAwtElementData.TOGGLE_BUTTON : {
      SoSwitch toggleSwitch;
      switch (elementIndex) {
      case TOGGLE_SKIN:
	toggleSwitch = v_MeshSkinSwitch;
	break;
      case TOGGLE_SKELETON:
	toggleSwitch = v_MeshSkeletonSwitch;
	break;
      case TOGGLE_LEVEL_SURF:
	toggleSwitch = v_MeshLevelSurfSwitch;
	break;
      case TOGGLE_CROSS_SECTION:
	toggleSwitch = v_MeshCrossSectionSwitch;
	break;
      case TOGGLE_CROSS_CONTOUR:
	toggleSwitch = v_MeshCrossContourSwitch;
	break;
      case TOGGLE_VEC_FIELD:
	toggleSwitch = v_MeshVecFieldSwitch;
	break;
	//      case TOGGLE_DATA_LEGEND:
	//	toggleSwitch = v_DataLegendSwitch;
	//	break;
	/*      case TOGGLE_DRAGGER:
		toggleSwitch = v_DraggerSwitch;
		break;*/
	// ADDED: Toggle button for transparency
      case TOGGLE_TRANSPARENCY:
	toggleSwitch = null;   // indicates that toggleSwitch won't be used
	{
	  SoMaterial material =  (SoMaterial)v_MeshSkin.getPart("appearance.material");
	  if (v_TransparencyValue == 0.0f) {
	    v_TransparencyValue = 0.5f;
	    float tranparencies[] = new float[material.diffuseColor.getNum()];
	    for (int i=0; i<material.diffuseColor.getNum(); i++) tranparencies[i] = v_TransparencyValue;
	    material.transparency.setValues(0,tranparencies);
	  } else {
	    v_TransparencyValue = 0.0f;
	    material.transparency.setValue(v_TransparencyValue);
	  }
	}
	break;
	// END OF ADDED PART
      case TOGGLE_CLIP:
	toggleSwitch = null;   // indicates that toggleSwitch won't be used
	if (v_ClipPlane.on.getValue ())
	  v_ClipPlane.on.setValue (false);
	else
	  v_ClipPlane.on.setValue(true);
	break;
      default:
	throw new IllegalArgumentException ("Caught an event from an unknown toggle button");
      }
      if (toggleSwitch != null)
	if (toggleSwitch.whichChild.getValue() == SoSwitch.SO_SWITCH_ALL)
	  toggleSwitch.whichChild.setValue (SoSwitch.SO_SWITCH_NONE);
	else
	  toggleSwitch.whichChild.setValue (SoSwitch.SO_SWITCH_ALL);
    } break;

    }
  }

  //--------------------------------------------------------------------------------------------
  private void setCrossSection(SbVec3f pos) {
    // set the dragger position
    v_Dragger.translation.setValue(pos) ;
    v_Dragger.rotation.setValue(new SbRotation(new SbVec3f(0,1,0), v_PlaneNormal)) ;
    v_Dragger.scaleFactor.setValue(v_ScaleFactor, v_ScaleFactor, v_ScaleFactor) ;

    updateAllConnectedToDragger(v_Dragger) ;
  }

  //--------------------------------------------------------------------------------------------
  private void updateAllConnectedToDragger(SoJackDragger dragger) {

    // get the dragger position
    SbVec3f drager_position = dragger.translation.getValue() ;

    // get the dragger scale factor
    SbVec3f drager_scale = dragger.scaleFactor.getValue() ;
    v_ScaleFactor = drager_scale.getX() ;

    // rotate the plane's normal by the drager rotation
    SbRotation rotation = dragger.rotation.getValue() ;
    v_PlaneNormal = rotation.multVec(new SbVec3f(0,1,0)) ;

    // translate cross sction and cross contour
    v_MeshCrossSection.plane.setValue(new SbPlane(v_PlaneNormal, drager_position)) ;
    v_MeshCrossContour.plane.setValue(new SbPlane(v_PlaneNormal, drager_position)) ;

    // translate clip plane
    SbVec3f clip_norm = v_PlaneNormal ;
    clip_norm.negate() ;

    v_ClipPlane.plane.setValue(new SbPlane(clip_norm, drager_position)) ;

    // translate vec-field
    v_MeshVecField.plane.setValue(new SbPlane(v_PlaneNormal, drager_position)) ;

    // move start-points of streamline
    // v_MeshStreamLines.startPoints.setValue(drager_position) ;
    SbVec3f n = v_PlaneNormal ;
    float[] xyz = n.getValue() ;
    if (xyz[2] != 0) n.setValue(-xyz[2], 0, xyz[0]) ;
    else if (xyz[0] != 0) n.setValue(-xyz[1], xyz[0], 0) ;
    else n.setValue(1,0,0) ;

    // move stream surface
    v_MeshStreamSurfaces.startPoints.set1Value(0, drager_position) ;
    v_MeshStreamSurfaces.rakeOrientation.set1Value(0, n) ;
    v_MeshStreamSurfaces.rakeLength.setValue(v_ScaleFactor * RADIUS_SRC_CIRCLE) ;

    // move stream lines, stream points & stream spheres
    moveStreamSource(dragger) ;
  }

  //--------------------------------------------------------------------------------------------
  private void moveStreamSource(SoJackDragger dragger) {
    // get the dragger position
    SbVec3f drager_position = dragger.translation.getValue() ;

    // get the dragger scale factor
    SbVec3f drager_scale = dragger.scaleFactor.getValue() ;

    // get the dragger rotation
    SbRotation rotation = dragger.rotation.getValue() ;

    // move stream lines, stream points & stream spheres
    double alpha = 0 ;
    double d_alpha = 2 * Math.PI / v_NumStartPoints ;

    SbVec3f start_point = new SbVec3f() ;

    SbMatrix m = new SbMatrix() ;
    m.setTransform(drager_position, rotation, drager_scale) ;
    v_MeshStreamLines.startPoints.setNum(v_NumStartPoints) ;

    for (int i=0 ; i<v_NumStartPoints ; i++, alpha += d_alpha) {
      start_point.setValue((float) (RADIUS_SRC_CIRCLE * Math.cos(alpha)),
			   0,
			   (float) (RADIUS_SRC_CIRCLE * Math.sin(alpha))) ;
      start_point = m.multVecMatrix(start_point) ;
      v_MeshStreamLines.startPoints.set1Value(i, start_point) ;
    }
  }

  //--------------------------------------------------------------------------------------------
  private SoSeparator readFile(String filename) {
    SoInput mySceneInput = new SoInput() ;
    boolean canBeRead ;
    try {
      canBeRead = mySceneInput.openFile(filename) || mySceneInput.openURL(new URL(filename)) ;
    }
    catch (MalformedURLException exc) { canBeRead = false ; }

    if (! canBeRead) {
      System.err.println("Cannot open "+filename);
      return null ;
    }

    // read the whole file into the database
    SoSeparator myGraph = SoDB.readAll(mySceneInput) ;
    if (myGraph == null) {
      System.err.println("Could not resolve any scene graph.") ;
      return null ;
    }

    mySceneInput.closeFile() ;
    return myGraph ;
  }

  //--------------------------------------------------------------------------------------------
  // Internal classes
  class DialogListener extends PoAwtDialog.Listener
  {
    @Override
    public void valueChanged(int elementIndex)
    {
      manageDialog(elementIndex);
    }
  }

  class MotionCallback extends SoDraggerCB {
    public void invoke (SoDragger dragger) {
      updateAllConnectedToDragger((SoJackDragger)dragger) ;
    }
  }
}
