package meshviz.mesh.sample.dialog;

import java.awt.*;
import java.awt.event.*;

/**
  *  PoAwtToggleButtonData
  *
  * Class to define a toggle button. This element is composed of a string on the
  * left, a toggle button in the center, and a string associated to the state of
  * the toggle on the right. By default the text of the label, the two strings
  * associated to the state of the toggle button are empty and the state of the
  * toggle button is OFF.
  *
  * @author Rodolphe Albout
  * @author Laurent ISTIN
  * @author Loic Vigneras
  * @author Patrick Vigneras
  */
public class PoAwtToggleButtonData extends PoAwtElementData {

  // Constants
  // ---------

  /** Constant for the 'on' (true) state of a toggle button.
    */
  public static final boolean ON = true;
  /** Constant for the 'off' (false) state of a toggle button.
    */
  public static final boolean OFF = false;

  // Number of pixels between the label and the toggle button
  private static final int MARGIN_OFFSET = 6;

  // Constructors
  // ------------

  /** Construct a new PoAwtToggleButtonData with no label, no state strings,
    * and set to the default value OFF.
    */
  public PoAwtToggleButtonData () {
    this (null, OFF, null, null);
  }

  /** Construct a new PoAwtToggleButtonData with the given label. State strings
    * are null, and it is set to the default value OFF.
    */
  public PoAwtToggleButtonData (String label) {
    this (label, OFF, null, null);
  }

  /** Construct a new PoAwtToggleButtonData with no label, no state strings,
    * and set to the given state.
    */
  public PoAwtToggleButtonData (boolean state) {
    this (null, state, null, null);
  }

  /** Construct a new PoAwtToggleButtonData with the given label and set to the
    * given state. State strings are null.
    */
  public PoAwtToggleButtonData (String label, boolean state) {
    this (label, state, null, null);
  }

  /** Construct a new PoAwtToggleButtonData with the given label, the given
    * state strings, and set to the given state.
    */
  public PoAwtToggleButtonData (String label, boolean state,
				String onString, String offString) {
    super ();
    _label = new PoAwtLabelData (label, PoAwtLabelData.ALIGNMENT_END);
    _toggle = new Checkbox ();
    linkInnerComponents ();
    setString (ON, onString);
    setString (OFF, offString);
    setState (state);
  }

  /** Construct a new PoAwtToggleButtonData with the same values as the given one.
    */
  protected PoAwtToggleButtonData (PoAwtToggleButtonData toggle) {
    super ();
    _label = new PoAwtLabelData (toggle.getLabel (), PoAwtLabelData.ALIGNMENT_END);
    _toggle = new Checkbox ();
    linkInnerComponents ();
    setString (ON, toggle.getString (ON));
    setString (OFF, toggle.getString (OFF));
    setState (toggle.getState ());
    setID (toggle.getID ());
  }

  // Methods
  // -------

  /** Set the text of the label on the left of toggle button.
    */
  public void setLabel (String label) {
    _label.setLabel (label);
  }

  /** Get the text of the label on the left of toggle button.
    */
  public String getLabel () {
    return _label.getLabel ();
  }

  /** Set a string for the given state.
    */
  public void setString (boolean state, String str) {
    if (state == ON)
      _stringOn = str;
    else
      _stringOff = str;
    setState (getState ());  // refresh the button and its label
  }
  /** Get the string for the given state.
    */
  public String getString (boolean state) {
    if (state == ON)
      return _stringOn;
    else
      return _stringOff;
  }

  /** Set the state of the toggle button.
    */
  public void setState (boolean state) {
    _toggle.setState (state);
    _toggle.setLabel (getString (state));
  }

  /** Get the state of the toggle button.
    */
  public boolean getState () {
    return _toggle.getState ();
  }

  /** Returns the type of the element.
    */
  public int getType () {
    return TOGGLE_BUTTON;
  }

  /** Creates and returns a new element which is a copy of this element.
    */
  protected Object clone () {
    return new PoAwtToggleButtonData (this);
  }

  /** Affect the internal data awtElement to current object.
    * The current object and awtElement have to be the same object type.
    * This method should be called by PoAwtDialog, and not by user code
    * directly.
    */
  public void updateValues (PoAwtElementData element) {
    if (element.getType () == TOGGLE_BUTTON) {
      PoAwtToggleButtonData toggle = (PoAwtToggleButtonData) element;

      setLabel  (toggle.getLabel ());
      setString (ON,  toggle.getString (ON ));
      setString (OFF, toggle.getString (OFF));
      setState  (toggle.getState ());
      setID     (toggle.getID ());
    }
  }

  /** Minimum width for the label of the element.
    * The mimimum width for a elementid with no label will be 0 (Default behaviour).
    * The width is the size in pixel.
    * This method should be called by PoAwtDialog, and not by user code
    * directly.
    */
  public int gelMinLabelWidth () {
    return _label.getLength ();
  }

  /** This is call when the dialog is resized, It's use for resize some awtElement.
    * This method should be called by PoAwtDialog, and not by user code
    * directly.
    */
  public void setMinLabelWidth (int minWidth) {
    _label.setLength (minWidth);
  }

  // Layout and link inner components to the panel of this element
  private void linkInnerComponents () {
    GridBagLayout gridbag = new GridBagLayout ();
    this.setLayout (gridbag);
    GridBagConstraints c = new GridBagConstraints ();
    c.fill = GridBagConstraints.BOTH;
    c.weightx = 0.0;
    c.weighty = 1.0;
    gridbag.setConstraints (_label, c);
    this.add (_label);
    c.weightx = 1.0;
    c.insets = new Insets (0, MARGIN_OFFSET, 0, 0);
    gridbag.setConstraints (_toggle, c);
    this.add (_toggle);
    _toggle.addItemListener(new ItemListener(){
      public void itemStateChanged(ItemEvent e) {
        invokeValueChanged();
        setState (getState ());      // refresh the button and its label
      }
    });
  }

  // Fields
  // ------

  // Inner components
  private PoAwtLabelData _label;
  private Checkbox _toggle;      // includes the state of this toggle button

  // Strings associated to the state of the toggle button
  private String _stringOn;
  private String _stringOff;

}
