package meshvizxlm.eclipsemeshviz.gui;

import java.awt.BorderLayout;

import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTree;
import javax.swing.UIManager;
import javax.swing.UIManager.LookAndFeelInfo;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.TreePath;

import meshvizxlm.eclipsemeshviz.gui.AnimationPanel.AnimationListener;
import meshvizxlm.eclipsemeshviz.gui.IsosurfacePanel.IsosurfaceMeshListener;
import meshvizxlm.eclipsemeshviz.pillargrid.PillarGrid;
import meshvizxlm.eclipsemeshviz.wrappers.PillarMesh;

public class EclipseMeshVizFrame extends JFrame implements MeshChangedListener
{
  private JPanel m_contentPane;
  private JPanel m_paramPanel;

  // Settings panels
  private FilteringPanel m_roiPanel;
  private MeshTypePanel m_meshPanel;
  private GeometryPanel m_geomPanel;
  private FullMeshPanel m_fullMeshPanel;
  private HullPanel m_hullPanel;
  private SlabPanel m_slabPanel;
  private GeometrySlicePanel m_geomSlicePanel;
  private IsosurfacePanel m_isosurfacePanel;
  private DownsamplingPanel m_downsamplingPanel;
  private AnimationPanel m_animationPanel;
  private InfoPanel m_infoPanel;
  private DrawStylePanel m_drawStylePanel;

  /**
   * Create the frame.
   */
  public EclipseMeshVizFrame()
  {
    // Set Nimbus look and feel
    if ( !System.getProperty("os.name").startsWith("Mac") )
    {
      LookAndFeelInfo[] lfs = UIManager.getInstalledLookAndFeels();
      for ( LookAndFeelInfo info : lfs )
      {
        if ( "Nimbus".equals(info.getName()) )
          try
          {
            UIManager.setLookAndFeel(info.getClassName());
          }
          catch (Exception exc)
          {
            System.out.println("cannot change look and feel " + exc.getMessage());
          }
      }
    }

    buildSettingsPanels();
    buildFrame();
    setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
  }

  public void registerListeners(ScenegraphManager sceneManager)
  {
    addGeometryListener(sceneManager);
    addFullMeshListener(sceneManager);
    addRenderingOptionListener(sceneManager);
    addHullListener(sceneManager);
    addSlabListener(sceneManager);
    addGeometrySliceListener(sceneManager);
    addROIListener(sceneManager);
    addDownsamplingListener(sceneManager);
    addAnimationListener(sceneManager);
    addIsosurfaceListener(sceneManager);
    sceneManager.addDownsamplingListener(m_downsamplingPanel);
  }

  public void addROIListener(FilteringPanel.FilteringListener listener)
  {
    m_roiPanel.addROIListener(listener);
  }

  public void addGeometryListener(GeometryPanel.GeometryListener listener)
  {
    m_geomPanel.addGeometryListener(listener);
  }

  public void addFullMeshListener(FullMeshPanel.FullMeshListener listener)
  {
    m_fullMeshPanel.addFullMeshListener(listener);
  }

  public void addRenderingOptionListener(RenderingOptionListener listener)
  {
    m_drawStylePanel.addRenderingOptionListener(listener);
  }

  public void addHullListener(HullPanel.HullListener listener)
  {
    m_hullPanel.addHullListener(listener);
  }

  public void addSlabListener(SlabAxisPanel.SlabListener listener)
  {
    m_slabPanel.addSlabListener(listener);
  }

  public void addGeometrySliceListener(GeometrySlicePanel.GeometrySliceListener listener)
  {
    m_geomSlicePanel.addGeometrySliceListener(listener);
  }

  public void addDownsamplingListener(DownsamplingListener listener)
  {
    m_downsamplingPanel.addListener(listener);
  }

  public void addAnimationListener(AnimationListener listener)
  {
    m_animationPanel.addListener(listener);
  }

  public void addIsosurfaceListener(IsosurfaceMeshListener listener)
  {
    m_isosurfacePanel.addIsosurfaceListener(listener);
  }

  @Override
  public void meshChanged(PillarMesh<PillarGrid> newMesh)
  {
    m_slabPanel.xSlab.updateSlabMax(newMesh.getDimI() - 1);
    m_slabPanel.ySlab.updateSlabMax(newMesh.getDimJ() - 1);
    m_slabPanel.zSlab.updateSlabMax(newMesh.getDimK() - 1);
    m_slabPanel.xSlab.setSlabOriginalValues(newMesh.getDimI() - 1);
    m_slabPanel.ySlab.setSlabOriginalValues(newMesh.getDimJ() - 1);
    m_slabPanel.zSlab.setSlabOriginalValues(newMesh.getDimK() - 1);
    m_roiPanel.updateROI(0, newMesh.getDimI(), 0, newMesh.getDimJ(), 0, newMesh.getDimK());
    m_roiPanel.updateDataRange((float) newMesh.getDataSet().getMin(), (float) newMesh.getDataSet().getMax());
    m_isosurfacePanel.updateDataRange((float) newMesh.getDataSet().getMin(), (float) newMesh.getDataSet().getMax());
    // update clip plane with selected axis and angle
    m_geomSlicePanel.update();
    m_infoPanel.meshChanged(newMesh);
    m_meshPanel.updateComboBox();
  }

  @Override
  public void propertyChanged(PillarMesh<PillarGrid> newMesh)
  {
    m_slabPanel.xSlab.updateSlabMax(newMesh.getDimI() - 1);
    m_slabPanel.ySlab.updateSlabMax(newMesh.getDimJ() - 1);
    m_slabPanel.zSlab.updateSlabMax(newMesh.getDimK() - 1);
    m_roiPanel.updateROI(0, newMesh.getDimI(), 0, newMesh.getDimJ(), 0, newMesh.getDimK());
    m_roiPanel.updateDataRange((float) newMesh.getDataSet().getMin(), (float) newMesh.getDataSet().getMax());
    m_isosurfacePanel.updateDataRange((float) newMesh.getDataSet().getMin(), (float) newMesh.getDataSet().getMax());
    // update clip plane with selected axis and angle
    m_geomSlicePanel.update();
  }

  private void buildFrame()
  {
    setBounds(50, 50, 800, 500);
    setTitle("Eclipse MeshViz Example");
    setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    m_contentPane = new JPanel();
    m_contentPane.setBorder(new EmptyBorder(5, 5, 5, 5));
    m_contentPane.setLayout(new BorderLayout(0, 0));
    setContentPane(m_contentPane);

    JSplitPane splitPane = new JSplitPane();
    splitPane.setResizeWeight(0.25);
    m_contentPane.add(splitPane, BorderLayout.CENTER);

    JScrollPane panelScrollPane = new JScrollPane();
    splitPane.setRightComponent(panelScrollPane);

    m_paramPanel = new JPanel(new BorderLayout());
    panelScrollPane.setViewportView(m_paramPanel);

    JScrollPane treeScrollPane = new JScrollPane();
    splitPane.setLeftComponent(treeScrollPane);

    JTree paramTree = makeTree();
    treeScrollPane.setViewportView(paramTree);

    JPanel panel = new JPanel();
    m_contentPane.add(panel, BorderLayout.SOUTH);
    panel.setLayout(new BorderLayout(0, 0));

    JPanel downsamplingPanel = new JPanel();
    downsamplingPanel
        .setBorder(new TitledBorder(null, "Resolution", TitledBorder.LEADING, TitledBorder.TOP, null, null));
    downsamplingPanel.add(m_downsamplingPanel, BorderLayout.CENTER);
    panel.add(downsamplingPanel, BorderLayout.NORTH);
  }

  private JTree makeTree()
  {
    DefaultMutableTreeNode meshNode = m_meshPanel != null ? new DefaultMutableTreeNode(m_meshPanel) : null;

    DefaultMutableTreeNode geometryNode = new DefaultMutableTreeNode(m_geomPanel);
    DefaultMutableTreeNode fullMeshNode = new DefaultMutableTreeNode(m_fullMeshPanel);
    DefaultMutableTreeNode hullNode = new DefaultMutableTreeNode(m_hullPanel);
    DefaultMutableTreeNode slabsNode = new DefaultMutableTreeNode(m_slabPanel);
    DefaultMutableTreeNode geomSliceNode = new DefaultMutableTreeNode(m_geomSlicePanel);
    DefaultMutableTreeNode isosurfNode = new DefaultMutableTreeNode(m_isosurfacePanel);

    geometryNode.add(fullMeshNode);
    geometryNode.add(hullNode);
    geometryNode.add(slabsNode);
    geometryNode.add(geomSliceNode);
    geometryNode.add(isosurfNode);

    DefaultMutableTreeNode roiNode = new DefaultMutableTreeNode(m_roiPanel);
    DefaultMutableTreeNode animationNode = new DefaultMutableTreeNode(m_animationPanel);
    DefaultMutableTreeNode drawStyleNode = new DefaultMutableTreeNode(m_drawStylePanel);
    DefaultMutableTreeNode infoNode = new DefaultMutableTreeNode(m_infoPanel);

    DefaultMutableTreeNode root = new DefaultMutableTreeNode();
    if ( meshNode != null )
      root.add(meshNode);
    root.add(geometryNode);
    root.add(roiNode);
    root.add(animationNode);
    root.add(drawStyleNode);
    root.add(infoNode);

    final JTree paramTree = new JTree(root);
    paramTree.setShowsRootHandles(true);
    paramTree.setRootVisible(false);

    DefaultTreeCellRenderer cellRenderer = new DefaultTreeCellRenderer();
    cellRenderer.setLeafIcon(null);
    cellRenderer.setOpenIcon(null);
    cellRenderer.setClosedIcon(null);
    paramTree.setCellRenderer(cellRenderer);

    paramTree.addTreeSelectionListener(new TreeSelectionListener()
    {
      @Override
      public void valueChanged(TreeSelectionEvent e)
      {
        DefaultMutableTreeNode selected = (DefaultMutableTreeNode) e.getPath().getLastPathComponent();
        Object userObj = selected.getUserObject();

        if ( userObj instanceof JPanel )
        {
          JPanel pn = (JPanel) userObj;
          m_paramPanel.removeAll();
          m_paramPanel.add(pn, BorderLayout.NORTH);
        }
        else if ( !selected.isLeaf() )
          paramTree.setSelectionPath(new TreePath(((DefaultMutableTreeNode) selected.getFirstChild()).getPath()));

        m_paramPanel.getParent().validate();
        m_paramPanel.getParent().repaint();
      }
    });
    for ( int i = 0; i < paramTree.getRowCount(); i++ )
      paramTree.expandRow(i);
    paramTree.setSelectionRow(0);

    return paramTree;
  }

  private void buildSettingsPanels()
  {
    m_meshPanel = new MeshTypePanel();
    m_geomPanel = new GeometryPanel();
    m_fullMeshPanel = new FullMeshPanel();
    m_hullPanel = new HullPanel();
    m_isosurfacePanel = new IsosurfacePanel();
    m_geomSlicePanel = new GeometrySlicePanel();
    m_roiPanel = new FilteringPanel();
    m_downsamplingPanel = new DownsamplingPanel();
    m_animationPanel = new AnimationPanel();
    m_infoPanel = new InfoPanel();
    m_slabPanel = new SlabPanel();
    m_drawStylePanel = new DrawStylePanel();
  }
}
