package meshvizxlm.eclipsemeshviz.pillargrid;

import java.util.Arrays;

import meshvizxlm.eclipsemeshviz.DemoSettings;

/**
 * This type of PillarGrid must store 8 zcorn values per level at each pillar.
 * Consecutive zcorn values are along the k dimension. The total number of zcorn
 * is 8(nx*ny*nz).
 */
public class PillarEGrid extends PillarGrid
{
  @Override
  public float getProperty(int i, int j, int k)
  {
    return getDepth(i, j, k);
  }

  @Override
  public float getDepth(int i, int j, int k)
  {
    int zcornIndex = m_numHFacesAtKLevel * k * m_4DimIDimJ + 2 * j * m_2DimI + 2 * i;
    return m_zcorn[zcornIndex];
  }

  @Override
  public float getPorosity(int i, int j, int k)
  {
    int poroIndex = m_dimIxJ * k + m_dimI * j + i;
    return m_poro[poroIndex];
  }

  @Override
  public int getNextUpperCellFaceZCornId(int zcornId)
  {
    return zcornId + m_numHFacesAtKLevel * m_4DimIDimJ;
  }

  @Override
  public int getNextUpperFaceZCornId(int zcornId)
  {
    return zcornId + m_4DimIDimJ;
  }

  @Override
  public void setDim(int numCellI, int numCellJ, int numCellK)
  {
    m_dimI = numCellI;
    m_dimJ = numCellJ;
    m_dimK = numCellK;
    m_dimIxJ = m_dimI * m_dimJ;
    m_2DimI = 2 * m_dimI;
    m_4DimIDimJ = 2 * m_2DimI * m_dimJ;
    // if horizontal faults enable count 2 times more zcorns (8 * ni * nj *nk)
    // otherwise only (4 * ni * nj * (nk+1))
    m_numZcorns = m_numHFacesAtKLevel * m_4DimIDimJ * (m_dimK + 2 - m_numHFacesAtKLevel);

    // allocate pillar coordinates
    int nPillars = (m_dimI + 1) * (m_dimJ + 1);
    int nData = 6 * nPillars;
    m_coord = new float[nData];
    Arrays.fill(m_coord, 0);

    // allocate actnum
    nData = m_dimK * m_dimIxJ;
    m_actnum = new boolean[nData];
    Arrays.fill(m_actnum, true);

    // allocate z corner
    m_zcorn = new float[(int) m_numZcorns];
    Arrays.fill(m_zcorn, 0);

    // allocate porosity
    m_poro = new float[nData];
    Arrays.fill(m_poro, 0);
  }

  @Override
  public double[] computePillarVertex(int zcornId)
  {
    int dij = zcornId % m_4DimIDimJ;
    int dj = dij / m_2DimI;
    int j = dj / 2;
    int jPos = dj % 2;
    int di = dij % m_2DimI;
    int i = di / 2;
    int iPos = di % 2;

    int pillarIndex = 6 * ((j + jPos) * (m_dimI + 1) + (i + iPos));

    float[] P12 = new float[6];
    P12[0] = m_coord[pillarIndex++];
    P12[1] = m_coord[pillarIndex++];
    P12[2] = m_coord[pillarIndex++];
    P12[3] = m_coord[pillarIndex++];
    P12[4] = m_coord[pillarIndex++];
    P12[5] = m_coord[pillarIndex];

    double[] xyz = new double[3];
    generatePillarVertex(P12, m_zcorn[zcornId], xyz);
    return xyz;
  }

  public void getPillarZCorns(int i, int j, int k, float[] coords, int[] indexedZCorns)
  {
    if ( DemoSettings.HORIZONTAL_FAULTS )
      getPillarZCornsHorizontalFaults(i, j, k, coords, indexedZCorns);
    else
      getPillarZCornsNoHorizontalFaults(i, j, k, coords, indexedZCorns);
  }

  private void getPillarZCornsHorizontalFaults(int i, int j, int k, float[] coords, int[] indexedZCorns)
  {
    int zcorn_index = m_numHFacesAtKLevel * k * m_4DimIDimJ + 2 * j * m_2DimI + 2 * i;
    int startIndex = 0;
    for ( int n = 0; n < 2; ++n )
    {
      coords[startIndex] = coords[startIndex + 1] = coords[startIndex + 2] = coords[startIndex + 3] = UNDEF;
      if ( k >= 0 && k < m_dimK )
      {
        if ( j < m_dimJ )
        {
          if ( i < m_dimI )
          {
            indexedZCorns[startIndex] = zcorn_index;
            coords[startIndex] = m_zcorn[zcorn_index];
          }
          if ( i > 0 )
          {
            indexedZCorns[startIndex + 1] = zcorn_index - 1;
            coords[startIndex + 1] = m_zcorn[zcorn_index - 1];
          }
        }
        if ( j > 0 )
        {
          if ( i > 0 )
          {
            indexedZCorns[startIndex + 2] = (zcorn_index - m_2DimI - 1);
            coords[startIndex + 2] = m_zcorn[indexedZCorns[startIndex + 2]];
          }
          if ( i < m_dimI )
          {
            indexedZCorns[startIndex + 3] = (zcorn_index - m_2DimI);
            coords[startIndex + 3] = m_zcorn[indexedZCorns[startIndex + 3]];
          }
        }
      }
      startIndex += 4;
      --k;
      zcorn_index -= m_4DimIDimJ;
    }
  }

  private void getPillarZCornsNoHorizontalFaults(int i, int j, int k, float[] coords, int[] indexedZCorns)
  {
    int zcorn_index = m_numHFacesAtKLevel * k * m_4DimIDimJ + 2 * j * m_2DimI + 2 * i;
    coords[0] = coords[1] = coords[2] = coords[3] = UNDEF;
    if ( k == m_dimK && m_numHFacesAtKLevel == 2 )
      zcorn_index -= m_4DimIDimJ;
    if ( j < m_dimJ )
    {
      if ( i < m_dimI )
      {
        indexedZCorns[0] = zcorn_index;
        coords[0] = m_zcorn[zcorn_index];
      }
      if ( i > 0 )
      {
        indexedZCorns[1] = zcorn_index - 1;
        coords[1] = m_zcorn[zcorn_index - 1];
      }
    }
    if ( j > 0 )
    {
      if ( i > 0 )
      {
        indexedZCorns[2] = (zcorn_index - m_2DimI - 1);
        coords[2] = m_zcorn[indexedZCorns[2]];
      }
      if ( i < m_dimI )
      {
        indexedZCorns[3] = (zcorn_index - m_2DimI);
        coords[3] = m_zcorn[indexedZCorns[3]];
      }
    }
  }
}
