package meshvizxlm.eclipsemeshviz.scenegraph;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import meshvizxlm.eclipsemeshviz.DemoSettings.MeshType;
import meshvizxlm.eclipsemeshviz.gui.MeshChangedListener;
import meshvizxlm.eclipsemeshviz.gui.MeshListener;
import meshvizxlm.eclipsemeshviz.pillargrid.PillarGrid;
import meshvizxlm.eclipsemeshviz.pillargrid.PillarGridBinReader;
import meshvizxlm.eclipsemeshviz.pillargrid.PillarGridGenerator;
import meshvizxlm.eclipsemeshviz.pillargrid.PillarGridZIPReader;
import meshvizxlm.eclipsemeshviz.wrappers.CachedPillarMesh;
import meshvizxlm.eclipsemeshviz.wrappers.CachedTopoPillarMesh;
import meshvizxlm.eclipsemeshviz.wrappers.PillarMesh;

public class MeshReader implements MeshListener
{
  private static class MeshReaderHolder
  {
    private static final MeshReader instance = new MeshReader();
  }

  public static MeshReader getInstance()
  {
    return MeshReaderHolder.instance;
  }

  private PillarGrid m_grid;
  private MeshType m_meshType;
  private PillarMesh<PillarGrid> m_mesh;
  private ArrayList<MeshChangedListener> m_listeners;

  private MeshReader()
  {
    m_grid = null;
    m_listeners = new ArrayList<MeshChangedListener>();
  }

  public void addListener(MeshChangedListener listener)
  {
    m_listeners.add(listener);
  }

  public void clearListeners()
  {
    m_listeners.clear();
  }

  private void readMesh(File file) throws Exception
  {
    System.out.println("Loading mesh from " + file.getPath());

    PillarGrid newGrid = PillarGrid.getNewInstance();
    String filename = file.getName();

    if ( filename.endsWith(".txt") )
    {
      PillarGridGenerator generator = new PillarGridGenerator();
      generator.generate(newGrid, file.getAbsolutePath(), null);
    }
    else if ( filename.endsWith(".zip") )
    {
      PillarGridZIPReader fileReader = new PillarGridZIPReader();
      fileReader.read(newGrid, file);
    }
    else
    {
      PillarGridBinReader fileReader = new PillarGridBinReader();
      fileReader.read(newGrid, file);
    }

    m_grid = newGrid;
    m_meshType = null;
  }

  private void loadMesh(MeshType type) throws UnsupportedOperationException
  {
    if ( m_meshType != type && m_grid != null )
    {
      m_meshType = type;
      loadMesh(m_grid);
    }
  }

  private void loadMesh(PillarGrid grid) throws UnsupportedOperationException
  {
    switch ( m_meshType )
    {
    case FULLCACHE :
      m_mesh = new CachedPillarMesh<PillarGrid>(grid);
      break;
    case TOPOCACHE :
      m_mesh = new CachedTopoPillarMesh<PillarGrid>(grid);
      break;
    case NOCACHE :
      m_mesh = new PillarMesh<PillarGrid>(grid);
      break;
    default:
      throw new UnsupportedOperationException("EclipseMeshViz: Can't load mesh, mesh type not supported: " + m_meshType);
    }
  }

  @Override
  public PillarMesh<PillarGrid> meshChanged(MeshType meshType, File file)
  {
    try
    {
      readMesh(file);
      loadMesh(meshType);
      notifyListeners();
      return m_mesh;
    }
    catch (Exception e)
    {
      e.printStackTrace();
    }
    return null;
  }

  @Override
  public PillarMesh<PillarGrid> meshTypeChanged(MeshType meshType)
  {
    loadMesh(meshType);
    notifyListeners();
    return m_mesh;
  }

  @Override
  public void propertyChanged(String property)
  {
    if ( property.equals("Depth") )
    {
      m_mesh.setCurrentDatasetToDepth();
    }
    else
      m_mesh.setCurrentDatasetToPorosity();

    for ( MeshChangedListener listener : m_listeners )
      listener.propertyChanged(m_mesh);
  }

  @Override
  public void meshSaved(File file)
  {
    PillarGridBinReader saveFile = new PillarGridBinReader();
    try
    {
      if ( m_grid != null )
        saveFile.write(m_grid, file);
    }
    catch (IOException e)
    {
      e.printStackTrace();
    }
  }

  private void notifyListeners()
  {
    for ( MeshChangedListener listener : m_listeners )
      listener.meshChanged(m_mesh);
  }

  public void init(MeshType defaultMeshType, File defaultFile)
  {
    meshChanged(defaultMeshType, defaultFile);
  }

  @Override
  public boolean isPorosityAvailable()
  {
    return m_grid.isPorosityAvailable();
  }

}
