package meshvizxlm.mapping.streamlines;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.border.EtchedBorder;

import com.openinventor.inventor.SbMatrix;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.draggers.SoDragger;
import com.openinventor.inventor.draggers.SoJackDragger;
import com.openinventor.inventor.misc.callbacks.SoDraggerCB;
import com.openinventor.inventor.nodes.SoEnvironment;
import com.openinventor.inventor.nodes.SoLightModel;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoPickStyle;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoShapeHints;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.meshvizxlm.mapping.nodes.*;
import com.openinventor.meshvizxlm.mesh.StorageLayoutIJK;
import com.openinventor.meshvizxlm.mesh.data.DataBinding;
import com.openinventor.meshvizxlm.mesh.data.MiScalardSetI;
import com.openinventor.meshvizxlm.mesh.data.MiScalardSetIjk;

import meshvizxlm.mapping.MyVec3Set;
import meshvizxlm.mesh.MbSampleMeshBuilder;
import meshvizxlm.mesh.volumes.MbHexahedronMeshIjk;
import meshvizxlm.mesh.volumes.MbVertexHexahedronMeshIjk;
import meshvizxlm.mesh.volumes.MbVolumeMeshHexahedron;
import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{

  class SpeedI implements MiScalardSetI
  {
    private MyVec3Set m_v;

    public SpeedI(MyVec3Set v)
    {
      m_v = v;
    }

    public double getMin()
    {
      return m_v.getMin().length();
    }

    public double getMax()
    {
      return m_v.getMax().length();
    }

    @Override
    public long getTimeStamp()
    {
      return m_v.getTimeStamp();
    }

    @Override
    public String getName()
    {
      return "Speed";
    }

    @Override
    public DataBinding getBinding()
    {
      return m_v.getBinding();
    }

    @Override
    public double get(long i)
    {
      double[] vec = m_v.get((int) i);
      double x = vec[0];
      double y = vec[1];
      double z = vec[2];

      return Math.sqrt(x * x + y * y + z * z);
    }
  }

  class SpeedIjk implements MiScalardSetIjk
  {
    private MyVec3Set m_v;

    public SpeedIjk(MyVec3Set v)
    {
      m_v = v;
    }

    public double getMin()
    {
      return m_v.getMin().length();
    }

    public double getMax()
    {
      return m_v.getMax().length();
    }

    @Override
    public long getTimeStamp()
    {
      return m_v.getTimeStamp();
    }

    @Override
    public String getName()
    {
      return "Speed";
    }

    @Override
    public DataBinding getBinding()
    {
      return m_v.getBinding();
    }

    @Override
    public double get(int i, int j, int k)
    {
      double[] vec = m_v.get(i, j, k);
      double x = vec[0];
      double y = vec[1];
      double z = vec[2];

      return Math.sqrt(x * x + y * y + z * z);
    }

    @Override
    public StorageLayoutIJK getStorageLayout()
    {
      return StorageLayoutIJK.KJI;
    }
  }

  class MotionCallback extends SoDraggerCB
  {
    @Override
    public void invoke(SoDragger dragger)
    {
      moveStreamSource((SoJackDragger) dragger);
    }
  }

  private static final int NUM_CELL_I = 10;
  private static final int NUM_CELL_J = 10;
  private static final int NUM_CELL_K = 10;
  private static final double[] MIN = { 0, 0, 0 };
  private static final double[] MAX = { 100, 100, 100 };
  private static final int NUM_START_POINTS = 7;
  private static final float RADIUS_SRC_CIRCLE = 0.25f;

  private IViewerExaminer m_renderArea;
  private MbSampleMeshBuilder m_meshBuilder;
  private MbVolumeMeshHexahedron m_meshHexa;
  private MbHexahedronMeshIjk m_meshHexaIjk;
  private MbVertexHexahedronMeshIjk m_meshVertexHexaIjk;
  private SbVec3f[] m_startPoints;
  private MoMeshStreamline m_streamline;
  private SoEnvironment m_envNode;
  private SoLightModel m_lightModel;
  private SoSwitch m_vectorSwitch;
  private MoDataBinding m_dataBinding;
  private MoMesh m_moMesh;
  private MyVec3Set m_vecSetPerNode;
  private MyVec3Set m_vecSetPerCell;
  private MoVec3SetI m_moVec3SetI;

  public Main()
  {
    m_meshBuilder = new MbSampleMeshBuilder();
    m_meshHexa = m_meshBuilder.getMeshHexahedron(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, MIN, MAX);
    m_meshHexaIjk = null;
    m_meshVertexHexaIjk = null;
    m_startPoints = new SbVec3f[NUM_START_POINTS];
    for ( int i = 0; i < NUM_START_POINTS; i++ )
      m_startPoints[i] = new SbVec3f();
  }

  @Override
  public void start()
  {
    SoNode root = buildSceneGraph();

    m_renderArea = ViewerComponentsFactory.createViewerExaminer();
    m_renderArea.setSceneGraph(root);
    m_renderArea.viewAll();

    // Options panel
    JPanel valuePanel = new JPanel(new GridBagLayout());

    // Show vectors
    JLabel lblShowVectors = new JLabel("Show vectors");
    GridBagConstraints gbc_lblShowVectors = new GridBagConstraints();
    gbc_lblShowVectors.anchor = GridBagConstraints.WEST;
    gbc_lblShowVectors.insets = new Insets(0, 0, 5, 5);
    gbc_lblShowVectors.gridx = 0;
    gbc_lblShowVectors.gridy = 0;
    valuePanel.add(lblShowVectors, gbc_lblShowVectors);

    JRadioButton yesVectors = new JRadioButton("yes");
    yesVectors.setSelected(m_vectorSwitch.whichChild.getValue() == SoSwitch.SO_SWITCH_ALL);
    yesVectors.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( yesVectors.isSelected() )
          m_vectorSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
        else
          m_vectorSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
      }
    });
    GridBagConstraints gbc_rdbtnYes = new GridBagConstraints();
    gbc_rdbtnYes.insets = new Insets(0, 0, 5, 5);
    gbc_rdbtnYes.gridx = 1;
    gbc_rdbtnYes.gridy = 0;
    valuePanel.add(yesVectors, gbc_rdbtnYes);

    JRadioButton noVectors = new JRadioButton("no");
    noVectors.setSelected(m_vectorSwitch.whichChild.getValue() == SoSwitch.SO_SWITCH_NONE);
    GridBagConstraints gbc_rdbtnNo = new GridBagConstraints();
    gbc_rdbtnNo.anchor = GridBagConstraints.WEST;
    gbc_rdbtnNo.insets = new Insets(0, 0, 5, 0);
    gbc_rdbtnNo.gridx = 2;
    gbc_rdbtnNo.gridy = 0;
    valuePanel.add(noVectors, gbc_rdbtnNo);

    ButtonGroup vectorsButtonGroup = new ButtonGroup();
    vectorsButtonGroup.add(yesVectors);
    vectorsButtonGroup.add(noVectors);

    // Ambient occlusion
    JLabel lblActivateAmbientOcclusion = new JLabel("Activate ambient occlusion");
    GridBagConstraints gbc_lblActivateAmbientOcclusion = new GridBagConstraints();
    gbc_lblActivateAmbientOcclusion.anchor = GridBagConstraints.WEST;
    gbc_lblActivateAmbientOcclusion.insets = new Insets(0, 0, 5, 5);
    gbc_lblActivateAmbientOcclusion.gridx = 0;
    gbc_lblActivateAmbientOcclusion.gridy = 1;
    valuePanel.add(lblActivateAmbientOcclusion, gbc_lblActivateAmbientOcclusion);

    JRadioButton yesAmbientOcclusion = new JRadioButton("yes");
    yesAmbientOcclusion.setSelected(m_envNode.ambientOcclusion.getValue());
    yesAmbientOcclusion.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        m_envNode.ambientOcclusion.setValue(yesAmbientOcclusion.isSelected());
      }
    });
    GridBagConstraints gbc_rdbtnYes_1 = new GridBagConstraints();
    gbc_rdbtnYes_1.insets = new Insets(0, 0, 5, 5);
    gbc_rdbtnYes_1.gridx = 1;
    gbc_rdbtnYes_1.gridy = 1;
    valuePanel.add(yesAmbientOcclusion, gbc_rdbtnYes_1);

    JRadioButton noAmbientOcclusion = new JRadioButton("no");
    noAmbientOcclusion.setSelected(!m_envNode.ambientOcclusion.getValue());
    GridBagConstraints gbc_rdbtnNo_1 = new GridBagConstraints();
    gbc_rdbtnNo_1.anchor = GridBagConstraints.WEST;
    gbc_rdbtnNo_1.insets = new Insets(0, 0, 5, 0);
    gbc_rdbtnNo_1.gridx = 2;
    gbc_rdbtnNo_1.gridy = 1;
    valuePanel.add(noAmbientOcclusion, gbc_rdbtnNo_1);

    ButtonGroup ambientOccButtonGroup = new ButtonGroup();
    ambientOccButtonGroup.add(yesAmbientOcclusion);
    ambientOccButtonGroup.add(noAmbientOcclusion);

    // Illuminated streamlines
    JLabel lblIlluminateStreamlines = new JLabel("Illuminate streamlines");
    GridBagConstraints gbc_lblIlluminateStreamlines = new GridBagConstraints();
    gbc_lblIlluminateStreamlines.anchor = GridBagConstraints.WEST;
    gbc_lblIlluminateStreamlines.insets = new Insets(0, 0, 5, 5);
    gbc_lblIlluminateStreamlines.gridx = 0;
    gbc_lblIlluminateStreamlines.gridy = 2;
    valuePanel.add(lblIlluminateStreamlines, gbc_lblIlluminateStreamlines);

    JRadioButton yesIlluminated = new JRadioButton("yes");
    yesIlluminated
        .setSelected(m_lightModel.model.getValue(SoLightModel.Models.class) == SoLightModel.Models.PER_VERTEX_PHONG);
    yesIlluminated.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( yesIlluminated.isSelected() )
          m_lightModel.model.setValue(SoLightModel.Models.PER_VERTEX_PHONG);
        else
          m_lightModel.model.setValue(SoLightModel.Models.BASE_COLOR);
      }
    });
    GridBagConstraints gbc_rdbtnYes_2 = new GridBagConstraints();
    gbc_rdbtnYes_2.insets = new Insets(0, 0, 5, 5);
    gbc_rdbtnYes_2.gridx = 1;
    gbc_rdbtnYes_2.gridy = 2;
    valuePanel.add(yesIlluminated, gbc_rdbtnYes_2);

    JRadioButton noIlluminated = new JRadioButton("no");
    noIlluminated.setSelected(m_lightModel.model.getValue(SoLightModel.Models.class) == SoLightModel.Models.BASE_COLOR);
    GridBagConstraints gbc_rdbtnNo_2 = new GridBagConstraints();
    gbc_rdbtnNo_2.anchor = GridBagConstraints.WEST;
    gbc_rdbtnNo_2.insets = new Insets(0, 0, 5, 0);
    gbc_rdbtnNo_2.gridx = 2;
    gbc_rdbtnNo_2.gridy = 2;
    valuePanel.add(noIlluminated, gbc_rdbtnNo_2);

    ButtonGroup illuminatedButtonGroup = new ButtonGroup();
    illuminatedButtonGroup.add(yesIlluminated);
    illuminatedButtonGroup.add(noIlluminated);

    // Speed vectors
    JLabel lblConvertSpeedVectors = new JLabel("Convert speed vectors into");
    GridBagConstraints gbc_lblConvertSpeedVectors = new GridBagConstraints();
    gbc_lblConvertSpeedVectors.anchor = GridBagConstraints.WEST;
    gbc_lblConvertSpeedVectors.insets = new Insets(0, 0, 10, 5);
    gbc_lblConvertSpeedVectors.gridx = 0;
    gbc_lblConvertSpeedVectors.gridy = 3;
    valuePanel.add(lblConvertSpeedVectors, gbc_lblConvertSpeedVectors);

    JRadioButton cellValuesButton = new JRadioButton("cell values");
    cellValuesButton.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( cellValuesButton.isSelected() )
        {
          m_moVec3SetI.setVec3Set(m_vecSetPerCell);
          m_dataBinding.dataBinding.setValue(MoDataBinding.DataBinding.PER_CELL);
        }
        else
        {
          m_moVec3SetI.setVec3Set(m_vecSetPerNode);
          m_dataBinding.dataBinding.setValue(MoDataBinding.DataBinding.PER_NODE);
        }
      }
    });
    GridBagConstraints gbc_rdbtnCellValues = new GridBagConstraints();
    gbc_rdbtnCellValues.anchor = GridBagConstraints.WEST;
    gbc_rdbtnCellValues.gridwidth = 2;
    gbc_rdbtnCellValues.insets = new Insets(0, 0, 10, 5);
    gbc_rdbtnCellValues.gridx = 1;
    gbc_rdbtnCellValues.gridy = 3;
    valuePanel.add(cellValuesButton, gbc_rdbtnCellValues);

    JRadioButton nodeValuesButton = new JRadioButton("node values");
    nodeValuesButton.setSelected(true);
    GridBagConstraints gbc_rdbtnNodeValues = new GridBagConstraints();
    gbc_rdbtnNodeValues.anchor = GridBagConstraints.WEST;
    gbc_rdbtnNodeValues.insets = new Insets(0, 0, 10, 5);
    gbc_rdbtnNodeValues.gridx = 3;
    gbc_rdbtnNodeValues.gridy = 3;
    gbc_rdbtnNodeValues.weightx = 1;
    gbc_rdbtnNodeValues.gridwidth = GridBagConstraints.REMAINDER;
    valuePanel.add(nodeValuesButton, gbc_rdbtnNodeValues);

    ButtonGroup speedGroup = new ButtonGroup();
    speedGroup.add(cellValuesButton);
    speedGroup.add(nodeValuesButton);

    // Mesh
    JLabel label = new JLabel("Mesh       ");
    String[] items = new String[] { "Hexahedron", "Hexahedron IJK", "Vertex Hexahedron IJK" };
    final JComboBox<String> meshBox = new JComboBox<>(items);
    meshBox.setSelectedIndex(0);
    meshBox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        switch ( meshBox.getSelectedIndex() )
        {
        case 0 : // hexahedron
          m_moMesh.setMesh(m_meshHexa);
          cellValuesButton.setSelected(m_dataBinding.dataBinding
              .getValue(MoDataBinding.DataBinding.class) == MoDataBinding.DataBinding.PER_CELL);
          cellValuesButton.setEnabled(true);
          nodeValuesButton.setEnabled(true);
          break;
        case 1 : // hexa IJK
          if ( m_meshHexaIjk == null )
            m_meshHexaIjk = m_meshBuilder.getHexahedronMeshIjk(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, MIN, MAX);
          m_moMesh.setMesh(m_meshHexaIjk);
          cellValuesButton.setSelected(m_dataBinding.dataBinding
              .getValue(MoDataBinding.DataBinding.class) == MoDataBinding.DataBinding.PER_CELL);
          cellValuesButton.setEnabled(true);
          nodeValuesButton.setEnabled(true);
          break;
        case 2 : // vertex hexa IJK
          if ( m_meshVertexHexaIjk == null )
          {
            m_meshVertexHexaIjk =
                m_meshBuilder.getVertexHexahedronMeshIjk(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, MIN, MAX);
          }
          m_moMesh.setMesh(m_meshVertexHexaIjk);
          cellValuesButton.setSelected(true);
          cellValuesButton.setEnabled(false);
          nodeValuesButton.setEnabled(false);
          break;
        default:
          break;
        }
      }
    });

    JPanel meshPanel = new JPanel();
    meshPanel.setLayout(new BoxLayout(meshPanel, BoxLayout.X_AXIS));
    meshPanel.add(label);
    meshPanel.add(meshBox);

    JPanel panel2 = new JPanel(new BorderLayout());
    panel2.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.LOWERED, null, null));
    panel2.add(valuePanel, BorderLayout.NORTH);
    panel2.add(meshPanel, BorderLayout.SOUTH);

    final Component component = m_renderArea.getComponent();
    component.setPreferredSize(new Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(panel2, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    m_renderArea.dispose();
  }

  private SoNode buildSceneGraph()
  {
    m_vecSetPerNode = new MyVec3Set(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, DataBinding.PER_NODE);
    SpeedI speedI = new SpeedI(m_vecSetPerNode);

    m_vecSetPerCell = new MyVec3Set(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, DataBinding.PER_CELL);
    SpeedIjk speedIjk = new SpeedIjk(m_vecSetPerCell);

    SoShapeHints sh = new SoShapeHints();
    sh.vertexOrdering.setValue(SoShapeHints.VertexOrderings.CLOCKWISE);

    m_moMesh = new MoMesh();
    m_moMesh.setMesh(m_meshHexa);

    MoPredefinedColorMapping colMap = new MoPredefinedColorMapping();
    colMap.minValue.setValue((float) speedI.getMin());
    colMap.maxValue.setValue((float) speedI.getMax());
    colMap.predefColorMap.setValue(MoPredefinedColorMapping.PredefColorMapping.STANDARD);

    m_dataBinding = new MoDataBinding();
    m_dataBinding.dataBinding.setValue(MoDataBinding.DataBinding.PER_NODE);

    m_moVec3SetI = new MoVec3SetI();
    m_moVec3SetI.setVec3Set(m_vecSetPerNode);

    MoVec3SetIjk moVec3SetIjk = new MoVec3SetIjk();
    moVec3SetIjk.setVec3Set(m_vecSetPerCell);

    // Scalar field
    MoScalarSetI moScalarSetI = new MoScalarSetI();
    moScalarSetI.setScalarSet(speedI);

    MoScalarSetIjk moScalarSetIjk = new MoScalarSetIjk();
    moScalarSetIjk.setScalarSet(speedIjk);

    SbVec3f sbCenter =
        new SbVec3f((float) (MAX[0] - MIN[0]) / 2.f, (float) (MAX[1] - MIN[1]) / 2.f, (float) (MAX[2] - MIN[2]) / 2.f);
    float scaleFactor = (float) Math.max(Math.max(MAX[0] - MIN[0], MAX[1] - MIN[1]), MAX[2] - MIN[2]);
    scaleFactor /= 10;

    SoJackDragger dragger = new SoJackDragger();
    dragger.translation.setValue(sbCenter);
    dragger.rotation.setValue(new SbRotation(new SbVec3f(0, 1, 0), new SbVec3f(1, 0, 0)));
    dragger.scaleFactor.setValue(scaleFactor, scaleFactor, scaleFactor);
    dragger.addMotionCallback(new MotionCallback(), this);

    m_envNode = new SoEnvironment();
    m_envNode.ambientOcclusion.setValue(true);

    m_lightModel = new SoLightModel();
    m_lightModel.model.setValue(SoLightModel.Models.PER_VERTEX_PHONG);

    m_streamline = new MoMeshStreamline();
    m_streamline.integrationStepLengthFactor.setValue(1);
    m_streamline.colorScalarSetId.setValue(0);
    m_streamline.vec3SetId.setValue(0);
    moveStreamSource(dragger);

    SoSeparator streamlineSep = new SoSeparator();
    {
      streamlineSep.addChild(m_envNode);
      streamlineSep.addChild(m_lightModel);
      streamlineSep.addChild(dragger);
      streamlineSep.addChild(m_streamline);
    }

    // Transparent Mesh Skin
    SoPickStyle pickStyle = new SoPickStyle();
    pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);

    MoDrawStyle drawStyle = new MoDrawStyle();
    drawStyle.displayEdges.setValue(true);
    drawStyle.displayFaces.setValue(false);

    MoMeshSkin skin = new MoMeshSkin();
    skin.colorScalarSetId.setValue(0);

    // This mesh is used to connect the streamlines generated mesh as input of
    // the vectors
    MoMesh lineMesh = new MoMesh();
    lineMesh.connectFrom(m_streamline);

    m_vectorSwitch = new SoSwitch();
    MoMeshVector vector = new MoMeshVector();
    vector.scaleFactor.setValue(7);
    vector.thicknessFactor.setValue(0.1f);
    vector.colorScalarSetId.setValue(-1);

    SoSeparator root = new SoSeparator();
    {
      m_vectorSwitch.addChild(vector);

      root.addChild(sh);
      root.addChild(m_moMesh);
      root.addChild(colMap);
      root.addChild(m_dataBinding);
      root.addChild(m_moVec3SetI);
      root.addChild(moVec3SetIjk);
      root.addChild(moScalarSetI);
      root.addChild(moScalarSetIjk);

      root.addChild(streamlineSep);

      root.addChild(pickStyle);
      root.addChild(drawStyle);
      root.addChild(skin);

      root.addChild(lineMesh);
      root.addChild(m_vectorSwitch);
    }

    return root;
  }

  private void moveStreamSource(SoJackDragger dragger)
  {
    // get the dragger position
    SbVec3f dragerPos = dragger.translation.getValue();

    // get the dragger scale factor
    SbVec3f drageScale = dragger.scaleFactor.getValue();

    // get the dragger rotation
    SbRotation rotation = dragger.rotation.getValue();

    // move stream lines, stream points & stream spheres
    double alpha = 0;
    double d_alpha = 2 * Math.PI / NUM_START_POINTS;

    SbMatrix m = new SbMatrix();
    m.setTransform(dragerPos, rotation, drageScale);

    for ( int i = 0; i < NUM_START_POINTS; i++, alpha += d_alpha )
    {
      m_startPoints[i].setValue((float) (RADIUS_SRC_CIRCLE * Math.cos(alpha)), 0,
          (float) (RADIUS_SRC_CIRCLE * Math.sin(alpha)));
      m_startPoints[i] = m.multVecMatrix(m_startPoints[i]);
    }
    m_streamline.startingPoints.setValues(0, m_startPoints);
  }

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain("Streamlines Example");
  }
}
