package meshvizxlm.mapping.vectors;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;

import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoShapeHints;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;
import com.openinventor.meshvizxlm.mapping.nodes.*;
import com.openinventor.meshvizxlm.mesh.cell.MiCellFilterIjk;
import com.openinventor.meshvizxlm.mesh.data.DataBinding;

import meshvizxlm.mapping.MyVec3Set;
import meshvizxlm.mapping.MyVec3Set.Magnitude;
import meshvizxlm.mesh.MbSampleMeshBuilder;
import meshvizxlm.mesh.volumes.MbHexahedronMeshIjk;
import meshvizxlm.mesh.volumes.MbVertexHexahedronMeshIjk;
import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{

  class CellFilterIjk implements MiCellFilterIjk
  {
    @Override
    public boolean acceptCell(int i, int j, int k)
    {
      if ( NUM_CELL_I / 3. < i && i <= 2 * NUM_CELL_I / 3. && NUM_CELL_J / 3. < j && j <= 2 * NUM_CELL_J / 3. )
        return false;

      return true;
    }

    @Override
    public long getTimeStamp()
    {
      return 0;
    }
  }

  private static final java.awt.Dimension SLIDER_TEXT_DIM = new java.awt.Dimension(60, 20);
  private static final int NUM_CELL_I = 10;
  private static final int NUM_CELL_J = 10;
  private static final int NUM_CELL_K = 10;

  private IViewerExaminer m_renderArea;
  private MbSampleMeshBuilder m_meshBuilder;
  private MbVertexHexahedronMeshIjk m_meshVertexHexaIjk;
  private MbHexahedronMeshIjk m_meshHexaIjk;
  private MoMesh m_moMesh;
  private MoPredefinedColorMapping m_colMap;
  private MoMeshVector m_vectors;
  private SoSwitch m_cellFilterSwitch;
  private MoCellRanges m_cellRanges;
  private MoDataBinding m_moDataBinding;

  public Main()
  {
    m_meshBuilder = new MbSampleMeshBuilder();

    double[] min = { 0, 0, 0 };
    double[] max = { NUM_CELL_I, NUM_CELL_J, NUM_CELL_K };

    m_meshVertexHexaIjk = m_meshBuilder.getVertexHexahedronMeshIjk(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, min, max);
    m_meshHexaIjk = null;
  }

  @Override
  public void start()
  {
    SoSeparator root = buildSceneGraph();

    m_renderArea = ViewerComponentsFactory.createViewerExaminer();
    m_renderArea.setSceneGraph(root);
    m_renderArea.viewAll();

    // SWING part
    final Component component = m_renderArea.getComponent();
    component.setPreferredSize(new Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(buildOptionsPanel(), BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    m_renderArea.dispose();
  }

  private SoSeparator buildSceneGraph()
  {
    SoShapeHints sh = new SoShapeHints();
    sh.vertexOrdering.setValue(SoShapeHints.VertexOrderings.CLOCKWISE);

    m_moMesh = new MoMesh();
    m_moMesh.setMesh(m_meshVertexHexaIjk);

    m_moDataBinding = new MoDataBinding();
    m_moDataBinding.dataBinding.setValue(MoDataBinding.DataBinding.PER_CELL);

    // Vector field
    MyVec3Set vecSetPerNode = new MyVec3Set(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, DataBinding.PER_NODE);
    vecSetPerNode.setName("MyVec3Set_per_node");
    Magnitude magnitudePerNode = vecSetPerNode.new Magnitude();

    MyVec3Set vecSetPerCell = new MyVec3Set(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, DataBinding.PER_CELL);
    vecSetPerCell.setName("MyVec3SetI_per_cell");
    Magnitude magnitudePerCell = vecSetPerCell.new Magnitude();

    MoVec3SetI moVec3SetI = new MoVec3SetI();
    moVec3SetI.setVec3Set(vecSetPerNode);

    MoVec3SetIjk moVec3SetIjk = new MoVec3SetIjk();
    moVec3SetIjk.setVec3Set(vecSetPerCell);

    // Color set
    MoScalarSetI moColorSetI = new MoScalarSetI();
    moColorSetI.setScalarSet(magnitudePerNode);

    MoScalarSetIjk moColorSetIjk = new MoScalarSetIjk();
    moColorSetIjk.setScalarSet(magnitudePerCell);

    // Color map
    m_colMap = new MoPredefinedColorMapping();
    m_colMap.minValue.setValue((float) magnitudePerCell.getMin());
    m_colMap.maxValue.setValue((float) magnitudePerCell.getMax());
    m_colMap.predefColorMap.setValue(MoPredefinedColorMapping.PredefColorMapping.STANDARD);

    m_cellRanges = new MoCellRanges();
    m_cellRanges.min.setValues(0, new long[] { 0, 0, 0 });
    m_cellRanges.max.setValues(0, new long[] { NUM_CELL_I - 1, NUM_CELL_J - 1, NUM_CELL_K - 1 });

    // Cell filter
    MoCellFilter cellFilter = new MoCellFilter();
    cellFilter.setCellFilter(new CellFilterIjk());
    m_cellFilterSwitch = new SoSwitch();
    m_cellFilterSwitch.addChild(cellFilter);
    m_cellFilterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);

    m_vectors = new MoMeshVector();
    m_vectors.vectorSetId.setValue(0);
    m_vectors.colorScalarSetId.setValue(0);
    m_vectors.scaleFactor.setValue(1.f);
    m_vectors.shiftFactor.setValue(0.f);
    m_vectors.thicknessFactor.setValue(0.5f);

    SoSeparator root = new SoSeparator();
    {
      root.addChild(m_moMesh);
      root.addChild(m_moDataBinding);
      root.addChild(moVec3SetI);
      root.addChild(moVec3SetIjk);
      root.addChild(moColorSetI);
      root.addChild(moColorSetIjk);
      root.addChild(m_cellRanges);
      root.addChild(m_cellFilterSwitch);
      root.addChild(m_colMap);
      root.addChild(m_vectors);
    }

    return root;
  }

  private JPanel buildOptionsPanel()
  {
    JPanel vectorOptionsPanel = new JPanel();
    vectorOptionsPanel
        .setBorder(new TitledBorder(new EtchedBorder(), "Vectors", TitledBorder.LEADING, TitledBorder.TOP, null, null));
    vectorOptionsPanel.setLayout(new GridLayout(0, 1, 0, 10));

    JPanel panel = new JPanel();
    panel.setLayout(new BorderLayout());
    final JCheckBox arrowsBox = new JCheckBox("Display arrows");
    arrowsBox.setSelected(m_vectors.arrow.getValue());
    arrowsBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        m_vectors.arrow.setValue(arrowsBox.isSelected());
      }
    });
    panel.add(arrowsBox, BorderLayout.WEST);

    JPanel bindingPanel = new JPanel();
    bindingPanel.setLayout(new BorderLayout());
    JLabel bindingLabel = new JLabel("    Data binding: ");
    bindingPanel.add(bindingLabel, BorderLayout.WEST);

    String[] bindingItems = new String[] { "AUTO", "PER_CELL", "PER_NODE" };
    final JComboBox<String> bindingBox = new JComboBox<>(bindingItems);
    bindingBox.setSelectedItem(m_moDataBinding.dataBinding.getValue(MoDataBinding.DataBinding.class).name());
    bindingBox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        switch ( bindingBox.getSelectedIndex() )
        {
        case 0 : // AUTO
          m_moDataBinding.dataBinding.setValue(MoDataBinding.DataBinding.AUTO);
          break;
        case 1 : // PER_CELL
          m_moDataBinding.dataBinding.setValue(MoDataBinding.DataBinding.PER_CELL);
          break;
        case 2 : // PER_NODE
          m_moDataBinding.dataBinding.setValue(MoDataBinding.DataBinding.PER_NODE);
          break;
        default:
          break;
        }
      }
    });
    bindingPanel.add(bindingBox);
    panel.add(bindingPanel, BorderLayout.CENTER);
    vectorOptionsPanel.add(panel);

    float initFactor = m_vectors.scaleFactor.getValue();
    final SliderPanel scaleSliderPanel = new SliderPanel(0.f, 2 * initFactor, initFactor, 2);
    ((FlowLayout) scaleSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    scaleSliderPanel.addInfoText("Scale factor");
    scaleSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_vectors.scaleFactor.setValue(value);
      }
    });
    vectorOptionsPanel.add(scaleSliderPanel);

    initFactor = m_vectors.shiftFactor.getValue();
    final SliderPanel shiftSliderPanel = new SliderPanel(-1.f, 0.f, initFactor, 2);
    ((FlowLayout) shiftSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    shiftSliderPanel.addInfoText("Shift factor");
    shiftSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_vectors.shiftFactor.setValue(value);
      }
    });
    vectorOptionsPanel.add(shiftSliderPanel);

    initFactor = m_vectors.thicknessFactor.getValue();
    final SliderPanel thicknessSliderPanel = new SliderPanel(0.f, 1.f, initFactor, 2);
    ((FlowLayout) thicknessSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    thicknessSliderPanel.addInfoText("Thickness factor");
    thicknessSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_vectors.thicknessFactor.setValue(value);
      }
    });
    vectorOptionsPanel.add(thicknessSliderPanel);

    JPanel cellOptionsPanel = new JPanel();
    cellOptionsPanel.setBorder(new TitledBorder(new EtchedBorder(), "Cell range and filter", TitledBorder.LEADING,
        TitledBorder.TOP, null, null));
    cellOptionsPanel.setLayout(new GridLayout(0, 1, 0, 0));

    final JCheckBox filterBox = new JCheckBox("Cell filter");
    filterBox.setSelected(m_cellFilterSwitch.whichChild.getValue() == SoSwitch.SO_SWITCH_ALL);
    filterBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( filterBox.isSelected() )
          m_cellFilterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
        else
          m_cellFilterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
      }
    });
    cellOptionsPanel.add(filterBox);

    final SliderPanel iMinSliderPanel = new SliderPanel(0, NUM_CELL_I - 1, (int) m_cellRanges.min.getValueAt(0));
    ((FlowLayout) iMinSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    iMinSliderPanel.addInfoText("I min");
    iMinSliderPanel.setInfoTextSize(SLIDER_TEXT_DIM);
    iMinSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_cellRanges.min.set1Value(0, (long) value);
      }
    });
    cellOptionsPanel.add(iMinSliderPanel);

    final SliderPanel iMaxSliderPanel = new SliderPanel(0, NUM_CELL_I - 1, (int) m_cellRanges.max.getValueAt(0));
    ((FlowLayout) iMaxSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    iMaxSliderPanel.addInfoText("I max");
    iMaxSliderPanel.setInfoTextSize(SLIDER_TEXT_DIM);
    iMaxSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_cellRanges.max.set1Value(0, (long) value);
      }
    });
    cellOptionsPanel.add(iMaxSliderPanel);

    final SliderPanel jMinSliderPanel = new SliderPanel(0, NUM_CELL_J - 1, (int) m_cellRanges.min.getValueAt(1));
    ((FlowLayout) jMinSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    jMinSliderPanel.addInfoText("J min");
    jMinSliderPanel.setInfoTextSize(SLIDER_TEXT_DIM);
    jMinSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_cellRanges.min.set1Value(1, (long) value);
      }
    });
    cellOptionsPanel.add(jMinSliderPanel);

    final SliderPanel jMaxSliderPanel = new SliderPanel(0, NUM_CELL_J - 1, (int) m_cellRanges.max.getValueAt(1));
    ((FlowLayout) jMaxSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    jMaxSliderPanel.addInfoText("J max");
    jMaxSliderPanel.setInfoTextSize(SLIDER_TEXT_DIM);
    jMaxSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_cellRanges.max.set1Value(1, (long) value);
      }
    });
    cellOptionsPanel.add(jMaxSliderPanel);

    final SliderPanel kMinSliderPanel = new SliderPanel(0, NUM_CELL_K - 1, (int) m_cellRanges.min.getValueAt(2));
    ((FlowLayout) kMinSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    kMinSliderPanel.addInfoText("K min");
    kMinSliderPanel.setInfoTextSize(SLIDER_TEXT_DIM);
    kMinSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_cellRanges.min.set1Value(2, (long) value);
      }
    });
    cellOptionsPanel.add(kMinSliderPanel);

    final SliderPanel kMaxSliderPanel = new SliderPanel(0, NUM_CELL_K - 1, (int) m_cellRanges.max.getValueAt(2));
    ((FlowLayout) kMaxSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    kMaxSliderPanel.addInfoText("K max");
    kMaxSliderPanel.setInfoTextSize(SLIDER_TEXT_DIM);
    kMaxSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_cellRanges.max.set1Value(2, (long) value);
      }
    });
    cellOptionsPanel.add(kMaxSliderPanel);

    JPanel meshPanel = new JPanel();
    meshPanel
        .setBorder(new TitledBorder(new EtchedBorder(), "Mesh", TitledBorder.LEADING, TitledBorder.TOP, null, null));
    meshPanel.setLayout(new BorderLayout());
    String[] items = new String[] { "Hexahedron IJK", "Vertex Hexahedron IJK" };
    final JComboBox<String> meshBox = new JComboBox<>(items);
    meshBox.setSelectedIndex(1);
    meshPanel.add(meshBox, BorderLayout.CENTER);
    meshBox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        switch ( meshBox.getSelectedIndex() )
        {
        case 0 : // Hexahedron IJK
          if ( m_meshHexaIjk == null )
          {
            double[] min = { 0, 0, 0 };
            double[] max = { NUM_CELL_I, NUM_CELL_J, NUM_CELL_K };
            m_meshHexaIjk = m_meshBuilder.getHexahedronMeshIjk(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, min, max);
          }
          m_moMesh.setMesh(m_meshHexaIjk);
          break;
        case 1 : // Vertex hexahedron IJK
          m_moMesh.setMesh(m_meshVertexHexaIjk);
          break;
        default:
          break;
        }
      }
    });

    JPanel leftPanel = new JPanel(new BorderLayout());
    leftPanel.add(vectorOptionsPanel, BorderLayout.CENTER);
    leftPanel.add(meshPanel, BorderLayout.SOUTH);

    JPanel optionsPanel = new JPanel(new BorderLayout());
    optionsPanel.add(leftPanel, BorderLayout.WEST);
    optionsPanel.add(cellOptionsPanel, BorderLayout.CENTER);

    return optionsPanel;
  }

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("Vectors Example");
  }
}
