package meshvizxlm.mesh;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

import meshvizxlm.mesh.data.MbScalarSetI;
import meshvizxlm.mesh.data.MbScalarSetIj;
import meshvizxlm.mesh.data.MbScalarSetIjk;
import meshvizxlm.mesh.data.MbVec3SetI;
import meshvizxlm.mesh.data.MbVec3SetIj;
import meshvizxlm.mesh.data.MbVec3SetIjk;
import meshvizxlm.mesh.surfaces.MbSurfaceMeshCurvilinear;
import meshvizxlm.mesh.surfaces.MbSurfaceMeshQuadrangle;
import meshvizxlm.mesh.volumes.MbHexahedronMeshIjk;
import meshvizxlm.mesh.volumes.MbVertexHexahedronMeshIjk;
import meshvizxlm.mesh.volumes.MbVolumeMeshHexahedron;
import meshvizxlm.mesh.volumes.MbVolumeMeshPyramid;
import meshvizxlm.mesh.volumes.MbVolumeMeshTetrahedron;
import meshvizxlm.mesh.volumes.MbVolumeMeshWedge;

import com.openinventor.meshvizxlm.mesh.StorageLayoutIJK;
import com.openinventor.meshvizxlm.mesh.data.DataBinding;
import com.openinventor.meshvizxlm.mesh.geometry.MiGeometryI;
import com.openinventor.meshvizxlm.mesh.topology.MiHexahedronTopologyExplicitIjk;

/**
 * Simple functor template used to compute scalars sets on the fly.
 *
 */
interface MbScalarFunctor
{
  public double apply(double x, double y, double z);
}

/**
 * Utility class that creates synthetic meshes for testing purpose.
 *
 */
public class MbSampleMeshBuilder
{

  private List<MbScalarFunctor> m_scalarFctors;
  private List<String> m_scalarFctNames;
  private MbDeadCellFctorI m_deadCellFctI;
  private MbDeadCellFctorIj m_deadCellFctIj;
  private MbDeadCellFctorIjk m_deadCellFctIjk;

  public MbSampleMeshBuilder()
  {
    m_scalarFctors = new ArrayList<MbScalarFunctor>();
    m_scalarFctNames = new ArrayList<String>();

    m_scalarFctors.add(new XFctor());
    m_scalarFctNames.add("$ScalarSetX");
    m_scalarFctors.add(new SphereFctor());
    m_scalarFctNames.add("$Sphere");

    m_deadCellFctI = null;
    m_deadCellFctIj = null;
    m_deadCellFctIjk = null;
  }

  /**
   * add function to set dead cells when building unstructured mesh
   */
  public void addDeadCellFctor(MbDeadCellFctorI fctor)
  {
    m_deadCellFctI = fctor;
  }

  /**
   * add function to set dead cells when building structured surface mesh
   */
  public void addDeadCellFctor(MbDeadCellFctorIj fctor)
  {
    m_deadCellFctIj = fctor;
  }

  /**
   * add function to set dead cells when building structured volume mesh
   */
  public void addDeadCellFctor(MbDeadCellFctorIjk fctor)
  {
    m_deadCellFctIjk = fctor;
  }

  public MbSurfaceMeshQuadrangle getSurfaceMeshQuadrangle(int numCellI, int numCellJ, double[] origin,
      double[] stepVecI, double[] stepVecJ, MbDeadCellFctorI dCellFctor)
  {
    final int numCells = numCellI * numCellJ;
    final int numN = (numCellI + 1) * (numCellJ + 1);

    double[] xArray = new double[numN];
    double[] yArray = new double[numN];
    double[] zArray = new double[numN];

    double[] xValues = new double[numN];
    double[] sphereValues = new double[numN];

    int i, j;

    for ( int n = 0; n < numN; n++ )
    {
      i = n % (numCellI + 1);
      j = n / (numCellI + 1);

      double x = origin[0] + i * stepVecI[0] + j * stepVecJ[0];
      double y = origin[1] + i * stepVecI[1] + j * stepVecJ[1];
      double z = origin[2] + i * stepVecI[2] + j * stepVecJ[2];

      xArray[n] = x;
      yArray[n] = y;
      zArray[n] = z;

      xValues[n] = x;
      sphereValues[n] = x * x + y * y + z * z;
    }

    double[] cellIds = new double[numCells];
    boolean[] deadCells = new boolean[numCells];
    int[] indices = new int[numCells * 4];
    int tmp;
    int iInd = 0;

    for ( int c = 0; c < numCells; c++ )
    {
      // get I, for cellId
      i = c % numCellI;
      j = c / numCellI;
      tmp = j * (numCellI + 1) + i;
      indices[iInd++] = tmp;
      indices[iInd++] = tmp + 1;
      indices[iInd++] = tmp + numCellI + 2;
      indices[iInd++] = tmp + numCellI + 1;

      cellIds[c] = c;
      if ( dCellFctor != null )
        deadCells[c] = dCellFctor.apply(c);
    }

    MbSurfaceMeshQuadrangle mesh = new MbSurfaceMeshQuadrangle(xArray, yArray, zArray);
    mesh.setTopology(indices);

    mesh.addScalarSet(new MbScalarSetI(cellIds, "$CellId", DataBinding.PER_CELL));
    mesh.addScalarSet(new MbScalarSetI(sphereValues, "$Sphere", DataBinding.PER_NODE));
    mesh.addScalarSet(new MbScalarSetI(xValues, "$ScalarSetX", DataBinding.PER_NODE));

    MbVec3SetI vecSet = new MbVec3SetI(xArray, yArray, zArray);
    vecSet.setName("$MyGeometry");
    mesh.addVec3Set(vecSet);

    if ( dCellFctor != null )
      mesh.setDeadCells(deadCells);

    return mesh;
  }

  public MbSurfaceMeshCurvilinear getSurfaceMeshCurvilinear(int numCellI, int numCellJ, double[] origin,
      double[] stepVecI, double[] stepVecJ)
  {
    final int numN = (numCellI + 1) * (numCellJ + 1);

    double[] xArray = new double[numN];
    double[] yArray = new double[numN];
    double[] zArray = new double[numN];

    MbScalarSetIj sphere = new MbScalarSetIj(numCellI + 1, numCellJ + 1, "$Sphere", DataBinding.PER_NODE);
    MbScalarSetIj scalarSetX = new MbScalarSetIj(numCellI + 1, numCellJ + 1, "$ScalarSetX", DataBinding.PER_NODE);
    MbVec3SetIj vecSet = new MbVec3SetIj(numCellI + 1, numCellJ + 1, "$MyGeometry", DataBinding.PER_NODE);

    int i, j;
    int n = 0;

    for ( j = 0; j < numCellJ + 1; j++ )
      for ( i = 0; i < numCellI + 1; i++ )
      {
        double x = origin[0] + i * stepVecI[0] + j * stepVecJ[0];
        double y = origin[1] + i * stepVecI[1] + j * stepVecJ[1];
        double z = origin[2] + i * stepVecI[2] + j * stepVecJ[2];

        xArray[n] = x;
        yArray[n] = y;
        zArray[n] = z;
        n++;

        scalarSetX.set(i, j, x);
        sphere.set(i, j, x * x + y * y + z * z);
        vecSet.set(i, j, x, y, z);
      }

    MbSurfaceMeshCurvilinear mesh = new MbSurfaceMeshCurvilinear(numCellI, numCellJ);
    mesh.setGeometry(xArray, yArray, zArray);

    mesh.addScalarSetIj(sphere);
    mesh.addScalarSetIj(scalarSetX);
    mesh.addVec3SetIj(vecSet);

    MbScalarSetIj cellIds = new MbScalarSetIj(numCellI, numCellJ, "$CellId", DataBinding.PER_CELL);

    int c = 0;
    for ( j = 0; j < numCellJ; j++ )
      for ( i = 0; i < numCellI; i++ )
      {
        cellIds.set(i, j, c);
        c++;
        if ( m_deadCellFctIj != null )
          mesh.setDeadCell(i, j, m_deadCellFctIj.apply(i, j));
      }

    mesh.addScalarSetIj(cellIds);

    return mesh;
  }

  /**
   * Returns a non indexed hexahedron IJK mesh.
   */
  public MbVertexHexahedronMeshIjk getVertexHexahedronMeshIjk(int dimX, int dimY, int dimZ, double[] min, double[] max)
  {
    return getVertexHexahedronMeshIjk(dimX, dimY, dimZ, min, max, StorageLayoutIJK.KJI);
  }

  /**
   * Returns a non indexed hexahedron IJK mesh.
   */
  public MbVertexHexahedronMeshIjk getVertexHexahedronMeshIjk(int dimX, int dimY, int dimZ, double[] min, double[] max,
      StorageLayoutIJK storageLayout)
  {
    MbHexahedronMeshIjk hexaMesh = getHexahedronMeshIjk(dimX, dimY, dimZ, min, max, storageLayout);
    return getVertexHexahedronMeshIjk(hexaMesh);
  }

  /**
   * Returns a non indexed hexahedron IJK mesh built from an indexed mesh.
   */
  public MbVertexHexahedronMeshIjk getVertexHexahedronMeshIjk(MbHexahedronMeshIjk hexaMesh)
  {
    MbVertexHexahedronMeshIjk mesh = new MbVertexHexahedronMeshIjk(hexaMesh);

    mesh.addScalarSetIjk(hexaMesh.getScalarSetIjk("$CellId"));

    MbVec3SetIjk vecIjk = hexaMesh.getVec3SetIjk("$MyGeometry");
    mesh.addVec3SetIjk(vecIjk);

    MbVec3SetIjk vecIjk2 = hexaMesh.getVec3SetIjk("$CellCenter");
    mesh.addVec3SetIjk(vecIjk2);

    SphereFctor functor = new SphereFctor();
    int dimX = hexaMesh.getTopology().getNumCellsI();
    int dimY = hexaMesh.getTopology().getNumCellsJ();
    int dimZ = hexaMesh.getTopology().getNumCellsK();
    MbScalarSetIjk sphere = new MbScalarSetIjk(dimX, dimY, dimZ, "$Sphere", DataBinding.PER_CELL);
    double[] vec;
    for ( int k = 0; k < dimZ; k++ )
      for ( int j = 0; j < dimY; j++ )
        for ( int i = 0; i < dimX; i++ )
        {
          vec = vecIjk.get(i, j, k);
          sphere.set(i, j, k, functor.apply(vec[0], vec[1], vec[2]));
        }

    mesh.addScalarSetIjk(sphere);
    return mesh;
  }

  /**
   * Returns an hexahedron IJK mesh.
   */
  public MbHexahedronMeshIjk getHexahedronMeshIjk(int dimX, int dimY, int dimZ, double[] min, double[] max)
  {
    return getHexahedronMeshIjk(dimX, dimY, dimZ, min, max, StorageLayoutIJK.KJI);
  }

  /**
   * Returns an hexahedron IJK mesh.
   */
  public MbHexahedronMeshIjk getHexahedronMeshIjk(int dimX, int dimY, int dimZ, double[] min, double[] max,
      StorageLayoutIJK storageLayout)
  {
    int[] nbFaults = { 0, 0, 0 };
    return getHexahedronMeshIjk(dimX, dimY, dimZ, min, max, nbFaults, null, storageLayout);
  }

  public MbHexahedronMeshIjk getHexahedronMeshIjk(int dimX, int dimY, int dimZ, double[] min, double[] max,
      int[] nbFaults, double[] rotations)
  {
    return getHexahedronMeshIjk(dimX, dimY, dimZ, min, max, nbFaults, rotations, StorageLayoutIJK.KJI);
  }

  /**
   * Returns a pillar mesh as a hexahedron IJK mesh. nbFaults is the number of
   * faults along each axis. rotations are the angle to rotate the mesh around
   * the line passing through its center and parallel to each axis
   */
  public MbHexahedronMeshIjk getHexahedronMeshIjk(int dimX, int dimY, int dimZ, double[] min, double[] max,
      int[] nbFaults, double[] rotations, StorageLayoutIJK storageLayout)
  {
    int ni = dimX + 1;
    int nj = dimY + 1;
    int nk = dimZ + 1;

    double[] step = { (max[0] - min[0]) / dimX, (max[1] - min[1]) / dimY, (max[2] - min[2]) / dimZ };
    int[] faultStep = { ni / (nbFaults[0] + 1) + 1, nj / (nbFaults[1] + 1) + 1, nk / (nbFaults[2] + 1) + 1 };
    double[] faultShifts = { 0.5 * step[0], 0.5 * step[1], 0.5 * step[2] };

    int numCoords = ni * nj * nk;
    double[] xArray = new double[numCoords];
    double[] yArray = new double[numCoords];
    double[] zArray = new double[numCoords];

    List<Double> xFaultList = new ArrayList<Double>();
    List<Double> yFaultList = new ArrayList<Double>();
    List<Double> zFaultList = new ArrayList<Double>();
    HashMap<Integer, Integer> faultCoords = new HashMap<Integer, Integer>();

    MbIjkToI adaptor = new MbIjkToI(ni, nj, nk, storageLayout);
    int coordIndex;
    boolean hasFault = false;

    for ( int k = 0; k < nk; k++ )
    {
      for ( int j = 0; j < nj; j++ )
      {
        for ( int i = 0; i < ni; i++ )
        {
          double x = min[0] + step[0] * i;
          double y = min[1] + step[1] * j;
          double z = min[2] + step[2] * k;

          coordIndex = (int) adaptor.getI(i, j, k);
          xArray[coordIndex] = x;
          yArray[coordIndex] = y;
          zArray[coordIndex] = z;

          if ( (i > 0 && i % faultStep[0] == 0) || (j > 0 && j % faultStep[1] == 0) )
          {
            z += faultShifts[2];
            hasFault = true;
          }
          if ( (k > 0 && k % faultStep[2] == 0) )
          {
            x += faultShifts[0];
            hasFault = true;
          }
          if ( hasFault )
          {
            faultCoords.put(coordIndex, numCoords + xFaultList.size());
            xFaultList.add(x);
            yFaultList.add(y);
            zFaultList.add(z);
            hasFault = false;
          }
        }
      }
    }

    final int nbFaultCoord = xFaultList.size();
    if ( nbFaultCoord > 0 )
    {
      final int oldSize = numCoords;
      numCoords += nbFaultCoord;
      xArray = Arrays.copyOf(xArray, numCoords);
      yArray = Arrays.copyOf(yArray, numCoords);
      zArray = Arrays.copyOf(zArray, numCoords);

      for ( int i = 0; i < nbFaultCoord; ++i )
      {
        xArray[i + oldSize] = xFaultList.get(i);
        yArray[i + oldSize] = yFaultList.get(i);
        zArray[i + oldSize] = zFaultList.get(i);
      }
    }

    if ( rotations != null )
    {
      double[] tmp = { (max[0] - min[0]) / 2., (max[1] - min[1]) / 2., (max[2] - min[2]) / 2. };
      for ( int a = 0; a < 3; ++a )
        rotate(tmp, rotations[a], a, xArray, yArray, zArray);
    }

    MbHexahedronMeshIjk mesh = new MbHexahedronMeshIjk(dimX, dimY, dimZ, xArray, yArray, zArray, storageLayout);

    int[] nodes = new int[8];
    boolean fi;
    boolean fj;
    boolean fk;

    for ( int i = 0; i < dimX; i++ )
      for ( int j = 0; j < dimY; j++ )
        for ( int k = 0; k < dimZ; k++ )
        {
          nodes[0] = (int) adaptor.getI(i, j, k); // i, j, k
          nodes[1] = (int) adaptor.getI(i + 1, j, k); // i+1, j, k
          nodes[2] = (int) adaptor.getI(i + 1, j + 1, k); // i+1, j+1, k
          nodes[3] = (int) adaptor.getI(i, j + 1, k); // i, j+1, k
          nodes[4] = (int) adaptor.getI(i, j, k + 1); // i, j, k+1
          nodes[5] = (int) adaptor.getI(i + 1, j, k + 1); // i+1, j, k+1
          nodes[6] = (int) adaptor.getI(i + 1, j + 1, k + 1); // i+1, j+1, k+1
          nodes[7] = (int) adaptor.getI(i, j + 1, k + 1); // i, j+1, k+1

          fi = i > 0 && i % faultStep[0] == 0;
          fj = j > 0 && j % faultStep[1] == 0;
          fk = k > 0 && k % faultStep[2] == 0;

          if ( fi || fj || fk )
            nodes[0] = faultCoords.get(nodes[0]);
          if ( fj || fk )
            nodes[1] = faultCoords.get(nodes[1]);
          if ( fk )
            nodes[2] = faultCoords.get(nodes[2]);
          if ( fi || fk )
            nodes[3] = faultCoords.get(nodes[3]);
          if ( fi || fj )
            nodes[4] = faultCoords.get(nodes[4]);
          if ( fj )
            nodes[5] = faultCoords.get(nodes[5]);
          if ( fi )
            nodes[7] = faultCoords.get(nodes[7]);

          mesh.setCellNodeIndices(i, j, k, nodes);
        }

    faultCoords.clear();

    MbVec3SetI vec = new MbVec3SetI(xArray, yArray, zArray);
    vec.setName("$MyGeometry");
    mesh.addVec3Set(vec);

    MiHexahedronTopologyExplicitIjk topology = mesh.getTopology();
    long[] nodeIndices = null;
    MbVec3SetIjk vecIjk = new MbVec3SetIjk(dimX, dimY, dimZ, "$MyGeometry", DataBinding.PER_CELL);
    MbVec3SetIjk vecIjkCellCenter = new MbVec3SetIjk(dimX, dimY, dimZ, "$CellCenter", DataBinding.PER_CELL);
    MbScalarSetIjk scalar = new MbScalarSetIjk(dimX, dimY, dimZ, "$CellId", DataBinding.PER_CELL);
    int n = 0;
    double xSum, ySum, zSum;

    for ( int k = 0; k < dimZ; k++ )
      for ( int j = 0; j < dimY; j++ )
        for ( int i = 0; i < dimX; i++ )
        {
          scalar.set(i, j, k, n);
          n++;

          xSum = 0;
          ySum = 0;
          zSum = 0;
          nodeIndices = topology.getCellNodeIndices(i, j, k, nodeIndices);
          vecIjk.set(i, j, k, xArray[(int) nodeIndices[0]], yArray[(int) nodeIndices[0]], zArray[(int) nodeIndices[0]]);

          for ( int iNode = 0; iNode < nodeIndices.length; ++iNode )
          {
            xSum += xArray[(int) nodeIndices[iNode]];
            ySum += yArray[(int) nodeIndices[iNode]];
            zSum += zArray[(int) nodeIndices[iNode]];
          }
          vecIjkCellCenter
              .set(i, j, k, xSum / nodeIndices.length, ySum / nodeIndices.length, zSum / nodeIndices.length);

          if ( m_deadCellFctIjk != null )
            mesh.setDeadCells(i, j, k, m_deadCellFctIjk.apply(i, j, k));
        }
    mesh.addScalarSetIjk(scalar);
    mesh.addVec3SetIjk(vecIjk);
    mesh.addVec3SetIjk(vecIjkCellCenter);

    addScalarSets(mesh, numCoords);

    return mesh;
  }

  public MbVolumeMeshPyramid getMeshPyramid(int dimX, int dimY, int dimZ, double[] min, double[] max)
  {
    return getMeshPyramid(dimX, dimY, dimZ, min, max, null);
  }

  public MbVolumeMeshPyramid getMeshPyramid(int dimX, int dimY, int dimZ, double[] min, double[] max, double[] rotation)
  {
    int ni = dimX + 1;
    int nj = dimY + 1;
    int nk = dimZ + 1;

    double[] step = { (max[0] - min[0]) / dimX, (max[1] - min[1]) / dimY, (max[2] - min[2]) / dimZ };

    int numCoords = (ni - 1) * (nj - 1) * (nk - 1) + (ni * nj * nk);
    double[] xArray = new double[numCoords];
    double[] yArray = new double[numCoords];
    double[] zArray = new double[numCoords];
    int ind = 0;

    // Pyramids peak coords
    for ( int i = 0; i < ni - 1; i++ )
    {
      for ( int j = 0; j < nj - 1; j++ )
      {
        for ( int k = 0; k < nk - 1; k++, ++ind )
        {
          xArray[ind] = min[0] + step[0] * i + step[0] / 2.;
          yArray[ind] = min[1] + step[1] * j + step[1] / 2.;
          zArray[ind] = min[2] + step[2] * k + step[2] / 2.;
        }
      }
    }

    // Cubes coords
    int cubeCoordsBeginIndex = ind;
    for ( int k = 0; k < nk; k++ )
    {
      for ( int j = 0; j < nj; j++ )
      {
        for ( int i = 0; i < ni; i++, ++ind )
        {
          xArray[ind] = min[0] + step[0] * i;
          yArray[ind] = min[1] + step[1] * j;
          zArray[ind] = min[2] + step[2] * k;
        }
      }
    }

    if ( rotation != null )
    {
      double[] tmp = { (max[0] - min[0]) / 2., (max[1] - min[1]) / 2., (max[2] - min[2]) / 2. };
      for ( int a = 0; a < 3; ++a )
        rotate(tmp, rotation[a], a, xArray, yArray, zArray);
    }

    int numCells = dimX * dimY * dimZ * 6;
    int[] nodes = new int[9];
    int[] nodeIndexes = new int[numCells * 5];
    int pyramidPeakIndex = 0;

    for ( int i = 0, iNode = 0; i < dimX; i++ )
    {
      for ( int j = 0; j < dimY; j++ )
      {
        for ( int k = 0; k < dimZ; k++ )
        {
          nodes[0] = cubeCoordsBeginIndex + k * nj * ni + j * ni + i;
          nodes[1] = nodes[0] + 1;
          nodes[2] = nodes[1] + ni;
          nodes[3] = nodes[2] - 1;
          nodes[4] = nodes[0] + nj * ni;
          nodes[5] = nodes[4] + 1;
          nodes[6] = nodes[5] + ni;
          nodes[7] = nodes[4] + ni;
          nodes[8] = pyramidPeakIndex;

          // 1
          nodeIndexes[iNode++] = nodes[0];
          nodeIndexes[iNode++] = nodes[1];
          nodeIndexes[iNode++] = nodes[5];
          nodeIndexes[iNode++] = nodes[4];
          nodeIndexes[iNode++] = nodes[8];

          // 2
          nodeIndexes[iNode++] = nodes[0];
          nodeIndexes[iNode++] = nodes[1];
          nodeIndexes[iNode++] = nodes[2];
          nodeIndexes[iNode++] = nodes[3];
          nodeIndexes[iNode++] = nodes[8];

          // 3
          nodeIndexes[iNode++] = nodes[1];
          nodeIndexes[iNode++] = nodes[5];
          nodeIndexes[iNode++] = nodes[6];
          nodeIndexes[iNode++] = nodes[2];
          nodeIndexes[iNode++] = nodes[8];

          // 4
          nodeIndexes[iNode++] = nodes[5];
          nodeIndexes[iNode++] = nodes[4];
          nodeIndexes[iNode++] = nodes[7];
          nodeIndexes[iNode++] = nodes[6];
          nodeIndexes[iNode++] = nodes[8];

          // 5
          nodeIndexes[iNode++] = nodes[0];
          nodeIndexes[iNode++] = nodes[4];
          nodeIndexes[iNode++] = nodes[7];
          nodeIndexes[iNode++] = nodes[3];
          nodeIndexes[iNode++] = nodes[8];

          // 6
          nodeIndexes[iNode++] = nodes[3];
          nodeIndexes[iNode++] = nodes[2];
          nodeIndexes[iNode++] = nodes[6];
          nodeIndexes[iNode++] = nodes[7];
          nodeIndexes[iNode++] = nodes[8];

          pyramidPeakIndex++;
        }
      }
    }

    MbVolumeMeshPyramid mesh = new MbVolumeMeshPyramid(xArray, yArray, zArray, nodeIndexes);

    addScalarSets(mesh, numCoords);

    return mesh;
  }

  public MbVolumeMeshWedge getMeshWedge(int dimX, int dimY, int dimZ, double[] min, double[] max)
  {
    return getMeshWedge(dimX, dimY, dimZ, min, max, null);
  }

  public MbVolumeMeshWedge getMeshWedge(int dimX, int dimY, int dimZ, double[] min, double[] max, double[] rotation)
  {
    int ni = dimX + 1;
    int nj = dimY + 1;
    int nk = dimZ + 1;

    double[] step = { (max[0] - min[0]) / dimX, (max[1] - min[1]) / dimY, (max[2] - min[2]) / dimZ };

    int numCoords = ni * nj * nk;
    double[] xArray = new double[numCoords];
    double[] yArray = new double[numCoords];
    double[] zArray = new double[numCoords];
    int ind = 0;

    for ( int k = 0; k < nk; k++ )
    {
      for ( int j = 0; j < nj; j++ )
      {
        for ( int i = 0; i < ni; i++, ++ind )
        {
          xArray[ind] = min[0] + step[0] * i;
          yArray[ind] = min[1] + step[1] * j;
          zArray[ind] = min[2] + step[2] * k;
        }
      }
    }

    if ( rotation != null )
    {
      double[] tmp = { (max[0] - min[0]) / 2., (max[1] - min[1]) / 2., (max[2] - min[2]) / 2. };
      for ( int a = 0; a < 3; ++a )
        rotate(tmp, rotation[a], a, xArray, yArray, zArray);
    }

    int numCells = dimX * dimY * dimZ * 2;
    int[] nodes = new int[8];
    int[] nodeIndexes = new int[numCells * 6];

    for ( int i = 0, iCell = 0; i < dimX; i++ )
    {
      for ( int j = 0; j < dimY; j++ )
      {
        for ( int k = 0; k < dimZ; k++, iCell += 2 )
        {
          nodes[0] = k * nj * ni + j * ni + i;
          nodes[1] = nodes[0] + 1;
          nodes[2] = nodes[1] + ni;
          nodes[3] = nodes[2] - 1;
          nodes[4] = nodes[0] + nj * ni;
          nodes[5] = nodes[4] + 1;
          nodes[6] = nodes[5] + ni;
          nodes[7] = nodes[4] + ni;

          splitHexaIn2Wedge(nodes, nodeIndexes, iCell * 6);
        }
      }
    }

    MbVolumeMeshWedge mesh = new MbVolumeMeshWedge(xArray, yArray, zArray, nodeIndexes);

    addScalarSets(mesh, numCoords);

    return mesh;
  }

  private void splitHexaIn2Wedge(int[] hexaNodes, int[] wedgeNodes, int startIndex)
  {
    // first wedge
    wedgeNodes[startIndex] = hexaNodes[0];
    wedgeNodes[startIndex + 1] = hexaNodes[5];
    wedgeNodes[startIndex + 2] = hexaNodes[4];
    wedgeNodes[startIndex + 3] = hexaNodes[3];
    wedgeNodes[startIndex + 4] = hexaNodes[6];
    wedgeNodes[startIndex + 5] = hexaNodes[7];

    // second wedge
    wedgeNodes[startIndex + 6] = hexaNodes[0];
    wedgeNodes[startIndex + 7] = hexaNodes[5];
    wedgeNodes[startIndex + 8] = hexaNodes[1];
    wedgeNodes[startIndex + 9] = hexaNodes[3];
    wedgeNodes[startIndex + 10] = hexaNodes[6];
    wedgeNodes[startIndex + 11] = hexaNodes[2];
  }

  /**
   * Returns a simple tetrahedron mesh.
   */
  public MbVolumeMeshTetrahedron getMeshTetrahedron()
  {
    double[] xArray = new double[] { 0, 10, 10, 0, 0, 10, 10, 0 };
    double[] yArray = new double[] { 0, 0, 10, 10, 0, 0, 10, 10 };
    double[] zArray = new double[] { 0, 0, 0, 0, 10, 10, 10, 10 };

    int[] nodeIndexes = { 0, 4, 6, 5, 0, 6, 2, 1, 0, 1, 5, 6, 0, 6, 4, 7, 0, 2, 6, 3, 0, 7, 3, 6 };

    MbVolumeMeshTetrahedron mesh = new MbVolumeMeshTetrahedron(xArray, yArray, zArray, nodeIndexes);

    double[] cellId = new double[] { 0, 1, 2, 3, 4, 5 };
    boolean[] deadCells = new boolean[6];

    if ( m_deadCellFctI != null )
      for ( int i = 0; i < 6; i++ )
        deadCells[i] = m_deadCellFctI.apply(i);

    MbScalarSetI scalar = new MbScalarSetI(cellId, "$CellId", DataBinding.PER_CELL);
    mesh.addScalarSet(scalar);

    addScalarSets(mesh, 8);

    MbVec3SetI vecSet = new MbVec3SetI(xArray, yArray, zArray);
    vecSet.setName("$MyGeometry");
    mesh.addVec3Set(vecSet);

    if ( m_deadCellFctI != null )
      mesh.setDeadCells(deadCells);

    return mesh;
  }

  /**
   * Returns a tetrahedron mesh.
   */
  public MbVolumeMeshTetrahedron getMeshTetrahedron(int dimX, int dimY, int dimZ, double[] min, double[] max)
  {
    return getMeshTetrahedron(dimX, dimY, dimZ, min, max, null, null, null);
  }

  /**
   * Returns a tetrahedron mesh. sroi and eroi define a region of interest in
   * the generated mesh by selecting only cells within the given range.
   */
  public MbVolumeMeshTetrahedron getMeshTetrahedron(int dimX, int dimY, int dimZ, double[] min, double[] max,
      int[] sroi, int[] eroi)
  {
    return getMeshTetrahedron(dimX, dimY, dimZ, min, max, null, sroi, eroi);
  }

  /**
   * Returns a tetrahedron mesh. rotations are the angle to rotate the mesh
   * around the lines passing through its center and parallel to each axis. sroi
   * and eroi define a region of interest in the generated mesh by selecting
   * only cells within the given range.
   */
  public MbVolumeMeshTetrahedron getMeshTetrahedron(int dimX, int dimY, int dimZ, double[] min, double[] max,
      double[] rotations, int[] sroi, int[] eroi)
  {
    int ni = dimX + 1;
    int nj = dimY + 1;
    int nk = dimZ + 1;

    double[] step = { (max[0] - min[0]) / dimX, (max[1] - min[1]) / dimY, (max[2] - min[2]) / dimZ };

    int numCoords = ni * nj * nk;
    double[] xArray = new double[numCoords];
    double[] yArray = new double[numCoords];
    double[] zArray = new double[numCoords];
    int ind = 0;

    for ( int k = 0; k < nk; k++ )
    {
      for ( int j = 0; j < nj; j++ )
      {
        for ( int i = 0; i < ni; i++, ++ind )
        {
          xArray[ind] = min[0] + step[0] * i;
          yArray[ind] = min[1] + step[1] * j;
          zArray[ind] = min[2] + step[2] * k;
        }
      }
    }

    if ( rotations != null )
    {
      double[] tmp = { (max[0] - min[0]) / 2., (max[1] - min[1]) / 2., (max[2] - min[2]) / 2. };
      for ( int a = 0; a < 3; ++a )
        rotate(tmp, rotations[a], a, xArray, yArray, zArray);
    }

    if ( sroi != null && eroi != null )
    {
      dimX = eroi[0] - sroi[0];
      dimY = eroi[1] - sroi[1];
      dimZ = eroi[2] - sroi[2];
    }
    else
    {
      sroi = new int[] { 0, 0, 0 };
      eroi = new int[] { dimX, dimY, dimZ };
    }

    final int numCells = dimX * dimY * dimZ * 6;
    int[] hexaNodes = new int[8];
    int[] nodeIndexes = new int[numCells * 4];

    double[] cellId = new double[numCells];
    boolean[] deadCells = null;
    if ( m_deadCellFctI != null )
      deadCells = new boolean[numCells];

    for ( int i = sroi[0], iCell = 0; i < eroi[0]; i++ )
    {
      for ( int j = sroi[1]; j < eroi[1]; j++ )
      {
        for ( int k = sroi[2]; k < eroi[2]; k++ )
        {
          hexaNodes[0] = k * nj * ni + j * ni + i;
          hexaNodes[1] = hexaNodes[0] + 1;
          hexaNodes[2] = hexaNodes[1] + ni;
          hexaNodes[3] = hexaNodes[2] - 1;
          hexaNodes[4] = hexaNodes[0] + nj * ni;
          hexaNodes[5] = hexaNodes[4] + 1;
          hexaNodes[6] = hexaNodes[5] + ni;
          hexaNodes[7] = hexaNodes[4] + ni;

          splitHexaIn6Tetra(hexaNodes, nodeIndexes, iCell * 4);

          for ( int c = 0; c < 6; ++c, ++iCell )
          {
            cellId[iCell] = iCell;
            if ( deadCells != null )
              deadCells[iCell] = m_deadCellFctI.apply(iCell);
          }
        }
      }
    }

    MbVolumeMeshTetrahedron mesh = new MbVolumeMeshTetrahedron(xArray, yArray, zArray, nodeIndexes);

    MbVec3SetI vec = new MbVec3SetI(xArray, yArray, zArray);
    vec.setName("$MyGeometry");
    mesh.addVec3Set(vec);

    MbScalarSetI scalar = new MbScalarSetI(cellId, "$CellId", DataBinding.PER_CELL);
    mesh.addScalarSet(scalar);

    addScalarSets(mesh, numCoords);

    if ( deadCells != null )
      mesh.setDeadCells(deadCells);

    return mesh;
  }

  private void splitHexaIn6Tetra(int[] hexaNodes, int[] tetraNodes, int startIndex)
  {
    // first wedge
    // tetra 0,4,6,5
    tetraNodes[startIndex] = hexaNodes[0];
    tetraNodes[startIndex + 1] = hexaNodes[4];
    tetraNodes[startIndex + 2] = hexaNodes[6];
    tetraNodes[startIndex + 3] = hexaNodes[5];
    // tetra 0,6,2,1,
    tetraNodes[startIndex + 4] = hexaNodes[0];
    tetraNodes[startIndex + 5] = hexaNodes[6];
    tetraNodes[startIndex + 6] = hexaNodes[2];
    tetraNodes[startIndex + 7] = hexaNodes[1];
    // tetra 0,1,5,6,
    tetraNodes[startIndex + 8] = hexaNodes[0];
    tetraNodes[startIndex + 9] = hexaNodes[1];
    tetraNodes[startIndex + 10] = hexaNodes[5];
    tetraNodes[startIndex + 11] = hexaNodes[6];
    // // second wedge
    // tetra 0,6,4,7,
    tetraNodes[startIndex + 12] = hexaNodes[0];
    tetraNodes[startIndex + 13] = hexaNodes[6];
    tetraNodes[startIndex + 14] = hexaNodes[4];
    tetraNodes[startIndex + 15] = hexaNodes[7];
    // tetra 0,2,6,3,
    tetraNodes[startIndex + 16] = hexaNodes[0];
    tetraNodes[startIndex + 17] = hexaNodes[2];
    tetraNodes[startIndex + 18] = hexaNodes[6];
    tetraNodes[startIndex + 19] = hexaNodes[3];
    // tetra 0,7,3,6
    tetraNodes[startIndex + 20] = hexaNodes[0];
    tetraNodes[startIndex + 21] = hexaNodes[7];
    tetraNodes[startIndex + 22] = hexaNodes[3];
    tetraNodes[startIndex + 23] = hexaNodes[6];
  }

  /**
   * Returns an hexahedron mesh.
   */
  public MbVolumeMeshHexahedron getMeshHexahedron(int dimX, int dimY, int dimZ, double[] min, double[] max)
  {
    return getMeshHexahedron(dimX, dimY, dimZ, min, max, null, null, null);
  }

  /**
   * Returns an hexahedron mesh. sroi and eroi define a region of interest in
   * the generated mesh by selecting only cells within the given range.
   */
  public MbVolumeMeshHexahedron getMeshHexahedron(int dimX, int dimY, int dimZ, double[] min, double[] max, int[] sroi,
      int[] eroi)
  {
    return getMeshHexahedron(dimX, dimY, dimZ, min, max, null, sroi, eroi);
  }

  /**
   * Returns an hexahedron mesh. rotations are the angle to rotate the mesh
   * around the lines passing through its center and parallel to each axis. sroi
   * and eroi define a region of interest in the generated mesh by selecting
   * only cells within the given range.
   */
  public MbVolumeMeshHexahedron getMeshHexahedron(int dimX, int dimY, int dimZ, double[] min, double[] max,
      double[] rotations, int[] sroi, int[] eroi)
  {
    int ni = dimX + 1;
    int nj = dimY + 1;
    int nk = dimZ + 1;

    double[] step = { (max[0] - min[0]) / dimX, (max[1] - min[1]) / dimY, (max[2] - min[2]) / dimZ };

    int numCoords = ni * nj * nk;
    double[] xArray = new double[numCoords];
    double[] yArray = new double[numCoords];
    double[] zArray = new double[numCoords];
    int ind = 0;

    for ( int k = 0; k < nk; k++ )
    {
      for ( int j = 0; j < nj; j++ )
      {
        for ( int i = 0; i < ni; i++, ++ind )
        {
          xArray[ind] = min[0] + step[0] * i;
          yArray[ind] = min[1] + step[1] * j;
          zArray[ind] = min[2] + step[2] * k;
        }
      }
    }

    if ( rotations != null )
    {
      double[] tmp = { (max[0] - min[0]) / 2., (max[1] - min[1]) / 2., (max[2] - min[2]) / 2. };
      for ( int a = 0; a < 3; ++a )
        rotate(tmp, rotations[a], a, xArray, yArray, zArray);
    }

    if ( sroi != null && eroi != null )
    {
      dimX = eroi[0] - sroi[0];
      dimY = eroi[1] - sroi[1];
      dimZ = eroi[2] - sroi[2];
    }
    else
    {
      sroi = new int[] { 0, 0, 0 };
      eroi = new int[] { dimX, dimY, dimZ };
    }

    final int numCells = dimX * dimY * dimZ;
    final int numCellNodes = 8;
    final int[] nodeIndexes = new int[numCells * numCellNodes];
    int iNode;
    boolean[] deadCells = null;

    if ( m_deadCellFctI != null )
      deadCells = new boolean[numCells];

    for ( int k = sroi[2], iCell = 0; k < eroi[2]; k++ )
    {
      for ( int j = sroi[1]; j < eroi[1]; j++ )
      {
        for ( int i = sroi[0]; i < eroi[0]; i++, ++iCell )
        {
          iNode = iCell * numCellNodes;
          nodeIndexes[iNode] = k * nj * ni + j * ni + i;
          nodeIndexes[iNode + 1] = nodeIndexes[iNode] + 1;
          nodeIndexes[iNode + 2] = nodeIndexes[iNode + 1] + ni;
          nodeIndexes[iNode + 3] = nodeIndexes[iNode + 2] - 1;
          nodeIndexes[iNode + 4] = nodeIndexes[iNode] + nj * ni;
          nodeIndexes[iNode + 5] = nodeIndexes[iNode + 4] + 1;
          nodeIndexes[iNode + 6] = nodeIndexes[iNode + 5] + ni;
          nodeIndexes[iNode + 7] = nodeIndexes[iNode + 4] + ni;

          if ( deadCells != null )
            deadCells[iCell] = m_deadCellFctI.apply(iCell);
        }
      }
    }

    MbVolumeMeshHexahedron mesh = new MbVolumeMeshHexahedron(xArray, yArray, zArray, nodeIndexes);

    addScalarSets(mesh, numCoords);

    if ( deadCells != null )
      mesh.setDeadCells(deadCells);

    return mesh;
  }

  private void addScalarSets(MbMesh mesh, int numPoints)
  {
    int numFcts = this.m_scalarFctors.size();
    double[] dataset;
    MbScalarSetI scalarSet;
    MbScalarFunctor fctor;
    MiGeometryI geometry = mesh.getGeometry();
    double[] xyz;

    for ( int f = 0; f < numFcts; ++f )
    {
      dataset = new double[numPoints];
      fctor = m_scalarFctors.get(f);

      for ( int n = 0; n < numPoints; ++n )
      {
        xyz = geometry.getCoord(n);
        dataset[n] = fctor.apply(xyz[0], xyz[1], xyz[2]);
      }

      scalarSet = new MbScalarSetI(dataset);

      if ( !("".equals(this.m_scalarFctNames.get(f))) )
        scalarSet.setName(this.m_scalarFctNames.get(f));

      mesh.addScalarSet(scalarSet);
    }
  }

  private static void rotate(double[] center, double angle, int rotAxis, double[] xArray, double[] yArray,
      double[] zArray)
  {
    if ( angle != 0. )
    {
      double cosa = Math.cos(angle);
      double sina = Math.sin(angle);
      double[] shift;
      double[] tcoord = new double[3];

      if ( rotAxis != 1 )
        shift = new double[] { center[0], center[1], center[2] };
      else
        shift = new double[] { 0, 0, 0 };

      switch ( rotAxis )
      {
      case 0 :
        for ( int i = 0; i < xArray.length; i++ )
        {
          tcoord[0] = xArray[i] - shift[0];
          tcoord[1] = yArray[i] - shift[1];
          tcoord[2] = zArray[i] - shift[2];

          xArray[i] = tcoord[0] + shift[0];
          yArray[i] = (tcoord[1] * cosa - tcoord[2] * sina) + shift[1];
          zArray[i] = (tcoord[1] * sina + tcoord[2] * cosa) + shift[2];
        }
        break;
      case 1 :
        for ( int i = 0; i < xArray.length; i++ )
        {
          tcoord[0] = xArray[i] - shift[0];
          tcoord[1] = yArray[i] - shift[1];
          tcoord[2] = zArray[i] - shift[2];

          xArray[i] = (tcoord[0] * cosa - tcoord[2] * sina) + shift[0];
          yArray[i] = tcoord[1] + shift[1];
          zArray[i] = (tcoord[0] * sina + tcoord[2] * cosa) + shift[2];
        }
        break;
      case 2 :
        for ( int i = 0; i < xArray.length; i++ )
        {
          tcoord[0] = xArray[i] - shift[0];
          tcoord[1] = yArray[i] - shift[1];
          tcoord[2] = zArray[i] - shift[2];

          xArray[i] = (tcoord[0] * cosa - tcoord[1] * sina) + shift[0];
          yArray[i] = (tcoord[0] * sina + tcoord[1] * cosa) + shift[1];
          zArray[i] = tcoord[2] + shift[2];
        }
        break;
      }
    }
  }

  // ///////////////////////
  // Inner classes
  // //////////////////////
  private class SphereFctor implements MbScalarFunctor
  {

    @Override
    public double apply(double x, double y, double z)
    {
      return x * x + y * y + z * z;
    }

  }

  private class XFctor implements MbScalarFunctor
  {

    @Override
    public double apply(double x, double y, double z)
    {
      return x;
    }
  }

}
