var database = require('./database'),
    httpProxy = require('http-proxy');

/**
 * Array of proxy
 */
var instanceProxies = [];

/**
 * Load configuration file.
 */
const config = require('./config/config.json');

/**
 * Database connection
 */
var db = new database.Database(config.database.path /*, log = true*/);

/**
 * HTTP callback that handles instance registration.
 * PATH: "/instance", METHOD: "POST"; Registers an instance from an IP address and a port.
 * Code 200 indicates success, returns the registered instance object in JSON.
 * Code 500 indicates error.
 */
var registerInstance = function (req, res) {
    try {
        const newInstance = req.body;

        let instance = new database.Instance(newInstance.ip, newInstance.port);

        if (instance.ip == undefined || instance.port == undefined) {
            throw new Error('Bad Post Data. Need fields ip and port.\n');
        }

        instance = db.registerInstance(instance.ip, instance.port);

        if (instance == null) {
            throw new Error('Unable to register the instance.\n');
        }

        console.log('The instance ' + instance.id + ' is registred.');

        res.writeHead(200, { "Content-Type": "application/json" });
        res.write(JSON.stringify(instance));
    } catch (err) {
        res.writeHead(500, { "Content-Type": "text/plain" });
        res.write(err.message);
    }
    res.end();
};

/**
 * HTTP callback that handles instance unregistration.
 * PATH: "/instance/:id", METHOD: "DELETE"; Unregisters an instance from an ID.
 * Code 200 indicates success and code 500 indicates error.
 */
var unregisterInstance = function (req, res) {
    const instanceId = req.params.id;
    const instance = db.getInstanceById(instanceId);

    try {
        db.deleteInstance(instance);

        console.log('The instance ' + instance.id + ' is unregistred.');

        res.writeHead(200, { "Content-Type": "text/plain" });
    } catch (err) {
        res.writeHead(500, { "Content-Type": "text/plain" });
    }

    res.end();
};

/**
 * HTTP callback that handles renderArea deletion.
 * PATH: "/renderArea/:id", METHOD: "DELETE": Considers the renderArea as destroyed.
 * Code 200 indicates success and code 500 indicates error.
 */
var deleteRenderArea = function (req, res) {
    const renderAreaId = req.params.id;
    const renderArea = new database.RenderArea(renderAreaId);

    try {
        db.deleteRenderArea(renderArea);

        res.writeHead(200, { "Content-Type": "text/plain" });
    } catch (err) {
        res.writeHead(500, { "Content-Type": "text/plain" });
    }

    res.end();
};

/**
 * Create a proxy to redirect the websocket connection to an instance.
 */
var createInstanceProxy = function (instance) {
    // Create proxy to handle websocket connection and store it.
    instanceProxies[instance.id] = new httpProxy.createProxyServer({
        target: {
            host: instance.ip,
            port: instance.port
        }
    });
    instanceProxies[instance.id].on('error', function () {
        console.log('An error occurs on the instance ' + instance.id);

        // Consider the instance as down
        db.deleteInstance(instance);

        // Remove the instance proxy from the proxy array
        const index = instanceProxies.indexOf(instance.id);
        if (index !== -1)
            instanceProxies.splice(index, 1);
    });
    instanceProxies[instance.id].on('close', function () {
        console.log('A client is disconnected from the instance ' + instance.id);
    });
}

/**
 * Returns the instance hosting the least renderArea.
 */
var getAvailableInstance = function () {
    const instances = db.getInstances();

    if (instances.length == 0)
        return null;

    let availableInstance = instances[0];
    let numberOfRenderAreaInstance = db.getNumRenderAreas(availableInstance);
    instances.forEach(function (item) {
        const numberOfRenderArea = db.getNumRenderAreas(item);
        if (numberOfRenderArea < numberOfRenderAreaInstance) {
            availableInstance = item;
            numberOfRenderAreaInstance = numberOfRenderArea;
        }
    });
    return availableInstance;
}

/**
 * Proxy the WebSocket requests:
 * Select an instance to handle the renderArea and 
 * redirect the Websocket connection.
 */
var proxyWebSocket = function (req, socket, head) {
    const renderAreaId = req.url.split('/').pop(-1);
    const renderArea = new database.RenderArea(renderAreaId);
    var instance = db.getInstance(renderArea);

    if (instance == null) {
        // No instance hosts the renderArea.
        console.log('No instance hosts the renderArea ' + renderArea.name);

        // Get an available instance to host the renderArea
        instance = getAvailableInstance();

        if (instance == null) {
            console.error("No instance available");
            return;
        }

        // Create the renderArea if not exists and associate it to the instance
        db.createRenderArea(renderArea, instance);
    }
    else {
        console.log('The instance ' + instance.id + ' hosts the renderArea ' + renderArea.name);
    }

    // Check if the instance has a proxy.
    if (instanceProxies[instance.id] == undefined) {
        createInstanceProxy(instance);
    }

    // Proxy the websocket connection
    instanceProxies[instance.id].ws(req, socket, head);
};

/**
 * Create proxy to handle HTTP requests.
 */
var webServerProxy = new httpProxy.createProxyServer({
    target: {
        host: config.webserver.ip,
        port: config.webserver.port
    }
});

/**
 * Catch web server proxy error.
 */
webServerProxy.on('error', function (err) {
  console.log("Web server proxy error: " + err);
});

/**
 *  Redirect HTTP request to the web server.
 */
var proxyWebServer = function (req, res) {
    webServerProxy.web(req, res);
}

/**
 *  Close the webserver proxy and the database.
 */
var close = function () {
    webServerProxy.close();

    db.close();
}

module.exports = {
    registerInstance,
    unregisterInstance,
    deleteRenderArea,
    proxyWebSocket,
    proxyWebServer,
    close
};
